#!/usr/bin/env node
/**
 * CosyVoice WebSocket Text-to-Speech Service - Node.js Version
 * Based on Express + Socket.IO + WebSocket
 */

const express = require('express');
const http = require('http');
const socketIO = require('socket.io');
const WebSocket = require('ws');
const path = require('path');
const { v4: uuidv4 } = require('uuid');

// Configuration
const PORT = process.env.PORT || 9000;
const API_KEY = process.env.DASHSCOPE_API_KEY;
const DASHSCOPE_URI = 'wss://dashscope.aliyuncs.com/api-ws/v1/inference/';

// Create Express application
const app = express();
const server = http.createServer(app);
const io = socketIO(server, {
  cors: {
    origin: '*',
    methods: ['GET', 'POST']
  }
});

// Static file service
app.use(express.static('public'));

// Routes
app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'views', 'index.html'));
});

// Client Manager
class ClientManager {
  constructor() {
    this.clients = new Map(); // socketId -> client data
  }

  createClient(socketId, voice) {
    // Clean up old client connection
    if (this.clients.has(socketId)) {
      const oldClient = this.clients.get(socketId);
      try {
        oldClient.client.close();
      } catch (e) {
        // Ignore close error
      }
      this.clients.delete(socketId);
    }

    // Create new client
    try {
      const client = new TTSClient(API_KEY, DASHSCOPE_URI, voice, socketId);
      this.clients.set(socketId, {
        client: client,
        voice: voice
      });
      client.run();
      return client;
    } catch (error) {
      console.error(`Failed to create TTS client: ${error.message}`);
      return null;
    }
  }

  getClient(socketId) {
    return this.clients.get(socketId);
  }

  removeClient(socketId) {
    if (this.clients.has(socketId)) {
      const clientData = this.clients.get(socketId);
      try {
        clientData.client.close();
      } catch (e) {
        // Ignore close error
      }
      this.clients.delete(socketId);
    }
  }
}

const clientManager = new ClientManager();

// TTS Client Class
class TTSClient {
  constructor(apiKey, uri, voice = 'longanyang', socketId = null) {
    this.apiKey = apiKey;
    this.uri = uri;
    this.voice = voice;
    this.socketId = socketId;
    this.taskId = uuidv4();
    this.ws = null;
    this.taskStarted = false;
    this.taskFinished = false;
    this.taskStartedPromise = null;
    this.taskStartedResolve = null;
  }

  onOpen() {
    try {
      const runTaskCmd = {
        header: {
          action: 'run-task',
          task_id: this.taskId,
          streaming: 'duplex'
        },
        payload: {
          task_group: 'audio',
          task: 'tts',
          function: 'SpeechSynthesizer',
          model: 'cosyvoice-v3-flash',
          parameters: {
            text_type: 'PlainText',
            voice: this.voice,
            format: 'mp3',
            sample_rate: 22050,
            volume: 50,
            rate: 1,
            pitch: 1
          },
          input: {}
        }
      };
      this.ws.send(JSON.stringify(runTaskCmd));
      console.log(`Sent run-task command (sid: ${this.socketId})`);
    } catch (error) {
      console.error(`Failed to send run-task command: ${error.message}`);
      this.sendError(`Failed to send command: ${error.message}`);
    }
  }

  onMessage(data) {
    try {
      // Try to convert Buffer to string and parse as JSON
      let isJson = false;
      let msgJson = null;

      if (Buffer.isBuffer(data)) {
        try {
          const text = data.toString('utf8');
          msgJson = JSON.parse(text);
          isJson = true;
        } catch (e) {
          // Not JSON, it's audio data
          isJson = false;
        }
      } else if (typeof data === 'string') {
        try {
          msgJson = JSON.parse(data);
          isJson = true;
        } catch (e) {
          isJson = false;
        }
      }

      if (isJson && msgJson) {
        const header = msgJson.header || {};
        const event = header.event || '';

        if (event === 'task-started') {
          this.taskStarted = true;
          if (this.taskStartedResolve) {
            this.taskStartedResolve();
          }
          io.to(this.socketId).emit('audio_start');
        } else if (event === 'task-finished' || event === 'task-failed') {
          this.taskFinished = true;
          io.to(this.socketId).emit('audio_end');
          this.close();
          console.log(`Task completed (sid: ${this.socketId})`);
        }
      } else {
        // Binary audio data
        io.to(this.socketId).emit('audio_chunk', { data: data });
      }
    } catch (error) {
      console.error(`Failed to process message: ${error.message}`);
      this.sendError(`Failed to process message: ${error.message}`);
    }
  }

  onError(error) {
    console.error(`WebSocket error (sid: ${this.socketId}): ${error.message}`);
    this.sendError(`WebSocket error: ${error.message}`);
  }

  onClose(code, reason) {
    console.log(`WebSocket closed (sid: ${this.socketId}): ${reason} (${code})`);
    clientManager.removeClient(this.socketId);
    if (this.taskStartedResolve) {
      this.taskStartedResolve();
    }
  }

  sendContinueTask(text) {
    if (!this.ws || this.ws.readyState !== WebSocket.OPEN) {
      console.error('WebSocket not connected, unable to send data');
      return false;
    }

    try {
      const cmd = {
        header: {
          action: 'continue-task',
          task_id: this.taskId,
          streaming: 'duplex'
        },
        payload: {
          input: {
            text: text
          }
        }
      };
      this.ws.send(JSON.stringify(cmd));
      return true;
    } catch (error) {
      console.error(`Failed to send continue-task: ${error.message}`);
      this.sendError(`Failed to send text: ${error.message}`);
      return false;
    }
  }

  sendFinishTask() {
    if (!this.ws || this.ws.readyState !== WebSocket.OPEN) {
      console.error('WebSocket not connected, unable to send finish command');
      return false;
    }

    try {
      const cmd = {
        header: {
          action: 'finish-task',
          task_id: this.taskId,
          streaming: 'duplex'
        },
        payload: {
          input: {}
        }
      };
      this.ws.send(JSON.stringify(cmd));
      console.log(`Sent finish-task command (sid: ${this.socketId})`);
      return true;
    } catch (error) {
      console.error(`Failed to send finish-task: ${error.message}`);
      this.sendError(`Failed to send finish command: ${error.message}`);
      return false;
    }
  }

  close() {
    try {
      if (this.ws && this.ws.readyState === WebSocket.OPEN) {
        this.ws.close();
      }
    } catch (e) {
      // Ignore close error
    }
  }

  run() {
    try {
      this.taskStartedPromise = new Promise((resolve) => {
        this.taskStartedResolve = resolve;
      });

      this.ws = new WebSocket(this.uri, {
        headers: {
          'Authorization': `bearer ${this.apiKey}`,
          'X-DashScope-DataInspection': 'enable'
        }
      });

      this.ws.on('open', () => this.onOpen());
      this.ws.on('message', (data) => this.onMessage(data));
      this.ws.on('error', (error) => this.onError(error));
      this.ws.on('close', (code, reason) => this.onClose(code, reason));
    } catch (error) {
      console.error(`Failed to start WebSocket: ${error.message}`);
      this.sendError(`Connection failed: ${error.message}`);
    }
  }

  sendError(message) {
    try {
      io.to(this.socketId).emit('synthesis_error', { message: message });
    } catch (e) {
      // Ignore send error
    } finally {
      clientManager.removeClient(this.socketId);
    }
  }

  async waitForTaskStarted(timeout = 10000) {
    const timeoutPromise = new Promise((resolve) => {
      setTimeout(() => resolve(false), timeout);
    });
    const result = await Promise.race([this.taskStartedPromise, timeoutPromise]);
    return result !== false;
  }
}

// Socket.IO Event Handling
io.on('connection', (socket) => {
  console.log(`Client connected: ${socket.id}`);

  socket.on('disconnect', () => {
    console.log(`Client disconnected: ${socket.id}`);
    clientManager.removeClient(socket.id);
  });

  socket.on('synthesize', async (data) => {
    const inputText = data.input || '';
    const voice = data.voice || 'longanyang';

    if (!inputText) {
      console.log(`Received empty text, ignoring (sid: ${socket.id})`);
      socket.emit('synthesis_error', { message: 'Input text cannot be empty' });
      return;
    }

    console.log(`Received synthesis request (sid: ${socket.id}): ${inputText.substring(0, 20)}... Voice: ${voice}`);

    try {
      // Create speech synthesis client for current client
      const client = clientManager.createClient(socket.id, voice);
      if (!client) {
        socket.emit('synthesis_error', { message: 'Failed to create synthesis client' });
        return;
      }

      // Wait for task to start (up to 10 seconds)
      const started = await client.waitForTaskStarted(10000);
      if (!started) {
        console.log(`Task start timeout (sid: ${socket.id})`);
        socket.emit('synthesis_error', { message: 'Task start timeout' });
        return;
      }

      // Split text by sentence boundaries
      const SENTENCE_DELIMITERS = ['.', '?', '!', '。', '？', '！', '\n'];
      const fragments = [];
      let startIndex = 0;
      let i = 0;

      while (i < inputText.length) {
        if (SENTENCE_DELIMITERS.includes(inputText[i])) {
          let endIndex = i + 1;
          while (endIndex < inputText.length && SENTENCE_DELIMITERS.includes(inputText[endIndex])) {
            endIndex++;
          }
          fragments.push(inputText.substring(startIndex, endIndex));
          startIndex = endIndex;
          i = endIndex - 1;
        }
        i++;
      }

      if (startIndex < inputText.length) {
        fragments.push(inputText.substring(startIndex));
      }

      // Send all text fragments
      for (const fragment of fragments) {
        if (!client.sendContinueTask(fragment)) {
          socket.emit('synthesis_error', { message: 'Failed to send text' });
          return;
        }
      }

      // Send finish task command
      if (!client.sendFinishTask()) {
        socket.emit('synthesis_error', { message: 'Failed to send finish command' });
        return;
      }
    } catch (error) {
      const errorMsg = `Failed to process request: ${error.message}`;
      console.error(`${errorMsg} (sid: ${socket.id})\n${error.stack}`);
      socket.emit('synthesis_error', { message: errorMsg });
    }
  });
});

// Check API Key
if (!API_KEY) {
  console.error('Error: DASHSCOPE_API_KEY environment variable not set');
  console.error('Please run: export DASHSCOPE_API_KEY=\'your-api-key\'');
  process.exit(1);
}

// Start Server
server.listen(PORT, '0.0.0.0', () => {
  console.log('='.repeat(50));
  console.log(`  CosyVoice WebSocket Service (Node.js)`);
  console.log('='.repeat(50));
  console.log(`  Service URL: http://localhost:${PORT}`);
  console.log(`  API Key: ${API_KEY.substring(0, 10)}...`);
  console.log('='.repeat(50));
  console.log('');
  console.log('Press Ctrl+C to stop the service');
});

// Graceful Shutdown
process.on('SIGINT', () => {
  console.log('\nShutting down server...');
  server.close(() => {
    console.log('Server closed');
    process.exit(0);
  });
});

process.on('SIGTERM', () => {
  console.log('\nShutting down server...');
  server.close(() => {
    console.log('Server closed');
    process.exit(0);
  });
});
