# 🎙️ CosyVoice WebSocket Speech Synthesis Service (Node.js)

Alibaba Cloud CosyVoice speech synthesis service based on Express + Socket.IO + WebSocket, Node.js implementation.

[![Node Version](https://img.shields.io/badge/node-14%2B-brightgreen)](https://nodejs.org/)
[![License](https://img.shields.io/badge/license-MIT-blue)](./LICENSE)
[![Platform](https://img.shields.io/badge/platform-Windows%20%7C%20macOS%20%7C%20Linux-lightgrey)](.)

## ✨ Features

- 🚀 **One-Click Start** - Auto-checks environment and installs dependencies
- 🌍 **Cross-Platform** - Supports Windows, macOS, Linux
- ⚡ **Real-Time Synthesis** - WebSocket streaming transmission
- 🎨 **Multiple Voices** - Supports all CosyVoice voice models
- 📦 **Lightweight & Efficient** - Node.js async non-blocking architecture

## 🚀 Quick Start

### Prerequisites

- Node.js 14 or higher
- npm or yarn
- Alibaba Cloud DashScope API Key

### 1. Get API Key

Obtain API Key according to official documentation and configure it as an environment variable

### 2. Start Service

**macOS/Linux:**
```bash
./start.sh
```

**Windows:**
```cmd
start.bat
```

**Using npm:**
```bash
npm install    # Required for first run
npm start
```

### 3. Use the Service

Open your browser and visit: **http://localhost:9000**

## 📁 Project Structure

```
cosyvoiceNodeJs_en/
├── server.js           # Node.js server main program ⭐
├── package.json        # npm configuration file
├── start.sh            # Shell startup script
├── get_api_key.js     # API Key retrieval guide
├── views/
│   └── index.html     # Web frontend page
```

## ⚙️ Configuration

### Set Environment Variable

**macOS/Linux:**
```bash
export DASHSCOPE_API_KEY='sk-xxxxxxxxxxxx'
```

**Windows (CMD):**
```cmd
set DASHSCOPE_API_KEY=sk-xxxxxxxxxxxx
```

**Windows (PowerShell):**
```powershell
$env:DASHSCOPE_API_KEY='sk-xxxxxxxxxxxx'
```

### Modify Port

Edit line 12 in `server.js`:
```javascript
const PORT = process.env.PORT || 9000;  // Modify default port
```

Or use environment variable:
```bash
PORT=8080 npm start
```

## 🛠️ Development Mode

Install development dependencies:
```bash
npm install
```

Use nodemon for auto-restart:
```bash
npm run dev
```

## 📦 Dependencies

### Core Dependencies
- **express** - Web framework
- **socket.io** - Real-time bidirectional communication
- **ws** - WebSocket client
- **uuid** - Unique ID generation

### Development Dependencies
- **nodemon** - Auto-restart tool


## 🔧 Troubleshooting

### Q1: Port Already in Use

**macOS/Linux:**
```bash
lsof -i :9000
kill -9 <PID>
```

**Windows:**
```cmd
netstat -ano | findstr :9000
taskkill /PID <PID> /F
```

### Q2: Dependency Installation Failed

Use Taobao mirror:
```bash
npm install --registry=https://registry.npmmirror.com
```

Or use cnpm:
```bash
npm install -g cnpm --registry=https://registry.npmmirror.com
cnpm install
```

### Q3: Node.js Version Too Low

Upgrade Node.js:
- **macOS:** `brew upgrade node`
- **Windows:** Download latest version from [nodejs.org](https://nodejs.org/)
- **Linux:** Use nvm or package manager to upgrade

## 🔗 API Documentation

### Socket.IO Events

**Client → Server:**
```javascript
socket.emit('synthesize', {
  input: 'Text to synthesize',
  voice: 'longanyang'  // Optional
});
```

**Server → Client:**
```javascript
// Start receiving audio
socket.on('audio_start', () => { ... });

// Audio data chunks
socket.on('audio_chunk', (data) => {
  const audioBuffer = data.data;
});

// Audio ended
socket.on('audio_end', () => { ... });

// Error message
socket.on('synthesis_error', (data) => {
  console.error(data.message);
});
```

## 📊 Performance Optimization

1. **Connection Reuse** - Each client maintains independent WebSocket connection
2. **Async Processing** - Node.js event loop naturally supports high concurrency
3. **Streaming Transmission** - Receive and play simultaneously, reduce latency
4. **Text Segmentation** - Auto-split by sentences, improve response speed

## 🔒 Security Recommendations

1. **Environment Variables** - Don't hardcode API Key in code
2. **HTTPS** - Use HTTPS + WSS in production
3. **Rate Limiting** - Add request frequency limits
4. **Authentication** - Add user authentication in production
