/*
 * Decompiled with CFR 0.152.
 */
package com.starrocks.sync;

import com.starrocks.common.Utils;
import com.starrocks.meta.ClusterInfo;
import com.starrocks.meta.Database;
import com.starrocks.meta.JobProgress;
import com.starrocks.meta.Table;
import com.starrocks.netty.SyncChannelInitalizer;
import com.starrocks.sync.ClusterMetaKeeper;
import com.starrocks.sync.DDLJob;
import com.starrocks.sync.ReplicationJob;
import com.starrocks.sync.SyncConf;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.ChannelFuture;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class SyncJob {
    private static final Logger LOG = LogManager.getLogger(SyncJob.class);
    private SyncConf syncConf;
    private final ClusterMetaKeeper clusterMetaKeeper;
    private final ConcurrentLinkedQueue<DDLJob> ddlRequestQueue;
    private final ConcurrentLinkedQueue<DDLJob> ddlRunning;
    private final Map<String, String> replicationJobState;
    private final Map<String, String> finishedTableStatus;
    private final Map<String, Map<String, String>> replicationTableStatus;
    private final ConcurrentLinkedDeque<ReplicationJob> replicationJobQueue;
    private boolean sourceClusterMetaUpdated = false;
    private boolean targetClusterMetaUpdated = false;
    private final Map<String, List<String>> progressReportDetail;
    private final ConcurrentLinkedQueue<String> compactionJobQueue;

    public SyncJob() throws IOException {
        this.initConf();
        this.ddlRequestQueue = new ConcurrentLinkedQueue();
        this.ddlRunning = new ConcurrentLinkedQueue();
        this.replicationJobQueue = new ConcurrentLinkedDeque();
        this.replicationJobState = new ConcurrentHashMap<String, String>();
        this.finishedTableStatus = new ConcurrentHashMap<String, String>();
        this.replicationTableStatus = new ConcurrentHashMap<String, Map<String, String>>();
        this.progressReportDetail = new ConcurrentHashMap<String, List<String>>();
        this.compactionJobQueue = new ConcurrentLinkedQueue();
        this.clusterMetaKeeper = new ClusterMetaKeeper(this);
    }

    public void initConf() throws IOException {
        URL versionUrl = this.getClass().getClassLoader().getResource("build.info");
        if (versionUrl == null) {
            LOG.warn("build.info cannot be found.");
        } else {
            List<String> buildInfo = Files.readAllLines(Paths.get(versionUrl.getPath(), new String[0]));
            for (String line : buildInfo) {
                LOG.info(line);
            }
        }
        InputStream syncInputStream = this.getClass().getClassLoader().getResourceAsStream("sync.properties");
        Properties syncProperties = new Properties();
        if (syncInputStream == null) {
            throw new IOException("sync.properties cannot be found.");
        }
        syncProperties.load(syncInputStream);
        InputStream hostsInputStream = this.getClass().getClassLoader().getResourceAsStream("hosts.properties");
        Properties hostsProperties = new Properties();
        if (hostsInputStream != null) {
            hostsProperties.load(hostsInputStream);
        }
        this.syncConf = new SyncConf(syncProperties, hostsProperties);
        LOG.info("config one_time_run_mode: {}", (Object)this.syncConf.isOneTimeRunMode());
        LOG.info("config source_fe_host: {}", (Object)this.syncConf.getSourceFeHost());
        LOG.info("config target_fe_host: {}", (Object)this.syncConf.getTargetFeHost());
        LOG.info("config target_cluster_storage_volume: {}", (Object)this.syncConf.getTargetClusterStorageVolume());
        LOG.info("config target_cluster_replication_num: {}", (Object)this.syncConf.getTargetClusterReplicationNum());
        LOG.info("config target_cluster_max_disk_used_percent: {}", (Object)this.syncConf.getTargetClusterMaxDiskUsedPct());
        LOG.info("config jdbc_connect_timeout_ms: {}", (Object)this.syncConf.getJdbcConnectTimeoutMs());
        LOG.info("config jdbc_socket_timeout_ms: {}", (Object)this.syncConf.getJdbcSocketTimeoutMs());
        LOG.info("config max_replication_data_size_per_job_in_gb: {}", (Object)this.syncConf.getMaxReplicationDataSizePerJobInGB());
        LOG.info("config meta_job_interval_seconds: {}", (Object)this.syncConf.getMetaJobIntervalSeconds());
        LOG.info("config meta_job_threads: {}", (Object)this.syncConf.getMetaJobThreads());
        LOG.info("config ddl_job_interval_seconds: {}", (Object)this.syncConf.getDdlJobIntervalSeconds());
        LOG.info("config ddl_job_batch_size: {}", (Object)this.syncConf.getDdlJobBatchSize());
        LOG.info("config ddl_job_allow_drop_target_only: {}", (Object)this.syncConf.isDdlJobAllowDropTargetOnly());
        LOG.info("config ddl_job_allow_drop_partition_target_only: {}", (Object)this.syncConf.isDdlJobAllowDropPartitionTargetOnly());
        LOG.info("config ddl_job_allow_drop_schema_change_table: {}", (Object)this.syncConf.isDdlJobAllowDropSchemaChangeTable());
        LOG.info("config ddl_job_allow_drop_inconsistent_partition: {}", (Object)this.syncConf.isDdlJobAllowDropInconsistentPartition());
        LOG.info("config ddl_job_allow_drop_inconsistent_time_partition: {}", (Object)this.syncConf.isDdlJobAllowDropInconsistentTimePartition());
        LOG.info("config enable_materialized_view_sync: {}", (Object)this.syncConf.isEnableMaterializedViewSync());
        LOG.info("config ddl_job_allow_drop_materialized_view_target_only: {}", (Object)this.syncConf.isDdlJobAllowDropMaterializedViewTargetOnly());
        LOG.info("config ddl_job_allow_drop_inconsistent_materialized_view: {}", (Object)this.syncConf.isDdlJobAllowDropInconsistentMaterializedView());
        LOG.info("config enable_view_sync: {}", (Object)this.syncConf.isEnableViewSync());
        LOG.info("config ddl_job_allow_drop_view_target_only: {}", (Object)this.syncConf.isDdlJobAllowDropViewTargetOnly());
        LOG.info("config ddl_job_allow_drop_inconsistent_view: {}", (Object)this.syncConf.isDdlJobAllowDropInconsistentView());
        LOG.info("config enable_bitmap_index_sync: {}", (Object)this.syncConf.isEnableBitmapIndexSync());
        LOG.info("config ddl_job_allow_drop_bitmap_index_target_only: {}", (Object)this.syncConf.isDdlJobAllowDropBitmapIndexTargetOnly());
        LOG.info("config replication_job_interval_seconds: {}", (Object)this.syncConf.getReplicationJobIntervalSeconds());
        LOG.info("config replication_job_batch_size: {}", (Object)this.syncConf.getReplicationJobBatchSize());
        LOG.info("config report_interval_seconds: {}", (Object)this.syncConf.getReportIntervalSeconds());
        LOG.info("config compaction_enable: {}", (Object)this.syncConf.isCompactionEnable());
        LOG.info("config compaction_job_interval_seconds: {}", (Object)this.syncConf.getCompactionJobIntervalSeconds());
        LOG.info("config compaction_job_batch_size: {}", (Object)this.syncConf.getCompactionJobBatchSize());
        LOG.info("config include databases: {}", (Object)Utils.toJson(this.syncConf.getIncludeDbs()));
        LOG.info("config include tables: {}", (Object)Utils.toJson(this.syncConf.getIncludeTables()));
        LOG.info("config exclude databases: {}", (Object)Utils.toJson(this.syncConf.getExcludeDbs()));
        LOG.info("config exclude tables: {}", (Object)Utils.toJson(this.syncConf.getExcludeTables()));
        LOG.info("config source cluster hosts: {}", (Object)this.syncConf.getSourceClusterHosts());
        LOG.info("config target cluster hosts: {}", (Object)this.syncConf.getTargetClusterHosts());
        LOG.info("config enable_table_property_sync: {}", (Object)this.syncConf.isEnableTablePropertySync());
        LOG.info("config target_cluster_use_builtin_storage_volume_only: {}", (Object)this.syncConf.isUseBuiltinStorageVolumeOnTarget());
    }

    public void start() throws InterruptedException {
        int httpPort;
        Thread metaHandler = new Thread(() -> {
            LOG.info("meta handler started.");
            while (true) {
                try {
                    while (true) {
                        this.clusterMetaKeeper.updateClusterMeta();
                        if (this.isSourceClusterMetaUpdated() && this.isTargetClusterMetaUpdated()) {
                            this.clusterMetaKeeper.produceDDL();
                            this.clusterMetaKeeper.produceReplicationJob();
                        } else {
                            LOG.warn("Cannot update cluster meta, waiting...");
                        }
                        Thread.sleep((long)this.syncConf.getMetaJobIntervalSeconds() * 1000L);
                    }
                }
                catch (Exception e) {
                    LOG.error("Failed to update cluster meta.", (Throwable)e);
                    continue;
                }
                break;
            }
        });
        metaHandler.setName("meta-handler");
        Thread ddlHandler = new Thread(() -> {
            LOG.info("DDL handler started.");
            while (true) {
                try {
                    while (true) {
                        Thread.sleep((long)this.syncConf.getDdlJobIntervalSeconds() * 1000L);
                        this.executeDDL();
                    }
                }
                catch (Exception e) {
                    LOG.error("Failed to send DDL task", (Throwable)e);
                    continue;
                }
                break;
            }
        });
        ddlHandler.setName("ddl-handler");
        Thread replicationJobHandler = new Thread(() -> {
            LOG.info("replication handler started.");
            while (true) {
                try {
                    while (true) {
                        Thread.sleep((long)this.syncConf.getReplicationJobIntervalSeconds() * 1000L);
                        this.sendReplicationJob();
                    }
                }
                catch (Exception e) {
                    LOG.error("Failed to send replication job", (Throwable)e);
                    continue;
                }
                break;
            }
        });
        replicationJobHandler.setName("replication-job-handler");
        Thread reportHandler = new Thread(() -> {
            LOG.info("report handler started.");
            while (true) {
                try {
                    while (true) {
                        this.report();
                        Thread.sleep((long)this.syncConf.getReportIntervalSeconds() * 1000L);
                    }
                }
                catch (Exception e) {
                    LOG.error("Failed to report sync progress", (Throwable)e);
                    continue;
                }
                break;
            }
        });
        reportHandler.setName("sync-reporter");
        if (this.syncConf.isCompactionEnable()) {
            Thread compactionJobHandler = new Thread(() -> {
                LOG.info("compaction handler started.");
                while (true) {
                    try {
                        while (true) {
                            Thread.sleep((long)this.syncConf.getCompactionJobIntervalSeconds() * 1000L);
                            this.executeCompaction();
                        }
                    }
                    catch (Exception e) {
                        LOG.error("Failed to send compaction task", (Throwable)e);
                        continue;
                    }
                    break;
                }
            });
            compactionJobHandler.setName("compaction-job-handler");
            compactionJobHandler.start();
        }
        if ((httpPort = this.syncConf.getHttpPort()) > 0) {
            Thread nettyHandler = new Thread(() -> {
                LOG.info("Netty handler started.");
                NioEventLoopGroup bossGroup = new NioEventLoopGroup();
                NioEventLoopGroup workerGroup = new NioEventLoopGroup();
                try {
                    ServerBootstrap bootstrap = new ServerBootstrap();
                    ((ServerBootstrap)bootstrap.group(bossGroup, workerGroup).channel(NioServerSocketChannel.class)).childHandler(new SyncChannelInitalizer(this.clusterMetaKeeper, this.replicationJobState, this.replicationTableStatus, this.progressReportDetail, this.finishedTableStatus, this.replicationJobQueue));
                    ChannelFuture cf = bootstrap.bind(httpPort);
                    cf.addListener(future -> {
                        if (future.isSuccess()) {
                            LOG.info("The HTTP service runs on port " + httpPort);
                        } else {
                            Throwable cause = future.cause();
                            LOG.error("Failed to open HTTP service.", cause);
                        }
                    });
                }
                catch (Exception e) {
                    LOG.error("Failed to start netty handler.", (Throwable)e);
                    workerGroup.shutdownGracefully();
                    bossGroup.shutdownGracefully();
                }
            });
            nettyHandler.setName("netty-handler");
            nettyHandler.start();
        }
        metaHandler.start();
        ddlHandler.start();
        replicationJobHandler.start();
        reportHandler.start();
        metaHandler.join();
        ddlHandler.join();
        replicationJobHandler.join();
        reportHandler.join();
    }

    private void executeDDL() {
        if (!this.clusterMetaKeeper.getTargetClusterInfo().isConnectionReady()) {
            LOG.info("Waiting for target cluster information to be updated.");
            return;
        }
        for (int sentCount = 0; !this.ddlRequestQueue.isEmpty() && sentCount < this.syncConf.getDdlJobBatchSize(); ++sentCount) {
            DDLJob ddlJob = this.ddlRequestQueue.poll();
            if (ddlJob == null) {
                LOG.warn("Failed to poll DDL job, waiting...");
                return;
            }
            this.ddlRunning.offer(ddlJob);
            Utils.execDDLJob(ddlJob, this.clusterMetaKeeper.getTargetClusterInfo());
            this.ddlRunning.remove(ddlJob);
        }
    }

    private void executeCompaction() {
        if (!this.clusterMetaKeeper.getTargetClusterInfo().isConnectionReady()) {
            LOG.info("Waiting for target cluster information to be updated.");
            return;
        }
        for (int sentCount = 0; !this.compactionJobQueue.isEmpty() && sentCount < this.syncConf.getCompactionJobBatchSize(); ++sentCount) {
            String compactionJob = this.compactionJobQueue.poll();
            if (compactionJob == null) {
                LOG.warn("Failed to poll compaction job, waiting...");
                return;
            }
            Utils.execCompactionJob(compactionJob, this.clusterMetaKeeper.getTargetClusterInfo());
        }
    }

    private void sendReplicationJob() throws InterruptedException {
        if (!this.clusterMetaKeeper.getTargetClusterInfo().isConnectionReady()) {
            LOG.info("Waiting for target cluster information to be updated.");
            return;
        }
        int replicationJobBatchSize = this.syncConf.getReplicationJobBatchSize();
        int sentCount = 0;
        while (!this.replicationJobQueue.isEmpty() && sentCount < replicationJobBatchSize) {
            ReplicationJob job = this.replicationJobQueue.poll();
            if (job == null) {
                LOG.warn("Failed to poll replication job, waiting...");
                return;
            }
            List<Boolean> result = Arrays.asList(false, false);
            String jobToken = job.getJobToken();
            for (int i = 0; i < replicationJobBatchSize / 2 + 1; ++i) {
                result = Utils.sendReplicationJob(job, this.clusterMetaKeeper.getTargetClusterInfo());
                ++sentCount;
                if (result.get(0).booleanValue()) {
                    this.replicationJobState.put(jobToken, JobState.INIT.name() + "_" + Utils.formatCurrentTimeMillis(System.currentTimeMillis()));
                    break;
                }
                if (!result.get(1).booleanValue()) break;
                Thread.sleep((long)this.syncConf.getReplicationJobIntervalSeconds() * 1000L);
            }
            if (result.get(0).booleanValue() || result.get(2).booleanValue()) continue;
            this.replicationJobState.put(jobToken, JobState.SENT_FAILED.name() + "_" + result.get(1) + "_" + System.currentTimeMillis());
        }
    }

    private boolean expiredJobState(String originalStatus) {
        int lastUnderline = originalStatus.lastIndexOf("_");
        String lastRecordTime = originalStatus.substring(lastUnderline + 1);
        return System.currentTimeMillis() - Long.parseLong(lastRecordTime) > 21600000L;
    }

    private void updateTableState(String state, Map<String, String> jobMap, String extraInfo) {
        int k = Integer.parseInt(jobMap.getOrDefault(state, "0")) + 1;
        jobMap.put(state, String.valueOf(k));
        String stateInfo = state + "_" + Utils.formatCurrentTimeMillis(System.currentTimeMillis());
        if (!extraInfo.isEmpty()) {
            stateInfo = stateInfo + "_" + extraInfo;
        }
        jobMap.put("LAST_STATUS", stateInfo);
    }

    private String convertJobTokenToDbAndTblName(String jobToken) {
        String identifier = jobToken.split("_")[0];
        String[] dbAndTable = identifier.split("-");
        long dbId = Long.parseLong(dbAndTable[0]);
        long tableId = Long.parseLong(dbAndTable[1]);
        LOG.debug("Convert job token {} to db and table name. dbId: {}, tableId: {}", (Object)jobToken, (Object)dbId, (Object)tableId);
        Database db = this.clusterMetaKeeper.getTargetClusterInfo().getDb(dbId);
        String dbName = db.getDbName();
        Table table = db.getTable(tableId);
        if (table == null) {
            LOG.info("Ignore replication job {} for deleted table.", (Object)jobToken);
            return null;
        }
        String tableName = table.getTableName();
        return dbName + "." + tableName;
    }

    private void report() {
        if (!this.clusterMetaKeeper.getTargetClusterInfo().isConnectionReady()) {
            LOG.info("Waiting for target cluster information to be updated.");
            return;
        }
        int ddlPending = this.ddlRequestQueue.size();
        LOG.info("DDL job queue size: {}", (Object)ddlPending);
        int jobPending = this.replicationJobQueue.size();
        LOG.info("Replication job queue size: {}", (Object)jobPending);
        long totalSizeToSync = 0L;
        for (ReplicationJob job : this.replicationJobQueue) {
            totalSizeToSync += job.getSrcTableDataSize();
        }
        LOG.info("Total estimated size of data pending synchronization: {}", (Object)totalSizeToSync);
        this.clusterMetaKeeper.reportDbProgress();
        int total = 0;
        int sent = 0;
        int ddlRunning = this.ddlRunning.size();
        int jobRunning = 0;
        int finished = 0;
        int sentFailed = 0;
        int failed = 0;
        int unknown = 0;
        HashMap<String, String> failedJobStatus = new HashMap<String, String>();
        HashMap<String, JobProgress.RunningJob> runningTables = new HashMap<String, JobProgress.RunningJob>();
        ArrayList<String> needRemoveFromJobState = new ArrayList<String>();
        JobProgress jobProgress = Utils.getJobProgress(this.clusterMetaKeeper.getTargetClusterInfo());
        Map<String, JobProgress.RunningJob> runningJobs = jobProgress.getRunningJobs();
        List<String> committedIds = jobProgress.getCommittedIds();
        List<String> abortedIds = jobProgress.getAbortedIds();
        for (String jobToken : this.replicationJobState.keySet()) {
            String jobId = jobToken.split("_")[1];
            String dbAndTblName = this.convertJobTokenToDbAndTblName(jobToken);
            if (dbAndTblName == null) {
                needRemoveFromJobState.add(jobToken);
                continue;
            }
            Map jobMap = this.replicationTableStatus.getOrDefault(dbAndTblName, new ConcurrentHashMap());
            String originalStatus = this.replicationJobState.get(jobToken);
            if (originalStatus.startsWith(JobState.FINISHED.name())) {
                ++finished;
                this.updateTableState(JobState.FINISHED.name(), jobMap, "");
                if (this.expiredJobState(originalStatus)) {
                    needRemoveFromJobState.add(jobToken);
                }
            } else if (originalStatus.startsWith(JobState.FAILED.name())) {
                ++failed;
                this.updateTableState(JobState.FAILED.name(), jobMap, "");
                if (this.expiredJobState(originalStatus)) {
                    needRemoveFromJobState.add(jobToken);
                }
            } else if (originalStatus.startsWith(JobState.SENT_FAILED.name())) {
                ++sentFailed;
                this.updateTableState(JobState.SENT_FAILED.name(), jobMap, originalStatus);
                if (this.expiredJobState(originalStatus)) {
                    needRemoveFromJobState.add(jobToken);
                }
            } else {
                String status = this.clusterMetaKeeper.getTxnStatus(jobToken);
                if (committedIds.contains(jobId)) {
                    status = JobState.FINISHED.name();
                } else if (abortedIds.contains(jobId)) {
                    status = JobState.FAILED.name();
                } else if (runningJobs.containsKey(jobId)) {
                    status = JobState.RUNNING.name();
                }
                if (!status.equals(JobState.UNKNOWN.name())) {
                    this.replicationJobState.put(jobToken, status + "_" + System.currentTimeMillis());
                }
                if (status.equals(JobState.RUNNING.name())) {
                    ++jobRunning;
                    runningTables.put(dbAndTblName, runningJobs.get(jobId));
                    this.updateTableState(JobState.RUNNING.name(), jobMap, "");
                } else if (status.equals(JobState.FINISHED.name())) {
                    ++finished;
                    this.finishedTableStatus.put(dbAndTblName, Utils.formatCurrentTimeMillis(System.currentTimeMillis()));
                    this.updateTableState(JobState.FINISHED.name(), jobMap, "");
                    if (this.syncConf.isCompactionEnable()) {
                        this.offerCompactionJob(dbAndTblName);
                    }
                } else if (status.startsWith(JobState.FAILED.name())) {
                    ++failed;
                    failedJobStatus.put(dbAndTblName, JobState.FAILED.name() + "_" + jobToken + "_" + status);
                    this.updateTableState(JobState.FAILED.name(), jobMap, jobToken + "_" + status);
                } else if (originalStatus.startsWith(JobState.INIT.name()) && status.equals(JobState.UNKNOWN.name())) {
                    ++sent;
                    this.updateTableState(JobState.SENT.name(), jobMap, jobToken + "_" + originalStatus + "_" + status);
                } else {
                    ++unknown;
                    failedJobStatus.put(dbAndTblName, JobState.UNKNOWN.name() + "_" + jobToken + "_" + originalStatus);
                    this.finishedTableStatus.put(dbAndTblName, Utils.formatCurrentTimeMillis(System.currentTimeMillis()));
                    this.updateTableState(JobState.UNKNOWN.name(), jobMap, jobToken + "_" + originalStatus);
                }
            }
            ++total;
            this.replicationTableStatus.put(dbAndTblName, jobMap);
        }
        for (String expiredJobState : needRemoveFromJobState) {
            this.replicationJobState.remove(expiredJobState);
        }
        int compactionPending = this.compactionJobQueue.size();
        double ratio = (total = total - unknown - failed - sentFailed + jobPending + ddlRunning + ddlPending + compactionPending) == 0 ? 0.0 : (double)finished * 1.0 / (double)total;
        LOG.info("Sync job progress: {}, total: {}, ddlPending: {}, ddlRunning: {}, jobPending: {}, sent: {}, jobRunning: {}, finished: {}, failed: {}, sent_failed: {}, unknown: {}, compactionPending: {}", Utils.formatDoubleToPercent(ratio), total, ddlPending, ddlRunning, jobPending, sent, jobRunning, finished, failed, sentFailed, unknown, compactionPending);
        ArrayList emptySnapshotJobs = new ArrayList();
        ArrayList txnTimeoutJobs = new ArrayList();
        ArrayList unknownJobs = new ArrayList();
        HashMap otherJobs = new HashMap();
        for (Map.Entry kv : failedJobStatus.entrySet()) {
            if (((String)kv.getValue()).contains("timeout by txn manager")) {
                txnTimeoutJobs.add(kv.getKey());
                continue;
            }
            if (((String)kv.getValue()).contains("Source snapshots is empty")) {
                emptySnapshotJobs.add(kv.getKey());
                continue;
            }
            if (((String)kv.getValue()).startsWith("UNKNOWN_")) {
                unknownJobs.add(kv.getKey());
                continue;
            }
            otherJobs.put(kv.getKey(), kv.getValue());
        }
        if (!emptySnapshotJobs.isEmpty()) {
            LOG.info("Ignored empty snapshot job detail: {}", (Object)emptySnapshotJobs);
        }
        if (!unknownJobs.isEmpty()) {
            LOG.info("Unknown job detail that may have already been finished: {}", (Object)unknownJobs);
        }
        if (!txnTimeoutJobs.isEmpty()) {
            LOG.warn("Failed txn timeout job detail: {}", (Object)txnTimeoutJobs);
        }
        if (!otherJobs.isEmpty()) {
            LOG.warn("Failed job detail: {}", (Object)otherJobs);
        }
        StringBuilder sb = new StringBuilder();
        for (Map.Entry entry : runningTables.entrySet()) {
            String dbAndTblName = (String)entry.getKey();
            JobProgress.RunningJob job = (JobProgress.RunningJob)entry.getValue();
            if (job == null) {
                sb.append(dbAndTblName).append(", ");
                continue;
            }
            String progress = Utils.formatDoubleToPercent(job.calculateRunningProgress());
            sb.append(dbAndTblName).append(":").append(progress).append(":").append(job.getState()).append(":").append(job.getFinishedTaskNum()).append("/").append(job.getTotalTaskNum()).append(":").append(job.getTxnId()).append(", ");
        }
        LOG.info("Running table detail: {}", (Object)sb.toString());
        int finishedTableCount = 0;
        int n = this.clusterMetaKeeper.getAllSourceTables().size();
        List<String> ignoredTables = this.clusterMetaKeeper.getAllIgnoredTables();
        List<String> dynamicIgnoredTables = this.clusterMetaKeeper.getDynamicIgnoredTables();
        List<String> partialReplicationTables = this.clusterMetaKeeper.getPartialReplicationTables();
        ArrayList<String> unfinishedTable = new ArrayList<String>();
        ArrayList<String> expiredTable = new ArrayList<String>();
        for (String sourceTable : this.clusterMetaKeeper.getAllSourceTables()) {
            Map jobMap = this.replicationTableStatus.computeIfAbsent(sourceTable, k -> new ConcurrentHashMap());
            int finishedCount = Integer.parseInt(jobMap.getOrDefault(JobState.FINISHED.name(), "0")) + Integer.parseInt(jobMap.getOrDefault(JobState.UNKNOWN.name(), "0"));
            if ((finishedCount > 0 || ignoredTables.contains(sourceTable)) && !partialReplicationTables.contains(sourceTable)) {
                ++finishedTableCount;
            } else {
                unfinishedTable.add(sourceTable);
            }
            if (dynamicIgnoredTables.contains(sourceTable)) continue;
            expiredTable.add(sourceTable);
        }
        double finishedTableRatio = n == 0 ? 0.0 : (double)finishedTableCount * 1.0 / (double)n;
        double expiredTableRatio = n == 0 ? 0.0 : (double)expiredTable.size() * 1.0 / (double)n;
        LOG.info("Sync table progress, finishedTableRatio: {}, expiredTableRatio: {}, total table: {}, finished table: {}, unfinished table: {}, expired table: {}.", (Object)Utils.formatDoubleToPercent(finishedTableRatio), (Object)Utils.formatDoubleToPercent(expiredTableRatio), (Object)n, (Object)finishedTableCount, (Object)unfinishedTable.size(), (Object)expiredTable.size());
        LOG.info("Unfinished table detail: {}", (Object)unfinishedTable);
        LOG.info("Expired table detail: {}", (Object)expiredTable);
        this.progressReportDetail.put("unfinished", unfinishedTable);
        this.progressReportDetail.put("expired", expiredTable);
        for (String key : this.replicationTableStatus.keySet()) {
            if (this.clusterMetaKeeper.getAllSourceTables().contains(key)) continue;
            this.replicationTableStatus.remove(key);
        }
        if (this.syncConf.isOneTimeRunMode() && finishedTableRatio >= 1.0 && ddlPending == 0) {
            LOG.info("All tables have been synchronized, exit.");
            System.exit(0);
        }
    }

    public void removeDroppedTableStatus(String dbAndTblName) {
        this.finishedTableStatus.remove(dbAndTblName);
        this.replicationTableStatus.remove(dbAndTblName);
    }

    public void offerReplicationJob(List<ReplicationJob> jobs) {
        ReplicationJob element;
        ArrayList<ReplicationJob> requests = new ArrayList<ReplicationJob>();
        if (this.syncConf.isOneTimeRunMode()) {
            for (ReplicationJob job : jobs) {
                boolean success = false;
                String dbAndTblName = job.getDbName() + "." + job.getTableName();
                String identifier = job.getJobToken().split("_")[0];
                for (String jobToken : this.replicationJobState.keySet()) {
                    if (jobToken.startsWith(identifier) && !this.clusterMetaKeeper.getPartialReplicationTables().contains(dbAndTblName) && (this.replicationJobState.get(jobToken).startsWith(JobState.FINISHED.name()) || this.replicationJobState.get(jobToken).startsWith(JobState.UNKNOWN.name()))) {
                        success = true;
                        continue;
                    }
                    if (!this.finishedTableStatus.containsKey(dbAndTblName)) continue;
                    success = true;
                }
                if (success) continue;
                requests.add(job);
            }
        } else {
            requests.addAll(jobs);
        }
        ArrayList<ReplicationJob> pendingJobs = new ArrayList<ReplicationJob>();
        while ((element = this.replicationJobQueue.poll()) != null) {
            pendingJobs.add(element);
        }
        pendingJobs.addAll(requests);
        HashMap<String, ReplicationJob> unfinishedTable = new HashMap<String, ReplicationJob>();
        HashMap<String, String> expiredTable = new HashMap<String, String>();
        HashMap<String, ReplicationJob> expiredTableJobs = new HashMap<String, ReplicationJob>();
        for (ReplicationJob job : pendingJobs) {
            String string = job.getDbName() + "." + job.getTableName();
            if (this.finishedTableStatus.containsKey(string)) {
                expiredTable.put(string, this.finishedTableStatus.get(string));
                expiredTableJobs.put(string, job);
                continue;
            }
            unfinishedTable.put(string, job);
        }
        ArrayList expiredEntries = new ArrayList(expiredTable.entrySet());
        expiredEntries.sort(Map.Entry.comparingByValue());
        this.replicationJobQueue.addAll(unfinishedTable.values());
        for (Map.Entry entry : expiredEntries) {
            this.replicationJobQueue.offer((ReplicationJob)expiredTableJobs.get(entry.getKey()));
        }
    }

    public void offerCompactionJob(String dbAndTblName) {
        if (this.clusterMetaKeeper.getTargetClusterInfo().getClusterRunMode() == ClusterInfo.RunMode.SHARED_DATA) {
            return;
        }
        String sql = String.format("ALTER TABLE %s BASE COMPACT", dbAndTblName);
        for (String compactionJob : this.compactionJobQueue) {
            if (!sql.equals(compactionJob)) continue;
            return;
        }
        this.compactionJobQueue.offer(sql);
    }

    public void offerDDLRequest(String dbName, String sql) {
        if (Utils.incompatibleDDL.contains(sql)) {
            LOG.debug("Ignore incompatible DDL job {}.", (Object)sql);
        }
        for (DDLJob ddlJob : this.ddlRequestQueue) {
            if (dbName != null && !dbName.equals(ddlJob.getDbName()) || !sql.equals(ddlJob.getDdlSql())) continue;
            LOG.debug("Ignore duplicated DDL job {}.", (Object)sql);
            return;
        }
        for (DDLJob ddlJob : this.ddlRunning) {
            if (dbName != null && !dbName.equals(ddlJob.getDbName()) || !sql.equals(ddlJob.getDdlSql())) continue;
            LOG.debug("Ignore duplicated running DDL job {}.", (Object)sql);
            return;
        }
        this.ddlRequestQueue.offer(new DDLJob(dbName, sql));
    }

    public SyncConf getSyncConf() {
        return this.syncConf;
    }

    public boolean isSourceClusterMetaUpdated() {
        return this.sourceClusterMetaUpdated;
    }

    public void setSourceClusterMetaUpdated(boolean sourceClusterMetaUpdated) {
        this.sourceClusterMetaUpdated = sourceClusterMetaUpdated;
    }

    public boolean isTargetClusterMetaUpdated() {
        return this.targetClusterMetaUpdated;
    }

    public void setTargetClusterMetaUpdated(boolean targetClusterMetaUpdated) {
        this.targetClusterMetaUpdated = targetClusterMetaUpdated;
    }

    public static void main(String[] args) throws Exception {
        SyncJob job = new SyncJob();
        job.start();
    }

    public static enum JobState {
        UNKNOWN,
        INIT,
        SENT,
        SENT_FAILED,
        RUNNING,
        FAILED,
        FINISHED;

    }
}

