/*
 * Copyright 2024 Alibaba Cloud, Inc. or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package crypto

import (
	"os"
	"syscall"

	"demo/pkg/meta" //FIXME
)

type CipherEntry struct {
	// cipher text path of entry relative to crypto engine's cipher dir, e.g.
	// absolute path on host: $CIPHER_DIR/path/to/encrypted/entry
	// this path: /path/to/encrypted/entry
	Path string

	// Inode number assigned to this path by cachefs meta engine.
	Inode meta.Ino
}

type CryptoEngine interface {
	// Encrypt given plain text entry name under given dir described by cipher entry, return
	// encrypted name.
	EncryptName(dirEntry *CipherEntry, name string) (string, syscall.Errno)

	// Decrypt given cipher entry name under given dir described by cipher entry, return
	// plaintext name and bool indicates whether the name is used internally or not. Caller
	// should ignore/filter internal names in operations such as Readdir.
	DecryptName(dirEntry *CipherEntry, cipherName string) (string, bool, syscall.Errno)

	// Decrypt symlink target
	DecryptSymlinkTarget(ctarget string) (string, error)

	// Get plain text regular file size with path `cPath` from the given cipher text file size.
	CipherSizeToPlainSize(cPath string, size uint64) (uint64, error)

	// Read (and decrypt) len(buf) plain text at off offset from given cipher text file, which
	// has `inode` as inode number assigned by Cachefs, return the number of bytes read to buf
	// and the error, if any.
	// Note that both len(buf) and off refer to plain text file.
	ReadAtCipherText(inode meta.Ino, file *os.File, buf []byte, off int64) (int, error)
}

// Dummy crypto engine that does nothing, can be embeded into other crypto engines to provide no-op
// default implementations.
type DummyCryptoEngine struct{}

func (de *DummyCryptoEngine) EncryptName(dirEntry *CipherEntry, name string) (string, syscall.Errno) {
	return name, 0
}

func (de *DummyCryptoEngine) DecryptName(dirEntry *CipherEntry, cname string) (string, bool, syscall.Errno) {
	return cname, false, 0
}

func (de *DummyCryptoEngine) DecryptSymlinkTarget(ctarget string) (string, error) {
	return ctarget, nil
}

func (de *DummyCryptoEngine) CipherSizeToPlainSize(cPath string, size uint64) (uint64, error) {
	return size, nil
}

func (de *DummyCryptoEngine) ReadAtCipherText(inode meta.Ino, file *os.File, buf []byte, off int64) (int, error) {
	return file.ReadAt(buf, off)
}
