package com.lingyang.qaexchanger.lingyangqaexchanger.util;

import com.lingyang.qaexchanger.lingyangqaexchanger.constants.BaseConstants;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.Map;

/**
 * @Description: 签名算法工具类
 * @Title: SignUtil
 * @Package com.lingyang.qaexchanger.lingyangqaexchanger.util
 * @Author: Levi
 * @Copyright Lingyang
 * @CreateTime: 2022/5/9 10:01
 */
public class SignUtil {

    /**
     * 签名加密
     *
     * @param params  请求参数
     * @param secret  签名密钥
     * @param signMethod 签名摘要算法
     * @return
     * @throws IOException
     */
    public static String sign(Map<String, String> params, String secret, String signMethod) throws IOException {
        // 第一步：检查参数是否已经排序
        String[] keys = params.keySet().toArray(new String[0]);
        Arrays.sort(keys);

        // 第二步：把所有参数名和参数值串在一起
        StringBuilder query = new StringBuilder();
        if (BaseConstants.SIGN_LOWER_MD5.equals(signMethod)) {
            query.append(secret);
        }

        for (String key : keys) {
            String value = params.get(key);
            if (StringUtils.isNotEmpty(key) && StringUtils.isNotEmpty(value)) {
                query.append(key).append(value);
            }
        }

        // 第三步：使用MD5加密
        query.append(secret);
        byte[] bytes = encryptMD5(query.toString());

        // 第四步：把二进制转化为大写的十六进制(正确签名应该为32大写字符串，此方法需要时使用)
        return byte2hex(bytes);
    }

    /**
     * 签名验证
     *
     * @param params
     * @param secret
     * @param signMethod
     * @param sign
     * @return
     * @throws IOException
     */
    public static boolean verify(Map<String, String> params, String secret, String signMethod, String sign) throws IOException {
        String reqSign = sign(params, secret, signMethod);
        return StringUtils.equalsIgnoreCase(reqSign, sign);
    }

    /**
     * 对字节流进行MD5摘要加密
     *
     * @param data 字节流
     * @return
     * @throws IOException
     */
    private static byte[] encryptMD5(String data) throws IOException {
        return encryptMD5(data.getBytes(BaseConstants.SIGN_UTF8));
    }

    /**
     * 对字节流进行MD5摘要加密
     *
     * @param data 字节流
     * @return
     * @throws IOException
     */
    private static byte[] encryptMD5(byte[] data) throws IOException {
        byte[] bytes = null;
        try {
            MessageDigest md = MessageDigest.getInstance(BaseConstants.SIGN_UPPER_MD5);
            bytes = md.digest(data);
        } catch (GeneralSecurityException gse) {
            throw new IOException(gse.toString());
        }
        return bytes;
    }

    /**
     * 对字节流进行进制转换
     *  - 二进制转化为大写的十六进制
     *  - 正确签名应该为32大写字符串
     *
     * @param bytes 字节流
     * @return
     */
    public static String byte2hex(byte[] bytes) {
        StringBuilder sign = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            String hex = Integer.toHexString(bytes[i] & 0xFF);
            if (hex.length() == 1) {
                sign.append("0");
            }
            sign.append(hex.toUpperCase());
        }
        return sign.toString();
    }

}
