package com.lingyang.qaexchanger.lingyangqaexchanger.util;

import com.lingyang.qaexchanger.lingyangqaexchanger.constants.BaseConstants;
import lombok.SneakyThrows;
import org.codehaus.jackson.JsonEncoding;
import org.codehaus.jackson.JsonGenerator;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.annotate.JsonSerialize.Inclusion;
import org.codehaus.jackson.type.JavaType;
import org.codehaus.jackson.type.TypeReference;
import org.springframework.util.StringUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;

/**
 * @Description: JSON工具
 * @Title: JasonUtil
 * @Package com.lingyang.qaexchanger.lingyangqaexchanger.util
 * @Author: Levi
 * @Copyright Lingyang
 * @CreateTime: 2022/5/8 23:00
 */
public class JasonUtil {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    /**
     * JSON转对象
     *
     * @param jason
     * @param clazz
     * @return
     * @throws JsonParseException
     * @throws JsonMappingException
     * @throws IOException
     */
    @SneakyThrows
    public static <T extends Object> T fromJson(String jason, Class<T> clazz)  {
        if (StringUtils.isEmpty(jason) || clazz == null) {
            return null;
        } else {
            T obj = MAPPER.readValue(jason, clazz);
            return obj;
        }
    }

    /**
     * Convert an object to joson format text.
     *
     * @param obj The object to be converted to jason
     * @return Joson format text
     * @throws IOException
     */
    public static String toJson(Object obj) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        JsonGenerator jsonGenerator = MAPPER.getJsonFactory().createJsonGenerator(baos, JsonEncoding.UTF8);
        MAPPER.setSerializationInclusion(Inclusion.NON_EMPTY);
        MAPPER.setSerializationInclusion(Inclusion.NON_NULL);
        jsonGenerator.writeObject(obj);
        String json = baos.toString();
        baos.close();
        return json;
    }

    public static Map<String, Object> objectToMap(Object obj) throws Exception{
        if (Objects.isNull(obj)) {
            return null;
        }
        Map<String, Object> map = new HashMap<>(16);
        Field[] fields = obj.getClass().getDeclaredFields();

        for(Field field:fields){
            field.setAccessible(true);
            map.put(field.getName(), field.get(obj));
        }

        return map;
    }

    public static String mapToJson(Map<String, ?> map) throws IOException {
        String json = MAPPER.writeValueAsString(map);
        return json;
    }

    public static Map<String, Object> jsonToMap(String json) throws IOException {
        Map<String, Object> map = MAPPER.readValue(json, new TypeReference<LinkedHashMap<String, Object>>() {
        });
        return map;
    }

    @SuppressWarnings("unchecked")
    public static <T extends Collection<?>> T jasonToCollection(String src, Class<T> collectionClass,
                                                                Class<?>... valueType) throws Exception {
        JavaType javaType = MAPPER.getTypeFactory().constructParametricType(collectionClass, valueType);
        return (T) MAPPER.readValue(src, javaType);
    }

    /**
     * 单位缩进字符串。
     */
    private static String SPACE = "&nbsp;";

    /**
     * 返回格式化JSON字符串。
     *
     * @param json 未格式化的JSON字符串。
     * @return 格式化的JSON字符串。
     */
    public static String formatJson(String json) {
        if (json == null) {
            return "json不能为空";
        }
        StringBuffer result = new StringBuffer();

        int length = json.length();
        int number = 0;
        char key = 0;

        // 遍历输入字符串。
        for (int i = 0; i < length; i++) {
            // 1、获取当前字符。
            key = json.charAt(i);

            // 2、如果当前字符是前方括号、前花括号做如下处理：
            if ((key == '[') || (key == '{')) {
                // （1）如果前面还有字符，并且字符为“：”，打印：换行和缩进字符字符串。
                if ((i - 1 > 0) && (json.charAt(i - 1) == ':')) {
                    result.append("<br/>");
                    result.append(indent(number));
                }

                // （2）打印：当前字符。
                result.append(key);

                // （3）前方括号、前花括号，的后面必须换行。打印：换行。
                result.append("<br/>");

                // （4）每出现一次前方括号、前花括号；缩进次数增加一次。打印：新行缩进。
                number++;
                result.append(indent(number));

                // （5）进行下一次循环。
                continue;
            }

            // 3、如果当前字符是后方括号、后花括号做如下处理：
            if ((key == ']') || (key == '}')) {
                // （1）后方括号、后花括号，的前面必须换行。打印：换行。
                result.append("<br/>");

                // （2）每出现一次后方括号、后花括号；缩进次数减少一次。打印：缩进。
                number--;
                result.append(indent(number));

                // （3）打印：当前字符。
                result.append(key);

                // （4）如果当前字符后面还有字符，并且字符不为“，”，打印：换行。
                if (((i + 1) < length) && (json.charAt(i + 1) != ',')) {
                    result.append("<br/>");
                }

                // （5）继续下一次循环。
                continue;
            }

            // 4、如果当前字符是逗号。逗号后面换行，并缩进，不改变缩进次数。
            if ((key == ',')) {
                result.append(key);
                result.append("<br/>");
                result.append(indent(number));
                continue;
            }

            // 5、打印：当前字符。
            result.append(key);
        }

        return result.toString();
    }

    /**
     * 返回指定次数的缩进字符串。每一次缩进三个空格，即SPACE。
     *
     * @param number 缩进次数。
     * @return 指定缩进次数的字符串。
     */
    private static String indent(int number) {
        StringBuffer result = new StringBuffer();
        for (int i = 0; i < number * BaseConstants.INT_5 + BaseConstants.INT_2; i++) {
            result.append(SPACE);
        }
        return result.toString();
    }
}
