package com.lingyang.qaexchanger.lingyangqaexchanger.service.impl;

import com.lingyang.qaexchanger.lingyangqaexchanger.constants.BaseConstants;
import com.lingyang.qaexchanger.lingyangqaexchanger.constants.BusinessErrorCode;
import com.lingyang.qaexchanger.lingyangqaexchanger.api.dto.CommonDTO;
import com.lingyang.qaexchanger.lingyangqaexchanger.exception.BusinessException;
import com.lingyang.qaexchanger.lingyangqaexchanger.service.ISignService;
import com.lingyang.qaexchanger.lingyangqaexchanger.util.SignUtil;
import com.lingyang.qaexchanger.lingyangqaexchanger.util.WebhookSignUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;

/**
 * @Description: 签名服务实现类
 * @Title: SignServiceImpl
 * @Package com.lingyang.qaexchanger.lingyangqaexchanger.service.impl
 * @Author: Levi
 * @Copyright Lingyang
 * @CreateTime: 2022/5/9 10:52
 */
@Slf4j
@Service
public class SignServiceImpl implements ISignService {

    /**
     * 签名验证开关
     */
    @Value("${business.sign.enabled}")
    private Boolean signEnabled;

    /**
     * webhook密钥信息
     */
    @Value("${client.ali-yun.webhook.secret}")
    private String webhookSecret;

    /**
     * 应用凭证信息
     */
    @Value("${business.auth.dt-mille.appkey}")
    private String dtMillerAppKey;

    /**
     * 应用凭证信息
     */
    @Value("${business.auth.dt-mille.appsecret}")
    private String dtMillerAppSecret;

    @Override
    public void verifyTokenSign(Map<String, String> params, CommonDTO commonDTO) {
        // 签名开关处理
        if (!signEnabled) {
            return;
        }
        String requestAppKey = params.get("appKey");
        if (!StringUtils.equalsIgnoreCase(dtMillerAppKey, requestAppKey)) {
            log.error("SIGN_VERIFY_TOKEN, oauth appKey is not null. requestId={}", commonDTO.getRequestId());
            throw new BusinessException(BusinessErrorCode.ERROR_0101002, "无效的appKey");
        }
        // 执行签名验证
        boolean isSuccess = false;
        try {
            isSuccess = SignUtil.verify(params, dtMillerAppSecret, commonDTO.getSignMethod(), commonDTO.getSign());
        } catch (Exception e) {
            log.error("SIGN_VERIFY_TOKEN, verify sign fail. requestId=" + commonDTO.getRequestId(), e);
            throw new BusinessException(BusinessErrorCode.ERROR_0101003, "签名验证失败");
        }
        // 签名验证不通过则终止当前请求
        if (!isSuccess) {
            log.info("SIGN_VERIFY_TOKEN, verify sign is not legal. requestId={}", commonDTO.getRequestId());
            throw new BusinessException(BusinessErrorCode.ERROR_0101001, "签名错误");
        }
    }

    @Override
    public void verifyWebhookSign(Long timestamp, String nonce, String signature, String requestId) {
        // 签名开关处理
        if (!signEnabled) {
            return;
        }
        boolean isLegal = WebhookSignUtil.makeSignature(webhookSecret, String.valueOf(timestamp), nonce, signature);
        if (!isLegal) {
            log.warn("SIGN_VERIFY_WEBHOOK, webhook signature verify fail, requestId: {}, signature: {}", requestId, signature);
            throw new BusinessException(BusinessErrorCode.ERROR_0101200, "签名错误");
        }
    }
}
