//
//  UEPDataCenter.h
//  uep
//
//  Created by JL on 2020/10/16.
//  Copyright © 2020 Alipay. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "UEPClickEvent.h"
#import "UEPPageEvent.h"
#import "UEPActBaseNode.h"

NS_ASSUME_NONNULL_BEGIN

typedef id (^UEPGetDataBlock)();
typedef void(^UEPGetDataAsyncBlock)(id result);

@protocol CEPListener <NSObject>

/**
 *  @param actionName 回调唯一标识，注册时或配置中定义
 *  @param type 匹配类型， 包括：match(匹配成功) , timeout: (完成部分匹配，但匹配超时)
 *  @param result 匹配返回的结果，配置定义了output，则返回map结构，否则返回匹配结棍数组
 *  任务配置中output可自定义匹配结果，result以自定义结构返回数据，若未配置output, 则以检查outputDefault = false,未配置则默认返回所有pattern对应匹配事件列表
 *  result 结果
 *   配置output                                               ->                  result以output定义输出
 *   未配置output,未配置outputDefault          ->                  result以系统默认的结构返回所有匹配事件
 *   未配置output, outputDefault = false        ->                  result返回nil
 */
- (void)onTrigger:(NSString *)actionName matchType:(NSString *)type result:(id)result;

@end

@interface UEPDataCenter : NSObject

+ (UEPDataCenter *)dataCenter;

/**
 *  获取最近点击事件，仅限主线程调用
 */
- (UEPClickEvent *)getLastClick;

/**
 *  获取最近页面事件，仅限主线程调用
 */
- (UEPPageEvent *)getLastPage;

/**
 *  上报所有页面jsapi调用信息
 */
- (void)reportPageJSApiData;

#pragma mark - storyline接口
/**
 *  获取UEP框架产生的当前页面page(访问)标识，与storyline结合
 */
- (NSString *)getCurrentPageSeq;

/*
 *  获取UEP框架产生的当前页面Scene(VC)标识，与storyline结合
 */
- (NSString *)getCurrentSceneSeq;

/**
 *  获取指定UIResponder/VC的最新页面访问标识pageSeq;对应PV标识
 */
- (NSString *)getPageSeq:(UIResponder *)responder;

/**
 *  获取指定UIResponder/VC标识，对应VC对象
 */
- (NSString *)getPageToken:(UIResponder *)responder;

/**
 * 获取当前appId, 通过storyline当前页获取，
 */
//- (NSString *)getCurrentAppId;

/**
 * 获取当前url，若存在，通过storyline当前页获取
 */
// - (NSString *)getCurrentUrl;

/**
 *  获取最近操作行为最短路径
 */
-(NSArray<UEPActBaseNode *> *)getShortPath;

/**
 *  获取最近操作行为全路径
 */
- (NSArray<UEPActBaseNode *> *)getFullPath;

// 默认输出路径，输出格式json字符，
// 此接口只包括app/page/click节点，定制输出使用getPathStringWithFilter
- (NSString *)getPathString;

/**
 * 定制输出路径节点，通过UEPOutputFilter入参控制输出规则
 *
 * @return 过滤后输出的Node节点信息，格式：JSON字符串
 * @param UEPOutput Filter定义入参需要检查的node节点，及节点类型
 * UEPActNodeType类型支持按位与操作，具体参考UEPActNodeType定义
 * BOOL(^UEPOutputFilter)(UEPActBaseNode *node, UEPActNodeType nodeType)
 * 注意：filter传空时，输出全部节点
 */
- (NSString *)getPathStringWithFilter:(UEPOutputFilter)filter;

#pragma mark -
/**
 *  获取用户指定自然天内打开次数前top的应用，返回appId和对应打开次数
 *  @param inDay 指定天数，最多30天
 *  @param limit 最多100个
 */
- (NSDictionary<NSString *, NSNumber *> *)getAppsStartInDays:(NSInteger)inDay limit:(NSInteger )limit;

/**
 *  获取指定appId,最近inDay自然天内打开次数，最多30天
 */
- (NSInteger)getAppStartInDays:(NSInteger)inDay appId:(NSString *)appId;

/**
 * 获取最新指定天内，appIds的打开次数
 */
- (NSDictionary<NSString *, NSNumber *> *)getAppsStartInDays:(NSInteger)inDay appIds:(NSArray *)appIds;

#pragma mark - SSP(DataSet)迭代计算数据查询接口

/**
 * DataSet组数据提取接口
  * parameter：configs: [{
                           @"name" : @"mf_dataset_homegrid_apps",
                           @"interval" : @3
                         },
                         {
                           @"name" : @"mf_dataset_xxxx",
                           @"interval" : @3
                         },
                         {
                           @"name" : @"mf_dataset_yyy",
                           @"selectKeys" : @[@"xxx", @"yyy", @"zzzz"],
                           @"interval" : @3
                          }]
    result： {
         @"mf_dataset_homegrid_apps" : @"10000021"
         @"mf_dataset_xxxx" : @5
         @"mf_dataset_yyyy" : @{
             @"xxx" : @5,
             @"yyy" : @"1233232223",
             @"zzz" : @[@"2000001", @"829898889", @"78817233"]
         }
    }
 */
- (NSDictionary *)getDatas:(NSArray<NSDictionary *> *)configs;

/**
 *  @param name  任务名称，任务唯一标识
 *  @param interval 取数据单元(unit)数,即时间长度, 比如5天, 任务unit配置day, interval传5，interval对应配置中group的中的一个值
 */
- (id)getData:(NSString *)name interval:(NSInteger)interval;

/**
 * @param name  任务名称，任务唯一标识
 * @param interval 取数据单元(unit)数, 比如5天, 任务unit配置day, interval传5，interval对应配置中group的中的一个值
 * @param selectKeys 要取值的组合，对应aggregate中多结果中取部分keys
 */
- (id)getData:(NSString *)name selectKeys:(NSArray<NSString *> *)selectKeys interval:(NSInteger)interval;

/**
 * @param name  任务名称，任务唯一标识
 * @param interval 取数据单元(unit)数, 比如5天, 任务unit配置day, interval传5，interval对应配置中group的中的一个值
 * @param selectKeys 要取值的组合，对应aggregate中多结果中取部分keys
 * @param keyby 选择分组key, 只对经过keyby算子的任务有效，指定只返回指定分组结果
 */
- (id)getData:(NSString *)name selectKeys:(NSArray<NSString *> *)selectKeys keyBy:(NSString *)keyby interval:(NSInteger)interval;

/**
 * 分组前缀查询，支持按分组key指定前缀查询
 */
- (id)getData:(NSString *)name keyByPrefix:(NSString *)prefix interval:(NSInteger)interval;

/**
 *  异步获取ssp数据
 *  响应block会在工作线程执行
*/
- (void)getDataAsync:(NSString *)name interval:(NSInteger)interval result:(UEPGetDataAsyncBlock)block;


#pragma mark - SSP(DataSet)迭代计算数据查询接口
/**
 *  检查任务是否存在数据, 不会清除增量任务缓存数据
 *  @param taskName  任务名称，任务唯一标识
 *  @param interval 取数据单元(unit)数, 比如5天, 任务unit配置day, interval传5，interval对应配置中group的中的一个值
 */
- (BOOL)existData:(NSString *)taskName interval:(NSInteger)interval;

/**
 *  检查任务是否存在指定分组指字输出keys数据，不会清除增量任务缓存数据
 *  @param taskName  任务名称，任务唯一标识
 *  @param selectKeys 要取值的组合，对应aggregate中多结果中取部分keys
 *  @param interval 取数据单元(unit)数, 比如5天, 任务unit配置day, interval传5，interval对应配置中group的中的一个值
 */
- (BOOL)existData:(NSString *)taskName selectKeys:(NSArray<NSString *> *)selectKeys interval:(NSInteger)interval;

/**
 *  检查任务是否存在指定分组指字输出keys数据， 不会清除增量任务缓存数据
 *  @param taskName  任务名称，任务唯一标识
 *  @param selectKeys 要取值的组合，对应aggregate中多结果中取部分keys
 *  @param keyby 选择分组key, 只对经过keyby算子的任务有效，指定只返回指定分组结果
 *  @param interval 取数据单元(unit)数, 比如5天, 任务unit配置day, interval传5，interval对应配置中group的中的一个值
 */
- (BOOL)existData:(NSString *)taskName selectKeys:(NSArray<NSString *> *)selectKeys keyBy:(NSString *)keyby interval:(NSInteger)interval;


/**
 *  同上接口
 *  支持多keyby入参，多keyby结果或操作返回
 *  即只要有一个keyby有数据，即返回true
 */
- (BOOL)existData:(NSString *)taskName selectKeys:(NSArray<NSString *> *)selectKeys keyByList:(NSArray<NSString *> *)keys interval:(NSInteger)interval;

#pragma mark - SSP 规则匹配接口

/**
 * SSP任务JSON格式配置字符串
 *  参考 https://yuque.antfin.com/docs/share/4d0eeebe-ca8a-4194-af19-a30b1038efc0?#
 */
- (BOOL)registSSPTask:(NSString *)taskConfig;

/**
 * 批量注册， 入参任务配置数组
 */
- (void)registSSPTasks:(NSArray<NSString *> *)taskConfigs;

// 取消注册，入参任务名
- (void)unregistSSPTask:(NSString *)taskName;

// 批量取消注册，入参任务名列表
- (void)unregistSSPTasks:(NSArray<NSString *> *)tasksList;

#pragma mark - CEP 规则匹配接口

/**
 *   注册规则任务，
 *   @return 注册是否成功
 *      * 同名task再次注册失败
 *      * listener可为空，为空时会发namte = action的广播，listener不为空时且不支持CEPListener协议会注册失败
 *      * 配置异常任务会注册失败，如：未设置有效action且配置中未配置action
 *   @param action 回调业务key，为空则使用taskConfig的action配置值,
 *   @param taskConfig  JSON string，任务配置，具体参考：https://yuque.antfin.com/docs/share/41a08b97-f3b7-4cc8-8439-ad8c74317431?# 《Tigger产品化- CEP部分能力》
 *   @param listener 匹配成功后通知回调实例，实现CEPListener协议接口
 *   注意：action与task支持1对多关系， 但不支持多对1
 */
- (BOOL)registCEPAction:(NSString *)action task:(NSString *)taskConfig observer:(id<CEPListener>)listener;

/**
 *  注册规则任务，同上
 */
- (BOOL)registCEPTask:(NSString *)taskConfig observer:(id<CEPListener>)listener;

// 批量注册任务，入传cep配置数组
- (BOOL)registCEPTasks:(NSArray<NSString *> *)taskConfigs observer:(id<CEPListener>)listener;

/**
 *   取消匹配任务， 入参任务action
 */
- (void)unregistCEPAction:(NSString *)action;

/**
 * 批量取消任务，入参任务名列表
 */
- (void)unregistCEPWithTasks:(NSArray<NSString *> *)tasksList;

@end

NS_ASSUME_NONNULL_END
