/**
 * \file asn1write.h
 *
 * \brief ASN.1 buffer writing functionality
 */
/*
 *  Copyright (C) 2006-2015, ARM Limited, All Rights Reserved
 *  SPDX-License-Identifier: Apache-2.0
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"); you may
 *  not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  This file is part of mbed TLS (https://tls.mbed.org)
 */
#ifndef ANTSSM_ASN1_WRITE_H
#define ANTSSM_ASN1_WRITE_H

#if !defined(ANTSSM_CONFIG_FILE)
#include "antssm/config.h"
#else
#include ANTSSM_CONFIG_FILE
#endif

#include "antssm/asn1.h"

#define ANTSSM_ASN1_CHK_ADD(g, f)                      \
    do                                                  \
    {                                                   \
        if( ( ret = (f) ) < 0 )                         \
            return( ret );                              \
        else                                            \
            (g) += ret;                                 \
    } while( 0 )

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief           Write a length field in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param len       The length value to write.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_len(unsigned char **p, unsigned char *start,
                           size_t len);

/**
 * \brief           Write an ASN.1 tag in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param tag       The tag to write.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_tag(unsigned char **p, unsigned char *start,
                           unsigned char tag);

/**
 * \brief           Write raw buffer data.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param buf       The data buffer to write.
 * \param size      The length of the data buffer.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_raw_buffer(unsigned char **p, unsigned char *start,
                                  const unsigned char *buf, size_t size);

#if defined(ANTSSM_MPI_C)
/**
 * \brief           Write a arbitrary-precision number (#ANTSSM_ASN1_INTEGER)
 *                  in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param X         The MPI to write.
 *                  It must be non-negative.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_mpi( unsigned char **p, unsigned char *start,
                            const mpaas_antssm_mpi_t *X );
#endif /* ANTSSM_MPI_C */

/**
 * \brief           Write a NULL tag (#ANTSSM_ASN1_NULL) with zero data
 *                  in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_null(unsigned char **p, unsigned char *start);

/**
 * \brief           Write an OID tag (#ANTSSM_ASN1_OID) and data
 *                  in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param oid       The OID to write.
 * \param oid_len   The length of the OID.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_oid(unsigned char **p, unsigned char *start,
                           const char *oid, size_t oid_len);

/**
 * \brief           Write an AlgorithmIdentifier sequence in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param oid       The OID of the algorithm to write.
 * \param oid_len   The length of the algorithm's OID.
 * \param par_len   The length of the parameters, which must be already written.
 *                  If 0, NULL parameters are added
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_algorithm_identifier(unsigned char **p,
                                            unsigned char *start,
                                            const char *oid, size_t oid_len,
                                            size_t par_len);

/**
 * \brief           Write a boolean tag (#ANTSSM_ASN1_BOOLEAN) and value
 *                  in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param boolean   The boolean value to write, either \c 0 or \c 1.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_bool(unsigned char **p, unsigned char *start,
                            int boolean);

/**
 * \brief           Write an int tag (#ANTSSM_ASN1_INTEGER) and value
 *                  in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param val       The integer value to write.
 *                  It must be non-negative.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_int(unsigned char **p, unsigned char *start, int val);

/**
 * \brief           Write an enum tag (#ANTSSM_ASN1_ENUMERATED) and value
 *                  in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param val       The integer value to write.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative \c ANTSSM_ERR_ASN1_XXX error code on failure.
 */
int mpaas_antssm_asn1_write_enum(unsigned char **p, unsigned char *start, int val);

/**
 * \brief           Write a string in ASN.1 format using a specific
 *                  string encoding tag.

 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param tag       The string encoding tag to write, e.g.
 *                  #ANTSSM_ASN1_UTF8_STRING.
 * \param text      The string to write.
 * \param text_len  The length of \p text in bytes (which might
 *                  be strictly larger than the number of characters).
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative error code on failure.
 */
int mpaas_antssm_asn1_write_tagged_string(unsigned char **p, unsigned char *start,
                                     int tag, const char *text,
                                     size_t text_len);

/**
 * \brief           Write a string in ASN.1 format using the PrintableString
 *                  string encoding tag (#ANTSSM_ASN1_PRINTABLE_STRING).
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param text      The string to write.
 * \param text_len  The length of \p text in bytes (which might
 *                  be strictly larger than the number of characters).
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative error code on failure.
 */
int mpaas_antssm_asn1_write_printable_string(unsigned char **p,
                                        unsigned char *start,
                                        const char *text, size_t text_len);

/**
 * \brief           Write a UTF8 string in ASN.1 format using the UTF8String
 *                  string encoding tag (#ANTSSM_ASN1_UTF8_STRING).
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param text      The string to write.
 * \param text_len  The length of \p text in bytes (which might
 *                  be strictly larger than the number of characters).
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative error code on failure.
 */
int mpaas_antssm_asn1_write_utf8_string(unsigned char **p, unsigned char *start,
                                   const char *text, size_t text_len);

/**
 * \brief           Write a string in ASN.1 format using the IA5String
 *                  string encoding tag (#ANTSSM_ASN1_IA5_STRING).
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param text      The string to write.
 * \param text_len  The length of \p text in bytes (which might
 *                  be strictly larger than the number of characters).
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative error code on failure.
 */
int mpaas_antssm_asn1_write_ia5_string(unsigned char **p, unsigned char *start,
                                  const char *text, size_t text_len);

/**
 * \brief           Write a bitstring tag (#ANTSSM_ASN1_BIT_STRING) and
 *                  value in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param buf       The bitstring to write.
 * \param bits      The total number of bits in the bitstring.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative error code on failure.
 */
int mpaas_antssm_asn1_write_bitstring(unsigned char **p, unsigned char *start,
                                 const unsigned char *buf, size_t bits);

/**
 * \brief           This function writes a named bitstring tag
 *                  (#ANTSSM_ASN1_BIT_STRING) and value in ASN.1 format.
 *
 *                  As stated in RFC 5280 Appendix B, trailing zeroes are
 *                  omitted when encoding named bitstrings in DER.
 *
 * \note            This function works backwards within the data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer which is used for bounds-checking.
 * \param buf       The bitstring to write.
 * \param bits      The total number of bits in the bitstring.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative error code on failure.
 */
int mpaas_antssm_asn1_write_named_bitstring(unsigned char **p,
                                       unsigned char *start,
                                       const unsigned char *buf,
                                       size_t bits);

/**
 * \brief           Write an octet string tag (#ANTSSM_ASN1_OCTET_STRING)
 *                  and value in ASN.1 format.
 *
 * \note            This function works backwards in data buffer.
 *
 * \param p         The reference to the current position pointer.
 * \param start     The start of the buffer, for bounds-checking.
 * \param buf       The buffer holding the data to write.
 * \param size      The length of the data buffer \p buf.
 *
 * \return          The number of bytes written to \p p on success.
 * \return          A negative error code on failure.
 */
int mpaas_antssm_asn1_write_octet_string(unsigned char **p, unsigned char *start,
                                    const unsigned char *buf, size_t size);

/**
 * \brief           Create or find a specific named_data entry for writing in a
 *                  sequence or list based on the OID. If not already in there,
 *                  a new entry is added to the head of the list.
 *                  Warning: Destructive behaviour for the val data!
 *
 * \param list      The pointer to the location of the head of the list to seek
 *                  through (will be updated in case of a new entry).
 * \param oid       The OID to look for.
 * \param oid_len   The size of the OID.
 * \param val       The associated data to store. If this is \c NULL,
 *                  no data is copied to the new or existing buffer.
 * \param val_len   The minimum length of the data buffer needed.
 *                  If this is 0, do not allocate a buffer for the associated
 *                  data.
 *                  If the OID was already present, enlarge, shrink or free
 *                  the existing buffer to fit \p val_len.
 *
 * \return          A pointer to the new / existing entry on success.
 * \return          \c NULL if if there was a memory allocation error.
 */
mpaas_antssm_asn1_named_data *
mpaas_antssm_asn1_store_named_data(mpaas_antssm_asn1_named_data **list,
                              const char *oid, size_t oid_len,
                              const unsigned char *val,
                              size_t val_len);

#ifdef __cplusplus
}
#endif

#endif /* ANTSSM_ASN1_WRITE_H */
