#!/bin/bash
# 一键发布所有 packages 下的包到 npm
# 排除 tingwu-example

# 颜色输出
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 配置
ROOT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PACKAGES_DIR="$ROOT_DIR/packages"
EXCLUDE_PACKAGES=("tingwu-example")

# 统计
SUCCESS_COUNT=0
FAILED_COUNT=0
SKIPPED_COUNT=0
FAILED_PACKAGES=()
SKIPPED_PACKAGES=()

echo "=========================================="
echo "开始发布所有包到 npm"
echo "=========================================="
echo ""

# 检查 npm 是否可用
if ! command -v npm &> /dev/null; then
  echo -e "${RED}错误: npm 命令不存在${NC}"
  exit 1
fi

# 检查是否已登录
if ! npm whoami &> /dev/null; then
  echo -e "${YELLOW}警告: 未登录到 npm registry${NC}"
  echo "请先运行: npm login"
  exit 1
fi

echo -e "${BLUE}当前登录用户: $(npm whoami)${NC}"
echo ""

# 获取所有包目录
packages=()
for package_dir in "$PACKAGES_DIR"/*; do
  if [ ! -d "$package_dir" ]; then
    continue
  fi
  
  package_name=$(basename "$package_dir")
  
  # 检查是否在排除列表中
  should_exclude=false
  for exclude in "${EXCLUDE_PACKAGES[@]}"; do
    if [ "$package_name" == "$exclude" ]; then
      should_exclude=true
      break
    fi
  done
  
  if [ "$should_exclude" == true ]; then
    continue
  fi
  
  packages+=("$package_name")
done

# 遍历所有包
for package_name in "${packages[@]}"; do
  package_dir="$PACKAGES_DIR/$package_name"
  
  # 检查是否存在 package.json
  if [ ! -f "$package_dir/package.json" ]; then
    echo -e "${YELLOW}跳过 $package_name: package.json 不存在${NC}"
    ((SKIPPED_COUNT++))
    SKIPPED_PACKAGES+=("$package_name")
    continue
  fi
  
  # 获取包名和版本
  pkg_name=$(cd "$package_dir" && node -p "require('./package.json').name" 2>/dev/null || echo "")
  pkg_version=$(cd "$package_dir" && node -p "require('./package.json').version" 2>/dev/null || echo "")
  
  if [ -z "$pkg_name" ] || [ -z "$pkg_version" ]; then
    echo -e "${YELLOW}跳过 $package_name: 无法读取 package.json${NC}"
    ((SKIPPED_COUNT++))
    SKIPPED_PACKAGES+=("$package_name")
    continue
  fi
  
  echo "----------------------------------------"
  echo -e "${BLUE}发布: $pkg_name@$pkg_version${NC}"
  echo "路径: $package_dir"
  echo ""
  
  # 进入包目录并发布
  cd "$package_dir"
  
  # 执行发布
  output=$(npm publish --access public 2>&1)
  result=$?
  
  if [ $result -eq 0 ]; then
    echo -e "${GREEN}✓ 成功发布: $pkg_name@$pkg_version${NC}"
    ((SUCCESS_COUNT++))
  elif echo "$output" | grep -q "previously published\|You cannot publish over"; then
    echo -e "${YELLOW}⚠ 版本已存在: $pkg_name@$pkg_version${NC}"
    ((SKIPPED_COUNT++))
    SKIPPED_PACKAGES+=("$pkg_name@$pkg_version")
  else
    echo -e "${RED}✗ 发布失败: $pkg_name@$pkg_version${NC}"
    echo "$output" | tail -5
    ((FAILED_COUNT++))
    FAILED_PACKAGES+=("$pkg_name@$pkg_version")
  fi
  
  echo ""
  cd "$ROOT_DIR"
done

# 输出统计信息
echo "=========================================="
echo "发布完成"
echo "=========================================="
echo -e "${GREEN}成功: $SUCCESS_COUNT${NC}"
echo -e "${YELLOW}跳过: $SKIPPED_COUNT${NC}"
echo -e "${RED}失败: $FAILED_COUNT${NC}"

if [ ${#SKIPPED_PACKAGES[@]} -gt 0 ]; then
  echo ""
  echo "跳过的包（版本已存在）:"
  for pkg in "${SKIPPED_PACKAGES[@]}"; do
    echo -e "  ${YELLOW}- $pkg${NC}"
  done
fi

if [ $FAILED_COUNT -gt 0 ]; then
  echo ""
  echo "失败的包:"
  for pkg in "${FAILED_PACKAGES[@]}"; do
    echo -e "  ${RED}- $pkg${NC}"
  done
  exit 1
fi

if [ $SUCCESS_COUNT -gt 0 ]; then
  echo ""
  echo -e "${GREEN}所有包发布成功！${NC}"
fi

