import { useSyncEffect } from "@tingwujs/common";
import {
  Controller,
  TranscriptingState,
  TranslateMode,
  PID,
  Agenda,
  Selection,
  TextPolishMode,
} from "@tingwujs/core";
import React, { useEffect } from "react";
import { atom, selector, useRecoilState, useSetRecoilState } from "recoil";

const prefix = "tingwu-transcription";

export const isLivingModeState = atom({
  key: `${prefix}/isLivingModeState`,
  default: false,
});

export const isReadonlyState = atom({
  key: `${prefix}/isReadonlyState`,
  default: false,
});

//
// double state not a mistake
export const transcriptingStateState = atom({
  key: `${prefix}/transcriptingState`,
  default: TranscriptingState.Disabled,
});

export const unfixSentenceState = atom({
  key: `${prefix}/unfixSentence`,
  default: "",
});

export const isHasUnfixParagraphState = atom({
  key: `${prefix}/isHasUnfixParagraphState`,
  default: false,
});

export const translateModeState = atom<TranslateMode>({
  key: `${prefix}/translateModeState`,
  default: TranslateMode.Disabled,
});
// AI 改写模式 状态管理
export const textPolishModeState = atom<TextPolishMode>({
  key: `${prefix}/textPolishModeState`,
  default: TextPolishMode.Disabled,
});

export const unfixPidState = atom<PID | undefined>({
  key: `${prefix}/unfixPidState`,
  default: undefined,
});

export const isUnfixEmptyContentState = atom<boolean>({
  key: `${prefix}/isUnfixEmptyContentState`,
  default: true,
});

// 是否转写中和将要转写结束
export const isRoughlyTranscripingState = selector({
  key: `${prefix}/roughlyTranscripingState`,
  get: ({ get }) => {
    const transcriptingState = get(transcriptingStateState);
    return (
      transcriptingState === TranscriptingState.Enabled ||
      transcriptingState === TranscriptingState.Disabing
    );
  },
});

export const isRenderUnfixParagraphState = selector({
  key: `${prefix}/isRenderUnfixParagraphState`,
  get: ({ get }) => {
    const transcriptingState = get(transcriptingStateState);
    const isUnfixEmptyContent = get(isUnfixEmptyContentState);

    return (
      transcriptingState === TranscriptingState.Enabled ||
      (transcriptingState === TranscriptingState.Disabing && !isUnfixEmptyContent)
    );
  },
});

export const agendasState = atom<Agenda[]>({
  key: `${prefix}/agendasState`,
  default: [],
});

export const selectionState = atom<Selection | undefined>({
  key: `${prefix}/selectionState`,
  default: undefined,
});

export interface CaretStyleState {
  pid: PID;
  style: React.CSSProperties;
  editorRect: DOMRect;
}

export const caretStyleState = atom<CaretStyleState | undefined>({
  key: `${prefix}/caretStyleState`,
  default: undefined,
});

export const useConnectStateWithController = (controller: Controller) => {
  const [isLivingMode, setIsLivingMode] = useRecoilState(isLivingModeState);
  const [transcriptingState, setTranscriptingState] = useRecoilState(transcriptingStateState);
  const [unfixSentence, setUnfixSentence] = useRecoilState(unfixSentenceState);
  const setIsHasUnfixParagraph = useSetRecoilState(isHasUnfixParagraphState);
  const setTranslateMode = useSetRecoilState(translateModeState);
  const setTextPolishMode = useSetRecoilState(textPolishModeState); // AI改写模式
  const setIsReadonly = useSetRecoilState(isReadonlyState);
  const setAgendas = useSetRecoilState(agendasState);
  const setSelection = useSetRecoilState(selectionState);
  const setIsUnfixEmptyContentState = useSetRecoilState(isUnfixEmptyContentState);
  const setCaretStyle = useSetRecoilState(caretStyleState);
  const [unfixPid, setUnfixPid] = useRecoilState(unfixPidState);

  useSyncEffect(() => {
    setIsLivingMode(controller.getLivingMode());
    return controller.on("livingModeChange", ({ livingMode }) => {
      setIsLivingMode(livingMode);
    });
  }, []);

  useSyncEffect(() => {
    setTranscriptingState(controller.getTranscriptingState());
    return controller.on("transcriptingStateChange", ({ transcriptingState: tState }) => {
      setTranscriptingState(tState);
    });
  }, [isLivingMode]);

  useSyncEffect(() => {
    setUnfixSentence(controller.getUnfixContent());
    return controller.on("unfixContentChange", ({ unfixContent }) => {
      setUnfixSentence(unfixContent);
    });
  }, []);

  useSyncEffect(() => {
    return controller.on("paragraphChangeSpecific", () => {
      const localUnfixPid = controller.getUnfixPid();
      setUnfixPid(localUnfixPid);
      const isHasUnfixParagraph = localUnfixPid !== null;
      setIsHasUnfixParagraph(isHasUnfixParagraph);
    });
  }, [isLivingMode, transcriptingState]);

  useEffect(() => {
    const fixSentence = unfixPid ? controller.getTextContent(unfixPid) : "";
    setIsUnfixEmptyContentState(!(fixSentence || unfixSentence));
  }, [controller, unfixPid, unfixSentence, setIsUnfixEmptyContentState]);

  useSyncEffect(() => {
    setTranslateMode(controller.getTranslateMode());
    return controller.on("translateModeChange", ({ translateMode }) => {
      setTranslateMode(translateMode);
    });
  }, []);

  // AI 改写模式变更监听
  useSyncEffect(() => {
    controller.getTextPolishMode();
    return controller.on("textPolishModeChange", ({ textPolishMode }) => {
      setTextPolishMode(textPolishMode);
    });
  }, []);

  useSyncEffect(() => {
    setIsReadonly(controller.getReadonly());
    return controller.on("readonlyChange", ({ readonly }) => {
      setIsReadonly(readonly);
    });
  }, []);

  useSyncEffect(() => {
    setAgendas(controller.getAgendas());
    return controller.on("agendasChange", ({ agendas }) => {
      setAgendas(agendas);
    });
  }, []);

  useSyncEffect(() => {
    setSelection(controller.getCurrentSelection());
    return controller.on("selectionChange", ({ selection }) => {
      setSelection(selection);
    });
  }, []);

  useSyncEffect(() => {
    setSelection(controller.getCurrentSelection());
    return controller.on("caretStyleChange", ({ value }) => {
      setCaretStyle(value);
    });
  }, []);
};
