import { AgendaSummaryController, AgendaSummaryCard } from "@tingwujs/sphere";
import React from "react";
import { calcCustomHeight } from "../../utils";
import { BaseExtension, GetParagraphTopHeightParams } from "../baseExtension";

export class AgendaSummaryExtension extends BaseExtension {
  agendaSummaryController: AgendaSummaryController;

  constructor(agendaSummaryController: AgendaSummaryController) {
    super();
    this.agendaSummaryController = agendaSummaryController;
  }

  providerParagraphsNeedRefresh(needRefresh: () => void) {
    return this.agendaSummaryController.onStateChange(() => {
      needRefresh();
    });
  }

  isHasParagraphTop(params: GetParagraphTopHeightParams) {
    const { pid } = params;
    const pidSummariesMap = this.agendaSummaryController.getState("pidSummariesMap");
    if (pidSummariesMap[pid] != null) {
      return true;
    }

    return false;
  }

  calcTextHeight(width: number, nodes: Array<Text | HTMLElement>) {
    const result = calcCustomHeight()
      .getElement(element => {
        element.style.fontSize = "14px";
        element.style.lineHeight = "24px";
        element.style.textAlign = "justify";
        element.style.overflowWrap = "anywhere";
        element.style.whiteSpace = "pre-wrap";
      })
      .andWidth(width)
      .andNodes(nodes)
      .getHeight();
    return result;
  }

  calcDescTextHeight(width: number, nodes: Array<Text | HTMLElement>) {
    const result = calcCustomHeight()
      .getElement(element => {
        element.style.fontSize = "12px";
        element.style.lineHeight = "20px";
        element.style.textAlign = "justify";
        element.style.overflowWrap = "anywhere";
        element.style.whiteSpace = "pre-wrap";
      })
      .andWidth(width)
      .andNodes(nodes)
      .getHeight();
    return result;
  }

  getParagraphTopHeight(params: GetParagraphTopHeightParams) {
    const { width, pid } = params;
    const summaries = this.agendaSummaryController.getSummariesByPid(pid);

    if (summaries.length === 0) {
      return 0;
    }

    const displayTimeWidth = this.agendaSummaryController.getDisplayTimeWidth();
    const otherWith = displayTimeWidth + 63; // 63 = 7 + 16 + 24 + 16
    const textWidth = width - otherWith;
    let height = 0;

    summaries.forEach(summary => {
      height += this.calcTextHeight(textWidth, [document.createTextNode(summary.agendaHeading)]);

      height += this.calcDescTextHeight(textWidth, [document.createTextNode(summary.content)]);

      height += 4; // 4 for marginBottom of `AgendaHeading`
      height += 30; // 30 for topbar height
      height += 24; // 24 for v-padding
    });

    height += 20; // tailing padding
    return height;
  }

  renderParagraphTop(params: GetParagraphTopHeightParams) {
    const { pid } = params;
    const summaries = this.agendaSummaryController.getSummariesByPid(pid);
    const readonly = this.agendaSummaryController.getReadonly();
    if (summaries.length === 0) {
      return null;
    }

    const isThreeUnitTime = this.agendaSummaryController.getState("isThreeUnitTime");
    const displayTimeWidth = this.agendaSummaryController.getDisplayTimeWidth();
    return summaries.map(summary => {
      return (
        <AgendaSummaryCard
          readonly={readonly}
          summary={summary}
          isThreeUnitTime={isThreeUnitTime}
          displayTimeWidth={displayTimeWidth}
          firstSummary={this.agendaSummaryController.isFirstSummary(summary.id)}
          lastSummary={this.agendaSummaryController.isLastSummary(summary.id)}
          star={summary.star}
          onRate={star => {
            this.agendaSummaryController.rateSummary(summary, star);
          }}
        />
      );
    });
  }
}
