/**
 *  *************************************************************
 *  ****************  大模型数据 Hooks   *************************
 *  *************************************************************
 */
import { useCallback, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
// *** action 相关 ***
import * as BigModelActions from './docBigModel.operator/docBigModel.actions';
// *** hooks 相关 ***
import { useDocumentData } from '../document.hooks';
// *** 类型定义相关 ***
import type { iDocDataUpdateBasePayload } from '../document.baseType';
import type { iDocumentState } from '../document.type';
import {
  eLabInfoStatus,
  type iDocBigModelData,
  type iDocBigModelDataMap,
  type iSpeakSummary,
} from './docBigModel.type';

/**
 *  @description 大模型数据 Hooks
 *  @returns
 */
const useDocBigModelData = () => {
  const dispatch = useDispatch();
  const { documentData } = useDocumentData();
  /**
   *  *************************************************************
   *  **********************  Model data   *********************
   *  *************************************************************
   */

  /** 大模型数据集合 */
  const docBigModelDataMap: iDocBigModelDataMap = useSelector(
    (state: { documentModel: iDocumentState }) =>
      state.documentModel.docBigModelDataMap,
  );

  /** 当前文档的大模型数据 */
  const currentDocBigModelData = useMemo((): iDocBigModelData | undefined => {
    if (!documentData?.docId) return;
    return docBigModelDataMap[documentData?.docId];
  }, [docBigModelDataMap, documentData?.docId]);

  /**
   *  *************************************************************
   *  **********  派生状态 非model 层数据 方便具体业务出使用   **********
   *  *************************************************************
   */
  const DocBigModelDataStatusList = useMemo(() => {
    if (!currentDocBigModelData) return [eLabInfoStatus.LOADING];
    return [
      currentDocBigModelData.keyWordInfoStatus,
      currentDocBigModelData.fullTextSummaryStatus,
      currentDocBigModelData.chapterOverviewStatus,
      currentDocBigModelData.speakSummaryStatus,
      currentDocBigModelData.keyPointsReviewStatus,
      currentDocBigModelData.todoListStatus,
      currentDocBigModelData.bigModelSceneInfoStatus,
      currentDocBigModelData.bigModelXmindInfoStatus,
    ];
  }, [currentDocBigModelData]);

  const DocBigModelDataList = useMemo(() => {
    if (!currentDocBigModelData) return [];
    return [
      currentDocBigModelData.keyWordInfo, // 关键词信息
      currentDocBigModelData.fullTextSummary?.value, // 全文概要
      currentDocBigModelData.chapterOverview, // 章节速览
      currentDocBigModelData.speakSummary, // 发言总结
      currentDocBigModelData.keyPointsReview, // 要点、问答回顾
      // currentDocBigModelData.pptTitleInfos, // PPT标题信息
      currentDocBigModelData.pptSummaryInfos?.contents, // PPT概要信息
      currentDocBigModelData.todoList, // 待办事项
    ];
  }, [currentDocBigModelData]);

  /**  大模型数据(除了PPT)是否全部加载完成 */
  const isDocBigModelDataAllFinish = useMemo(
    () => !DocBigModelDataStatusList.includes(eLabInfoStatus.LOADING),
    [DocBigModelDataStatusList],
  );

  /**  大模型数据是否含有某一项加载完成 */
  const isDocBigModelDataHasValue = useMemo(() => {
    const find = DocBigModelDataList.find((item) => item && item.length);
    return find !== undefined;
  }, [DocBigModelDataList]);

  /**
   *  *************************************************************
   *  **********************  ActionHandles   *********************
   *  *************************************************************
   */

  /** 更新指定文档🆔的大模型数据 */
  const updateDocBigModelData = useCallback(
    async (params: iDocDataUpdateBasePayload<Partial<iDocBigModelData>>) => {
      return await dispatch<any>(
        BigModelActions.updateDocBigModelDataAction(params, true),
      );
    },
    [dispatch],
  );

  return {
    /** 大模型数据集合 */
    docBigModelDataMap,
    /** 更新指定文档🆔的大模型数据 */
    currentDocBigModelData,
    /** 更新指定文档🆔的大模型数据 */
    updateDocBigModelData,
    /**
     *  *************************************************************
     *  **********  派生状态 非model 层数据 方便具体业务处使用   **********
     *  *************************************************************
     */
    isDocBigModelDataAllFinish,
    isDocBigModelDataHasValue,
  };
};

export { useDocBigModelData };
