import { myLocalStorage, storageKeys } from '../localStorage';

describe('myLocalStorage', () => {
  beforeEach(() => {
    // Clear localStorage before each test
    localStorage.clear();
  });

  describe('getItem and setItem', () => {
    it('should get and set item with JSON parsing', () => {
      const value = { test: 'value', number: 42 };
      myLocalStorage.setItem('detailRightContentWidth', value);
      const retrieved = myLocalStorage.getItem('detailRightContentWidth');

      expect(retrieved).toEqual(value);
    });

    it('should get and set item without JSON parsing', () => {
      const value = 'plain string';
      myLocalStorage.setItem('detailRightContentWidth', value, false);
      const retrieved = myLocalStorage.getItem('detailRightContentWidth', false);

      expect(retrieved).toBe(value);
    });

    it('should return null for non-existent item', () => {
      const retrieved = myLocalStorage.getItem('detailRightContentWidth');
      expect(retrieved).toBeNull();
    });
  });

  describe('removeItem', () => {
    it('should remove item from localStorage', () => {
      myLocalStorage.setItem('detailRightContentWidth', { test: 'value' });
      myLocalStorage.removeItem('detailRightContentWidth');
      const retrieved = myLocalStorage.getItem('detailRightContentWidth');

      expect(retrieved).toBeNull();
    });
  });

  describe('isHasId', () => {
    it('should return true if id exists in comma-separated list', () => {
      myLocalStorage.setItem('showedLiveSpeakerTip', 'id1,id2,id3', false);
      expect(myLocalStorage.isHasId('showedLiveSpeakerTip', 'id2')).toBe(true);
    });

    it('should return false if id does not exist', () => {
      myLocalStorage.setItem('showedLiveSpeakerTip', 'id1,id2,id3', false);
      expect(myLocalStorage.isHasId('showedLiveSpeakerTip', 'id4')).toBe(false);
    });

    it('should return false for empty storage', () => {
      expect(myLocalStorage.isHasId('showedLiveSpeakerTip', 'id1')).toBe(false);
    });
  });

  describe('setItemById', () => {
    it('should add id to comma-separated list', () => {
      myLocalStorage.setItemById('showedLiveSpeakerTip', 'id1');
      expect(myLocalStorage.isHasId('showedLiveSpeakerTip', 'id1')).toBe(true);
    });

    it('should not add duplicate id', () => {
      myLocalStorage.setItemById('showedLiveSpeakerTip', 'id1');
      myLocalStorage.setItemById('showedLiveSpeakerTip', 'id1');
      const value = myLocalStorage.getItem('showedLiveSpeakerTip', false);
      const ids = value?.split(',') || [];
      expect(ids.filter((id: string) => id === 'id1').length).toBe(1);
    });

    it('should limit to MAX_ITEMS (5000)', () => {
      // Add more than MAX_ITEMS
      for (let i = 0; i < 5002; i++) {
        myLocalStorage.setItemById('showedLiveSpeakerTip', `id${i}`);
      }
      const value = myLocalStorage.getItem('showedLiveSpeakerTip', false);
      const ids = value?.split(',') || [];
      expect(ids.length).toBeLessThanOrEqual(5000);
    });
  });
});

