//
//  StreamInputTtsViewController.m
//  NUIdemo
//
//  Created by lengjiayi on 2024/4/17.
//  Copyright © 2024 Alibaba idst. All rights reserved.
//

#import "nuisdk.framework/Headers/StreamInputTts.h"
#import "StreamInputTtsViewController.h"
#import "HWOptionButton.h"
#import "NuiSdkUtils.h"

#import "audio/AudioController.h"
#import <AVFoundation/AVFoundation.h>

// 本样例展示在线语音合成（流式输入流式输出）使用方法
// iOS SDK  说明：https://help.aliyun.com/zh/isi/developer-reference/streaming-text-to-speech-synthesis-android-sdk
@interface StreamInputTtsViewController ()<ConvVoiceRecorderDelegate, UITextFieldDelegate, HWOptionButtonDelegate, StreamInputTtsDelegate> {
    IBOutlet UIButton *StartButton;
    IBOutlet UIButton *PlayButton;
    IBOutlet UIButton *StopButton;
    IBOutlet UIButton *CancelButton;
    
    IBOutlet UITextView *textViewContent;
    IBOutlet UITextView *textViewEvent;
}

@property(nonatomic,strong) StreamInputTts* streamInputTtsSdk;
@property(nonatomic, weak) HWOptionButton *fontName;
@property(nonatomic, weak) HWOptionButton *formatType;
@property(nonatomic, weak) HWOptionButton *pitch;
@property(nonatomic, weak) HWOptionButton *rate;
@property(nonatomic, weak) HWOptionButton *volume;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NuiSdkUtils *utils;
@property(nonatomic) int ttsSampleRate;

@property (nonatomic, assign) BOOL isPlaying;
@property (nonatomic, assign) BOOL shouldStopSending; // 控制发送线程退出
@property (nonatomic, strong) NSString *currentSessionId; // 当前会话ID，用于匹配
@end

@implementation StreamInputTtsViewController

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static dispatch_queue_t tts_work_queue;

#pragma mark view controller methods

- (void)viewDidLoad {
    [super viewDidLoad];
    TLog(@"StreamInputTtsViewController did load");

    _streamInputTtsSdk = [StreamInputTts get_instance];
    _streamInputTtsSdk.delegate = self;
    
    [self InitView];

    _utils = [NuiSdkUtils alloc];

    tts_work_queue = dispatch_queue_create("NuiStreamTtsController", DISPATCH_QUEUE_CONCURRENT);
}

-(void)dealloc {
    NSLog(@"%s", __FUNCTION__);
    if (_audioController != nil) {
        [_audioController cleanPlayerBuffer];
    }
    if (_streamInputTtsSdk != nil) {
        [_streamInputTtsSdk releaseStreamInputTts];
        _streamInputTtsSdk.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
}

- (void)dismissKeyboard:(id)sender {
    [self.view endEditing:YES];
}

-(void)viewDidAppear:(BOOL)animated {
    TLog(@"StreamInputTtsViewController-viewDidAppear");
    [super viewDidAppear:animated];
    [self InitView];
}

-(void)viewWillDisappear:(BOOL)animated {
    NSLog(@"%s", __FUNCTION__);
    [super viewWillDisappear:animated];

    // 触发停止
    self.shouldStopSending = YES;
    self.currentSessionId = @"";

    if (_audioController != nil) {
        [_audioController cleanPlayerBuffer];
    }
    if (_streamInputTtsSdk != nil) {
        [_streamInputTtsSdk releaseStreamInputTts];
        _streamInputTtsSdk.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - Audio Player Delegate
-(void)playerDidFinish {
    //播放被中止后回调。
    TLog(@"playerDidFinish");
    _isPlaying = NO;
}
-(void)playerDrainDataFinish {
    //播放数据自然播放完成后回调。
    TLog(@"playerDrainDataFinish");
    _isPlaying = NO;
}
#pragma mark -private methods

-(NSString *)genTickets {
    NSString *debug_path = [_utils createDir];
    TLog(@"debug_path:%@", debug_path);

    NSMutableDictionary *ticketJsonDict = [NSMutableDictionary dictionary];

    //获取账号访问凭证：
    [_utils getTicket:ticketJsonDict Type:get_token_from_server_for_online_features];
    if ([ticketJsonDict objectForKey:@"token"] != nil) {
        NSString *tokenValue = [ticketJsonDict objectForKey:@"token"];
        if ([tokenValue length] == 0) {
            TLog(@"The 'token' key exists but the value is empty.");
        }
    } else {
        TLog(@"The 'token' key does not exist.");
    }

    [ticketJsonDict setObject:@"wss://nls-gateway-cn-beijing.aliyuncs.com/ws/v1" forKey:@"url"]; // 必填

    [ticketJsonDict setObject:@"empty_device_id" forKey:@"device_id"]; // 必填, 推荐填入具有唯一性的id, 方便定位问题

    //debug目录，当初始化SDK时的save_log参数取值为true时，该目录用于保存中间音频文件。
    [ticketJsonDict setObject:debug_path forKey:@"debug_path"];
    //过滤SDK内部日志通过回调送回到用户层
    [ticketJsonDict setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_NONE] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [ticketJsonDict setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    [ticketJsonDict setObject:@"10000" forKey:@"complete_waiting_ms"];

    NSData *data = [NSJSONSerialization dataWithJSONObject:ticketJsonDict options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString *)genParameters:(BOOL)ssml {
    NSMutableString *jsonString = [NSMutableString string];
    
    // 接口说明：https://help.aliyun.com/zh/isi/developer-reference/interface-description
    @try {
        NSMutableDictionary *object = [[NSMutableDictionary alloc] init];
        
        NSArray<NSString *> *font_components = [_fontName.title componentsSeparatedByString:@"-"];
        NSString *firstPart = font_components.firstObject;
        [object setObject:firstPart forKey:@"voice"];

        /**
         * 音频编码格式。
         * 所有模型均支持的编码格式：pcm、wav和mp3（默认）
         * */
        [object setObject:_formatType.title forKey:@"format"];
        BOOL enableDecoder = [_formatType.title isEqualToString:@"mp3"] || [_formatType.title isEqualToString:@"opus"];
        [object setObject:@(enableDecoder) forKey:@"enable_audio_decoder"];
        /**
         * 音频采样率，支持下述采样率（单位：Hz）：
         * 8000, 16000,  24000 , 48000。
         * */
        [object setObject:@(16000) forKey:@"sample_rate"];
        /**
         * 音量，取值范围：0～100。默认值：50。
         * */
        NSInteger volume = [_volume.title integerValue];
        [object setObject:@(volume) forKey:@"volume"];
        /**
         * 朗读语速，范围是-500~500，默认是0。
         * */
        float rate = [_rate.title floatValue];
        [object setObject:@(rate) forKey:@"speech_rate"];
        /**
         * 朗读语调，范围是-500~500，默认是0。
         * */
        float pitch = [_pitch.title floatValue];
        [object setObject:@(pitch) forKey:@"pitch_rate"];

        if (ssml) {
            /**
             * 是否开启SSML功能。
             * 该参数设为 true 后，仅允许发送一次文本，支持纯文本或包含SSML的文本。
             * */
            [object setObject:@(YES) forKey:@"enable_ssml"];
        }

        NSError *jsonError;
        NSData *finalData = [NSJSONSerialization dataWithJSONObject:object options:NSJSONWritingPrettyPrinted error:&jsonError];
        if (finalData) {
            jsonString = [NSMutableString stringWithString:[[NSString alloc] initWithData:finalData encoding:NSUTF8StringEncoding]];
        } else {
            NSLog(@"生成 JSON 失败: %@", jsonError.localizedDescription);
        }

    } @catch (NSException *exception) {
        NSLog(@"异常: %@", exception);
    }

    NSLog(@"user parameters: %@", jsonString);

    return [jsonString copy];
}

-(CGFloat)InitTextView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    // ---- textViewContent ---
    CGFloat textViewContent_width = global_size.width - border * 2;
    CGFloat textViewContent_height = global_size.height/SCREEN_HEIGHT_BASE * 200;
    CGFloat textViewContent_x = border;
    CGFloat textViewContent_y = startY;

    CGRect textViewContent_rect = CGRectMake(textViewContent_x, textViewContent_y, textViewContent_width, textViewContent_height);
    if (!textViewContent) {
        textViewContent = [[UITextView alloc] initWithFrame:textViewContent_rect];
    }
    textViewContent.layer.borderWidth = 0.6;
    textViewContent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewContent.layer.cornerRadius = 10;
    [textViewContent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewContent.scrollEnabled = YES;
    textViewContent.text = @"唧唧复唧唧，木兰当户织。不闻机杼声，唯闻女叹息。\n问女何所思，问女何所忆。女亦无所思，女亦无所忆。\n昨夜见军帖，可汗大点兵，军书十二卷，卷卷有爷名。\n阿爷无大儿，木兰无长兄，愿为市鞍马，从此替爷征。\n东市买骏马，西市买鞍鞯，南市买辔头，北市买长鞭。旦辞爷娘去，暮宿黄河边，不闻爷娘唤女声，但闻黄河流水鸣溅溅。\n旦辞黄河去，暮至黑山头，不闻爷娘唤女声，但闻燕山胡骑鸣啾啾。万里赴戎机，关山度若飞。朔气传金柝，寒光照铁衣。将军百战死，壮士十年归。\n归来见天子，天子坐明堂。策勋十二转，赏赐百千强。可汗问所欲，木兰不用尚书郎，愿驰千里足，送儿还故乡。爷娘闻女来，出郭相扶将；阿姊闻妹来，当户理红妆；小弟闻姊来，磨刀霍霍向猪羊。\n开我东阁门，坐我西阁床。脱我战时袍，著我旧时裳。当窗理云鬓，对镜帖花黄。出门看火伴，火伴皆惊忙：同行十二年，不知木兰是女郎。\n雄兔脚扑朔，雌兔眼迷离；双兔傍地走，安能辨我是雄雌？";
    textViewContent.textColor = [UIColor darkGrayColor];
    textViewContent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewContent];
    
    // ---- textViewEvent ---
    CGFloat textViewEvent_width = textViewContent_width;
    CGFloat textViewEvent_height = global_size.height/SCREEN_HEIGHT_BASE * 100;
    CGFloat textViewEvent_x = border;
    CGFloat textViewEvent_y = startY + textViewContent_height + border;

    CGRect textViewEvent_rect = CGRectMake(textViewEvent_x, textViewEvent_y, textViewEvent_width, textViewEvent_height);
    if (!textViewEvent) {
        textViewEvent = [[UITextView alloc] initWithFrame:textViewEvent_rect];
    }
    textViewEvent.layer.borderWidth = 0.6;
    textViewEvent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewEvent.layer.cornerRadius = 10;
    [textViewEvent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewEvent.scrollEnabled = YES;
    textViewEvent.text = @"事件展示";
    textViewEvent.textColor = [UIColor darkGrayColor];
    textViewEvent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewEvent];
    
    return textViewEvent_y + textViewEvent_height;
}

-(CGFloat)InitButtonView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    // ---- StartButton ---
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat new_border = (global_size.width - button_width * 4) / 5;
    CGFloat start_x = new_border;
    CGFloat start_y = startY;
    
    StartButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StartButton.frame = CGRectMake(start_x, start_y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [StartButton setBackgroundImage:image forState:UIControlStateNormal];
    [StartButton setTitle:@"流式合成" forState:UIControlStateNormal];
    [StartButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StartButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StartButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StartButton];
    
    // ---- PlayButton ---
    CGFloat PlayButton_x = start_x + button_width + new_border;
    CGFloat PlayButton_y = startY;
    
    PlayButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PlayButton.frame = CGRectMake(PlayButton_x, PlayButton_y, button_width, button_height);
    UIImage *Play_image = [UIImage imageNamed:@"button_start"];
    [PlayButton setBackgroundImage:Play_image forState:UIControlStateNormal];
    [PlayButton setTitle:@"长文本" forState:UIControlStateNormal];
    [PlayButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PlayButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PlayButton addTarget:self action:@selector(playTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PlayButton];

    // ---- StopButton ---
    CGFloat StopButton_x = PlayButton_x + button_width + new_border;
    CGFloat StopButton_y = startY;

    StopButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StopButton.frame = CGRectMake(StopButton_x, StopButton_y, button_width, button_height);
    UIImage *StopButton_image = [UIImage imageNamed:@"button_start"];
    [StopButton setBackgroundImage:StopButton_image forState:UIControlStateNormal];
    [StopButton setTitle:@"停止" forState:UIControlStateNormal];
    [StopButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StopButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StopButton addTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StopButton];

    // ---- CancelButton ---
    CGFloat CancelButton_x = StopButton_x + button_width + new_border;
    CGFloat CancelButton_y = startY;

    CancelButton = [UIButton buttonWithType:UIButtonTypeCustom];
    CancelButton.frame = CGRectMake(CancelButton_x, CancelButton_y, button_width, button_height);
    UIImage *CancelButton_image = [UIImage imageNamed:@"button_start"];
    [CancelButton setBackgroundImage:CancelButton_image forState:UIControlStateNormal];
    [CancelButton setTitle:@"取消" forState:UIControlStateNormal];
    [CancelButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    CancelButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [CancelButton addTarget:self action:@selector(cancelTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:CancelButton];

    return CancelButton_y + button_height;
}

-(CGFloat)InitHWOptionsView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    CGFloat hw_height = global_size.height / SCREEN_HEIGHT_BASE * 40;

    // ---- modelName ---
    CGFloat modelName_y = startY;

    // ---- fontName ---
    CGFloat fontName_width = global_size.width - border * 2;
    CGFloat fontName_x = border;
    CGFloat fontName_y = modelName_y + hw_height;

    HWOptionButton *fontNameBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(fontName_x, fontName_y, fontName_width, hw_height)];
    NSArray<NSString *> *voice_list = [_utils getVoiceList:@"StreamInputTts"];
//    NSLog(@"voice list: %@", voice_list);
    fontNameBtn.array = voice_list;
    fontNameBtn.selectIndex = 0;
    fontNameBtn.backgroundColor = [UIColor whiteColor];
    fontNameBtn.delegate = self;
    fontNameBtn.showPlaceholder = NO;
    fontNameBtn.showSearchBar = NO;
    fontNameBtn.dropdownTitle = @" - 说话人-请于官网查看更多说话人 - ";
    [self.view addSubview:fontNameBtn];
    self.fontName = fontNameBtn;

    // ---- formatType ---
    CGFloat formatType_width = global_size.width/2;
    CGFloat formatType_x = border;
    CGFloat formatType_y = fontName_y + hw_height;

    HWOptionButton *formatTypeBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(formatType_x, formatType_y, formatType_width, hw_height)];
    formatTypeBtn.array = @[@"mp3", @"pcm", @"wav"];
    formatTypeBtn.selectIndex = 0;
    formatTypeBtn.backgroundColor = [UIColor whiteColor];
    formatTypeBtn.delegate = self;
    formatTypeBtn.showPlaceholder = NO;
    formatTypeBtn.showSearchBar = NO;
    formatTypeBtn.dropdownTitle = @" - 音频编码格式 - ";
    [self.view addSubview:formatTypeBtn];
    self.formatType = formatTypeBtn;

    // ---- pitch ---
    CGFloat pitch_width = global_size.width / 3 - border * 2;
    CGFloat pitch_x = border;
    CGFloat pitch_y = formatType_y + hw_height;

    HWOptionButton *pitchBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(pitch_x, pitch_y, pitch_width, hw_height)];
    pitchBtn.backgroundColor = [UIColor whiteColor];
    pitchBtn.array = @[@"0", @"-500", @"150", @"300", @"500"];
    pitchBtn.selectIndex = 0;
    pitchBtn.delegate = self;
    pitchBtn.showSearchBar = NO;
    pitchBtn.showPlaceholder = NO;
    pitchBtn.dropdownTitle = @" - 语调 - ";
    [self.view addSubview:pitchBtn];
    self.pitch = pitchBtn;
    
    // ---- rate ---
    CGFloat rate_width = global_size.width / 3 - border * 2;
    CGFloat rate_x = pitch_x + pitch_width + border;
    CGFloat rate_y = pitch_y;

    HWOptionButton *rateBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(rate_x, rate_y, rate_width, hw_height)];
    rateBtn.backgroundColor = [UIColor whiteColor];
    rateBtn.array = @[@"1.0", @"0.5", @"0.8", @"1.5", @"2.0"];
    rateBtn.array = @[@"0", @"-500", @"150", @"300", @"500"];
    rateBtn.selectIndex = 0;
    rateBtn.delegate = self;
    rateBtn.showSearchBar = NO;
    rateBtn.showPlaceholder = NO;
    rateBtn.dropdownTitle = @" - 语速 - ";
    [self.view addSubview:rateBtn];
    self.rate = rateBtn;
    
    // ---- volume ---
    CGFloat volume_width = global_size.width / 3 - border * 2;
    CGFloat volume_x = rate_x + rate_width + border;
    CGFloat volume_y = pitch_y;

    HWOptionButton *volumeBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(volume_x, volume_y, volume_width, hw_height)];
    volumeBtn.backgroundColor = [UIColor whiteColor];
    volumeBtn.array = @[@"50", @"10", @"30", @"70", @"80", @"100"];
    volumeBtn.selectIndex = 0;
    volumeBtn.delegate = self;
    volumeBtn.showSearchBar = NO;
    volumeBtn.showPlaceholder = NO;
    volumeBtn.dropdownTitle = @" - 音量 - ";
    [self.view addSubview:volumeBtn];
    self.volume = volumeBtn;
    
    return volume_y + hw_height;
}

-(void)InitView {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"流式语音合成";
    CGFloat border = global_size.width/SCREEN_WIDTH_BASE * 13;
    CGFloat division = global_size.height/SCREEN_HEIGHT_BASE * 10;
    CGFloat end_y = 0;

    CGFloat button_start_y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    end_y = [self InitButtonView:border startY:button_start_y];

    CGFloat hwoptions_start_y = end_y + division;
    end_y = [self InitHWOptionsView:border startY:hwoptions_start_y];
    
    CGFloat view_start_y = end_y + division;
    end_y = [self InitTextView:border startY:view_start_y];
}

- (void)sendTextInChunksWithSessionId:(NSString *)sessionId {
    @autoreleasepool {
        // 获取要发送的文本（在主线程捕获，避免跨线程访问 UI）
        __block NSString *textToSend = nil;
        dispatch_sync(dispatch_get_main_queue(), ^{
            textToSend = [self->textViewContent.text copy];
        });

        if (textToSend.length == 0) {
            TLog(@"No text to send.");
            return;
        }

        // 按句子切分（简单按句号、问号、感叹号分割）
        NSCharacterSet *delimiters = [NSCharacterSet characterSetWithCharactersInString:@".。!！?？\n"];
        NSArray<NSString *> *sentences = [textToSend componentsSeparatedByCharactersInSet:delimiters];
        
        // 过滤空句子
        NSMutableArray<NSString *> *validSentences = [NSMutableArray array];
        for (NSString *originalSentence in sentences) {
            NSString *trimmedSentence = [originalSentence stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceAndNewlineCharacterSet]];
            if (trimmedSentence.length > 0) {
                [validSentences addObject:trimmedSentence];
            }
        }

        TLog(@"Total sentences to send: %lu", (unsigned long)validSentences.count);

        // 逐句发送
        for (NSUInteger i = 0; i < validSentences.count; i++) {
            // 检查是否被要求停止
            __block BOOL shouldStop = NO;
            dispatch_sync(dispatch_get_main_queue(), ^{
                shouldStop = (self.shouldStopSending || ![self.currentSessionId isEqualToString:sessionId]);
            });
            if (shouldStop) {
                TLog(@"Sending stopped by user or new session.");
                break;
            }

            NSString *sentence = validSentences[i];
            TLog(@"Sending sentence %lu: %@", (unsigned long)(i+1), sentence);

            // 调用 SDK 发送
            int ret = [_streamInputTtsSdk sendStreamInputTts:[sentence UTF8String]];
            if (ret != SUCCESS) {
                TLog(@"sendStreamInputTts failed (%d) for sentence: %@", ret, sentence);
                break;
            }

            // 等待 1 秒（但允许提前退出）
            if (i < validSentences.count - 1) { // 最后一句不等
                NSDate *startTime = [NSDate date];
                NSTimeInterval waitTime = 0.5;
                while ([[NSDate date] timeIntervalSinceDate:startTime] < waitTime) {
                    usleep(100000); // 休眠 0.1 秒
                    __block BOOL stopNow = NO;
                    dispatch_sync(dispatch_get_main_queue(), ^{
                        stopNow = (self.shouldStopSending || ![self.currentSessionId isEqualToString:sessionId]);
                    });
                    if (stopNow) {
                        TLog(@"Early exit from waiting.");
                        break;
                    }
                }
            }
        } // for

        // 通过sendStreamInputTts发送完所有文本后，必须调用asyncStopStreamInputTts或者stopStreamInputTts
        // 通知服务完成合成任务。否则会有超时报错。
        if (validSentences.count > 0) {
            [_streamInputTtsSdk asyncStopStreamInputTts];
        }

        // 清理会话
        dispatch_async(dispatch_get_main_queue(), ^{
            if ([self.currentSessionId isEqualToString:sessionId]) {
                self.currentSessionId = @"";
            }
        });
    }
}

#pragma mark - Button Action
- (IBAction)startTTS:(UIButton *)sender {
    if (_isPlaying) {
        return;
    }
    if (_streamInputTtsSdk != nil) {
        // 防止重复启动
        if (self.shouldStopSending == NO && self.currentSessionId.length > 0) {
            TLog(@"Already running a session.");
            return;
        }

        if (_audioController == nil) {
            // 注意：这里audioController模块仅用于播放示例，用户可根据业务场景自行实现这部分代码
            _audioController = [[AudioController alloc] init:only_player];
            _audioController.delegate = self;
            _ttsSampleRate = 16000;
            [_audioController setPlayerSampleRate:_ttsSampleRate];
        }

        NSString *tickets = [self genTickets];
        NSString *parameters = [self genParameters:NO];

        dispatch_async(tts_work_queue, ^{
            NSUUID *uuid = [NSUUID UUID];
            NSString *uuidString = [uuid UUIDString];
            int ret = [_streamInputTtsSdk startStreamInputTts:[tickets UTF8String]
                                                   parameters:[parameters UTF8String]
                                                    sessionId:[uuidString UTF8String]
                                                     logLevel:NUI_LOG_LEVEL_VERBOSE
                                                      saveLog:YES];
            if (ret != SUCCESS) {
                TLog(@"stream input tts start fail(%d)", ret);
            } else {
                TLog(@"stream input tts start success with id(%@)", uuidString);
                
                _isPlaying = YES;

                self.currentSessionId = uuidString;
                self.shouldStopSending = NO;
                // 启动发送线程（在同一个 queue，避免并发问题）
                dispatch_async(tts_work_queue, ^{
                    [self sendTextInChunksWithSessionId:uuidString];
                });
            }
        });
    }
}

- (IBAction)playTTS:(UIButton *)sender {
    if (_isPlaying) {
        return;
    }
    if (_streamInputTtsSdk != nil) {
        if (_audioController == nil) {
            // 注意：这里audioController模块仅用于播放示例，用户可根据业务场景自行实现这部分代码
            _audioController = [[AudioController alloc] init:only_player];
            _audioController.delegate = self;
            _ttsSampleRate = 16000;
            [_audioController setPlayerSampleRate:_ttsSampleRate];
        }

        NSString *tickets = [self genTickets];
        NSString *parameters = [self genParameters:YES];

        dispatch_async(tts_work_queue, ^{
            NSUUID *uuid = [NSUUID UUID];
            NSString *uuidString = [uuid UUIDString];
            int ret = [_streamInputTtsSdk asyncPlayStreamInputTts:[tickets UTF8String]
                                                       parameters:[parameters UTF8String]
                                                             text:[self->textViewContent.text UTF8String]
                                                        sessionId:[uuidString UTF8String]
                                                         logLevel:NUI_LOG_LEVEL_DEBUG
                                                          saveLog:YES];
            if (ret != SUCCESS) {
                TLog(@"stream input tts play fail(%d)", ret);
            } else {
                TLog(@"stream input tts play success with id(%@)", uuidString);
                
                _isPlaying = YES;
            }
        });
    }
}

- (IBAction)stopTTS:(UIButton *)sender {
    if (_streamInputTtsSdk != nil) {
        TLog(@"StreamInputTTSViewController stop tts");
        
        // 设置停止标志
        dispatch_async(dispatch_get_main_queue(), ^{
            self.shouldStopSending = YES;
        });
        
        dispatch_async(tts_work_queue, ^{
            [_streamInputTtsSdk asyncStopStreamInputTts];
        });
    } else {
        TLog(@"in stopTTS, _streamInputTtsSdk == nil.");
    }
}

- (IBAction)cancelTTS:(UIButton *)sender {
    if (_streamInputTtsSdk != nil) {
        TLog(@"StreamInputTTSViewController cancel tts");

        // 设置停止标志
        dispatch_async(dispatch_get_main_queue(), ^{
            self.shouldStopSending = YES;
        });
        
        dispatch_async(tts_work_queue, ^{
            [_streamInputTtsSdk cancelStreamInputTts];
            if (_audioController != nil) {
                [_audioController stopPlayer];
            }
        });
        
        _isPlaying = NO;
    } else {
        TLog(@"in cancelTTS, _streamInputTtsSdk == nil.");
    }
}

#pragma stream input tts callback
- (void)onStreamInputTtsEventCallback:(StreamInputTtsCallbackEvent)event
                               taskId:(char*)taskid sessionId:(char*)sessionId
                             ret_code:(int)ret_code error_msg:(char*)error_msg
                            timestamp:(char*)timestamp all_response:(char*)all_response {
//    NSString *log = [NSString stringWithFormat:@"\n事件回调（%d）：%s", event, all_response];
//    TLog(@"%@", log);

    if (event == TTS_EVENT_SYNTHESIS_STARTED) {
        dispatch_async(dispatch_get_main_queue(), ^{
            self->textViewEvent.text = @"TTS_EVENT_SYNTHESIS_STARTED";
        });
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SYNTHESIS_STARTED");
        if (_audioController != nil) {
            // 启动播放器
            [_audioController startPlayer];
        }
    } else if (event == TTS_EVENT_SENTENCE_BEGIN) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_BEGIN");
    } else if (event == TTS_EVENT_SENTENCE_SYNTHESIS) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_SYNTHESIS");
    } else if (event == TTS_EVENT_SENTENCE_END) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_END");
    } else if (event == TTS_EVENT_SYNTHESIS_COMPLETE) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SYNTHESIS_COMPLETE");
        if (_audioController != nil) {
            // 注意这里的event事件是指语音合成完成，而非播放完成，播放完成需要由voicePlayer对象来进行通知
            [_audioController drain];
        }
        
        dispatch_async(dispatch_get_main_queue(), ^{
            self->textViewEvent.text = @"TTS_EVENT_SYNTHESIS_COMPLETE";
        });
        TLog(@"onStreamInputTtsEventCallback (%d) TTS_EVENT_SYNTHESIS_COMPLETE", event);
    } else if (event == TTS_EVENT_TASK_FAILED) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_TASK_FAILED:%s", error_msg);
        if (_audioController != nil) {
            // 注意这里的event事件是指语音合成完成，而非播放完成，播放完成需要由voicePlayer对象来进行通知
            [_audioController drain];
            [_audioController stopPlayer];
        }
        
        NSString *errorCode = @"";
        NSString *allResponse = [NSString stringWithUTF8String:all_response];
        if (allResponse) {
            NSError *jsonError;
            NSData *jsonData = [allResponse dataUsingEncoding:NSUTF8StringEncoding];
            NSDictionary *jsonObject = [NSJSONSerialization JSONObjectWithData:jsonData options:0 error:&jsonError];
            if (!jsonError) {
                NSDictionary *header = jsonObject[@"header"];
                if ([header isKindOfClass:[NSDictionary class]] && header[@"error_code"]) {
                    errorCode = header[@"error_code"];
                }
            }
        }
        NSString *content = [NSString stringWithFormat:@"TTS_EVENT_TASK_FAILED error_message(%s) error_code(%@)",
                              error_msg,
                               errorCode];
        dispatch_async(dispatch_get_main_queue(), ^{
            self->textViewEvent.text = content;
        });
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_TASK_FAILED");
    }
}

- (void)onStreamInputTtsDataCallback:(char*)buffer len:(int)len {
//    NSString *log = [NSString stringWithFormat:@"\n音频回调 %d bytes", len];
//    TLog(log);

    if (buffer != NULL && len > 0 && _audioController != nil) {
        [_audioController write:(char*)buffer Length:(unsigned int)len];
    }
}

-(void)onStreamInputTtsLogTrackCallback:(NuiSdkLogLevel)level
                             logMessage:(const char *)log {
    TLog(@"onStreamInputTtsLogTrackCallback log level:%d, message -> %s", level, log);
}


@end
