# -*- coding: utf-8 -*-
# pylint: disable=protected-access
import logging
import functools

from mcp.types import CallToolResult
from agentscope.message import TextBlock
from agentscope.tool import ToolResponse
from agentscope.mcp import MCPClientBase


logger = logging.getLogger(__name__)


def sandbox_tool_adapter(func):
    """
    Sandbox Tool Adapter.

    Wraps a sandbox tool function so that its output is always converted
    into a ToolResponse object, which is required by the Toolkit.

    This adapter preserves the original function signature and docstring
    so that JSON schemas can be correctly generated by the Toolkit.

    Args:
        func: Original sandbox tool function (may return dict, string, etc.).

    Returns:
        A callable that produces ToolResponse instead of raw data.
    """

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        res = func(*args, **kwargs)
        if isinstance(res, ToolResponse):
            return res

        try:
            mcp_res = CallToolResult.model_validate(res)
            as_content = MCPClientBase._convert_mcp_content_to_as_blocks(
                mcp_res.content,
            )
            resp = ToolResponse(
                content=as_content,
                metadata=mcp_res.meta,
            )
            return resp
        except Exception as e:
            logger.warning(
                (
                    f"Failed to convert tool result to ToolResponse. "
                    f"Function: {func.__name__}, "
                    f"Args: {args}, "
                    f"Kwargs: {kwargs}, "
                    f"Result type: {type(res).__name__}, "
                    f"Result: {res!r}, "
                    f"Error: {e}"
                ),
                exc_info=True,
            )
            return ToolResponse(
                content=[
                    TextBlock(
                        text=str(res),
                    ),
                ],
            )

    return wrapper
