package com.aliyun.test;

import com.aliyun.auth.credentials.Credential;
import com.aliyun.auth.credentials.provider.StaticCredentialProvider;
import com.aliyun.sdk.gateway.pop.Configuration;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.sdk.service.rdsai20250507.AsyncClient;
import com.aliyun.sdk.service.rdsai20250507.models.*;
import darabonba.core.ResponseIterable;
import darabonba.core.client.ClientOverrideConfiguration;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;

public class CustomAgentDemo {
    public interface AnswerCallback {
        void onAnswerChunk(ChatMessagesResponseBody chunk);
    }

    private AsyncClient createClient() throws Exception {
        StaticCredentialProvider provider = StaticCredentialProvider.create(
                Credential.builder()
                        // 请确保已设置环境变量ALIBABA_CLOUD_ACCESS_KEY_ID和ALIBABA_CLOUD_ACCESS_KEY_SECRET。
                        .accessKeyId(System.getenv("ALIBABA_CLOUD_ACCESS_KEY_ID"))
                        .accessKeySecret(System.getenv("ALIBABA_CLOUD_ACCESS_KEY_SECRET"))
                        //.securityToken(System.getenv("ALIBABA_CLOUD_SECURITY_TOKEN")) // 使用STS token
                        .build());
        AsyncClient client = AsyncClient.builder()
                .region("cn-hangzhou")
                .credentialsProvider(provider)
                .serviceConfiguration(Configuration.create()
                        .setSignatureVersion(SignatureVersion.V3)
                ).overrideConfiguration(
                        ClientOverrideConfiguration.create()
                                .setProtocol("HTTPS")
                                // Endpoint 请参考 https://api.aliyun.com/product/RdsAi
                                .setEndpointOverride("rdsai.aliyuncs.com")
                ).build();
        return client;
    }

    /**
     * 发送对话消息
     */
    public void chatMessages(String query, String conversationId, ChatMessagesRequest.Inputs inputs, CustomAgentDemo.AnswerCallback callback) throws Exception {
        try (AsyncClient client = createClient()) {
            ChatMessagesRequest request = ChatMessagesRequest.builder()
                    .conversationId(conversationId)
                    .inputs(inputs)
                    .query(query)
                    .build();
            ;
            ResponseIterable<ChatMessagesResponseBody> iterable = client.chatMessagesWithResponseIterable(request);
            for (ChatMessagesResponseBody event : iterable) {
                callback.onAnswerChunk(event);
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    /**
     * 创建专属Agent
     */
    public String createCustomAgent(String agentName, String prompt, boolean enableTools, List<String> tools) throws Exception {
        CompletableFuture<CreateCustomAgentResponse> response;
        try (AsyncClient client = createClient()) {
            CreateCustomAgentRequest request = CreateCustomAgentRequest.builder()
                    .name(agentName)
                    .systemPrompt(prompt)
                    .enableTools(enableTools)
                    .tools(tools)
                    .build();
            response = client.createCustomAgent(request);
            return response.get().getBody().getId();
        }
    }

    /**
     * 通过名称获取专属Agent
     */
    public ListCustomAgentResponseBody.Data getCustomAgent(String agentName) throws Exception {
        CompletableFuture<ListCustomAgentResponse> response;
        try (AsyncClient client = createClient()) {
            ListCustomAgentRequest request = ListCustomAgentRequest.builder()
                    .pageNumber(1L)
                    .pageSize(30L)
                    .build();
            response = client.listCustomAgent(request);
            for (ListCustomAgentResponseBody.Data agent : response.get().getBody().getData()) {
                if (agent.getName().equals(agentName)) {
                    return agent;
                }
            }
        }
        return null;
    }

    /**
     * 更新专属Agent
     */
    public String updateCustomAgent(String agentId, String prompt, boolean enableTools, List<String> tools) throws Exception {
        CompletableFuture<UpdateCustomAgentResponse> response;
        try (AsyncClient client = createClient()) {
            UpdateCustomAgentRequest request = UpdateCustomAgentRequest.builder()
                    .customAgentId(agentId)
                    .systemPrompt(prompt)
                    .enableTools(enableTools)
                    .tools(tools)
                    .build();
            response = client.updateCustomAgent(request);
            return response.get().getBody().getId();
        }
    }

    /**
     * 删除专属Agent
     */
    public String deleteCustomAgent(String agentId) throws Exception {
        CompletableFuture<DeleteCustomAgentResponse> response;
        try (AsyncClient client = createClient()) {
            DeleteCustomAgentRequest request = DeleteCustomAgentRequest.builder()
                    .customAgentId(agentId)
                    .build();
            response = client.deleteCustomAgent(request);
            return response.get().getBody().getResult();
        }
    }

    public static void main(String[] args) throws Exception {
        CustomAgentDemo messagesMultiTurnDemo = new CustomAgentDemo();

        String agentName = "copilot-agent";
        // 创建用户Agent
        String customAgentId = messagesMultiTurnDemo.createCustomAgent(agentName, "copilot-agent-prompt", true, Arrays.asList("describe_db_instances"));

        // 更新用户Agent
        String updateCustomAgentId = messagesMultiTurnDemo.updateCustomAgent(customAgentId, "你是一个专门获取指定地域下实例信息列表的专属agent，你可以调用[describe_db_instances]工具来帮你获取信息，并为我生成详细的信息总结。", true, Arrays.asList("describe_db_instances"));

        // 查询用户Agent
        ListCustomAgentResponseBody.Data customAgent = messagesMultiTurnDemo.getCustomAgent(agentName);
        if (!customAgent.getName().equals(agentName)) {
            throw new RuntimeException("Agent name mismatch: expected " + agentName + ", got " + customAgent.getName());
        }
        if (!customAgent.getSystemPrompt().equals("你是一个专门获取指定地域下实例信息列表的专属agent，你可以调用[describe_db_instances]工具来帮你获取信息，并为我生成详细的信息总结。")) {
            throw new RuntimeException("Agent prompt mismatch: expected '你是一个专门获取指定地域下实例信息列表的专属agent，你可以调用[describe_db_instances]工具来帮你获取信息，并为我生成详细的信息总结。', got " + customAgent.getSystemPrompt());
        }

        ChatMessagesRequest.Inputs inputs = ChatMessagesRequest.Inputs.builder()
                .timezone("Asia/Shanghai")
                .regionId("cn-hangzhou")
                .language("zh-CN")
                .customAgentId(customAgentId)
                .build();

        // 开始发送对话
        messagesMultiTurnDemo.chatMessages("查询杭州地域有哪些实例即将到期", null, inputs, new CustomAgentDemo.AnswerCallback() {
            @Override
            public void onAnswerChunk(ChatMessagesResponseBody event) {
                if ("message".equals(event.getEvent())) {
                    String answer = event.getAnswer();
                    if (answer != null) {
                        System.out.print(answer);
                    }
                }
            }
        });

        //删除用户Agent
        String deleteCustomAgent = messagesMultiTurnDemo.deleteCustomAgent(customAgentId);
        System.out.println(System.lineSeparator() + deleteCustomAgent);
    }
}