package com.aliyun.test;

import com.alibaba.fastjson.JSON;
import com.aliyun.auth.credentials.Credential;
import com.aliyun.auth.credentials.provider.StaticCredentialProvider;
import com.aliyun.sdk.gateway.pop.Configuration;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.sdk.service.rdsai20250507.AsyncClient;
import com.aliyun.sdk.service.rdsai20250507.models.*;
import darabonba.core.ResponseIterable;
import darabonba.core.client.ClientOverrideConfiguration;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;

public class ChatMessagesMultiTurnDemo {
    public interface AnswerCallback {
        void onAnswerChunk(ChatMessagesResponseBody chunk);
    }

    private AsyncClient createClient() throws Exception {
        StaticCredentialProvider provider = StaticCredentialProvider.create(
                Credential.builder()
                        // 请确保已设置环境变量ALIBABA_CLOUD_ACCESS_KEY_ID和ALIBABA_CLOUD_ACCESS_KEY_SECRET。
                        .accessKeyId(System.getenv("ALIBABA_CLOUD_ACCESS_KEY_ID"))
                        .accessKeySecret(System.getenv("ALIBABA_CLOUD_ACCESS_KEY_SECRET"))
                        //.securityToken(System.getenv("ALIBABA_CLOUD_SECURITY_TOKEN")) // 使用STS token
                        .build());
        AsyncClient client = AsyncClient.builder()
                .region("cn-hangzhou")
                .credentialsProvider(provider)
                .serviceConfiguration(Configuration.create()
                        .setSignatureVersion(SignatureVersion.V3)
                ).overrideConfiguration(
                        ClientOverrideConfiguration.create()
                                .setProtocol("HTTPS")
                                // Endpoint 请参考 https://api.aliyun.com/product/RdsAi
                                .setEndpointOverride("rdsai.aliyuncs.com")
                ).build();
        return client;
    }

    /**
     * 发送对话消息
     */
    public void chatMessages(String query, String conversationId, ChatMessagesRequest.Inputs inputs, ChatMessagesMultiTurnDemo.AnswerCallback callback) throws Exception {
        try (AsyncClient client = createClient()) {
            ChatMessagesRequest request = ChatMessagesRequest.builder()
                    .conversationId(conversationId)
                    .inputs(inputs)
                    .query(query)
                    .build();
            ;
            ResponseIterable<ChatMessagesResponseBody> iterable = client.chatMessagesWithResponseIterable(request);
            for (ChatMessagesResponseBody event : iterable) {
                callback.onAnswerChunk(event);
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    /**
     * 查看具体对话消息
     */
    public GetMessagesResponseBody getMessages(String conversationId) {
        try (AsyncClient client = createClient()) {
            GetMessagesRequest request = GetMessagesRequest.builder()
                    .conversationId(conversationId)
                    .build();
            CompletableFuture<GetMessagesResponse> messages = client.getMessages(request);
            return messages.get().getBody();
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    public static void main(String[] args) throws Exception {
        ChatMessagesMultiTurnDemo messagesMultiTurnDemo = new ChatMessagesMultiTurnDemo();
        final AtomicReference<String> conversationId = new AtomicReference<>();

        ChatMessagesRequest.Inputs inputs = ChatMessagesRequest.Inputs.builder()
                .timezone("Asia/Shanghai")
                .regionId("cn-hangzhou")
                .language("zh-CN")
                .build();
        // 从初次对话中获取会话ID
        messagesMultiTurnDemo.chatMessages("查询杭州地域的实例列表", null, inputs, new ChatMessagesMultiTurnDemo.AnswerCallback() {
            @Override
            public void onAnswerChunk(ChatMessagesResponseBody event) {
                if ("message".equals(event.getEvent())) {
                    String answer = event.getAnswer();
                    if (answer != null) {
                        System.out.print(answer);
                    }
                    conversationId.set(event.getConversationId());
                }
            }
        });
        System.out.println(System.lineSeparator() + "conversationId:" + conversationId.get());
        // 发起二次对话，使用获取到的会话ID
        messagesMultiTurnDemo.chatMessages("查询第一个实例的详细信息", conversationId.get(), inputs, new ChatMessagesMultiTurnDemo.AnswerCallback() {
            @Override
            public void onAnswerChunk(ChatMessagesResponseBody event) {
                if ("message".equals(event.getEvent())) {
                    String answer = event.getAnswer();
                    if (answer != null) {
                        System.out.print(answer);
                    }
                }
            }
        });

        GetMessagesResponseBody messagesResponseBody = messagesMultiTurnDemo.getMessages(conversationId.get());
        System.out.println(System.lineSeparator() + "messagesResponseBody:" + JSON.toJSONString(messagesResponseBody));

    }
}