# -*- coding: utf-8 -*-
import os

from alibabacloud_rdsai20250507.client import Client as RdsAi20250507Client
from alibabacloud_tea_openapi import models as open_api_models
from alibabacloud_rdsai20250507 import models as rds_ai_20250507_models
from alibabacloud_tea_util import models as util_models

class CustomAgentDemo:
    def __init__(self):
        pass

    @staticmethod
    def create_client() -> RdsAi20250507Client:

        # 请确保已设置环境变量ALIBABA_CLOUD_ACCESS_KEY_ID和ALIBABA_CLOUD_ACCESS_KEY_SECRET
        config = open_api_models.Config(
            access_key_id=os.getenv("ALIBABA_CLOUD_ACCESS_KEY_ID"),
            access_key_secret=os.getenv("ALIBABA_CLOUD_ACCESS_KEY_SECRET")
        )
        # Endpoint 请参考 https://api.aliyun.com/product/RdsAi
        config.endpoint = 'rdsai.aliyuncs.com'
        return RdsAi20250507Client(config)

    # 发送对话消息
    def do_chat_messages(self, search_query: str, custom_agent_id: str, conversation_id: str):
        runtime = util_models.RuntimeOptions(read_timeout=1000 * 100)
        # 指定专属Agent对话则传入专属AgentID，专属AgentID在调用CreateCustomAgent接口成功后会返回，也可以通过接口ListCustomAgent获取当前创建的专属Agent列表中查询，详情查看ListCustomAgent接口文档
        # inputs = rds_ai_20250507_models.ChatMessagesRequestInputs(language="zh-CN", region_id="cn-hangzhou", timezone="Asia/Shanghai",
        #                                                  custom_agent_id="5f1bbe8a-88d8-4a72-81e5-2a5d9d43****")
        inputs = rds_ai_20250507_models.ChatMessagesRequestInputs(
            language="zh-CN",
            region_id="cn-hangzhou",
            timezone="Asia/Shanghai",
            custom_agent_id=custom_agent_id
        )
        # 多轮对话需要指定conversationId（会话ID），会话ID在调用ChatMessages接口成功后会返回，详情查看ChatMessages接口文档
        request = rds_ai_20250507_models.ChatMessagesRequest(
            query=search_query,
            inputs=inputs,
            conversation_id=conversation_id
        )
        sse_response = self.create_client().chat_messages_with_sse(tmp_req=request, runtime=runtime)
        return sse_response

    # 创建专属Agent
    def do_create_custom_agent(self, agent_name: str, prompt: str,
                               enable_tools: bool, tools: list) -> str:
        try:
            req = rds_ai_20250507_models.CreateCustomAgentRequest(name=agent_name, system_prompt=prompt, enable_tools=enable_tools, tools=tools)
            resp = self.create_client().create_custom_agent(req)
            return resp.body.id
        except Exception as e:
            raise RuntimeError(e)

    # 更新专属Agent
    def do_update_custom_agent(self, custom_agent_id: str, system_prompt: str) -> rds_ai_20250507_models.UpdateCustomAgentResponseBody:
        try:
            req = rds_ai_20250507_models.UpdateCustomAgentRequest(custom_agent_id=custom_agent_id, system_prompt=system_prompt)
            resp = self.create_client().update_custom_agent(req)
            return resp.body
        except Exception as e:
            raise RuntimeError(e)

    def do_get_custom_agent(self, agent_name: str, page_number: int, page_size: int):
        try:
            req = rds_ai_20250507_models.ListCustomAgentRequest(page_number=page_number, page_size=page_size)
            resp = self.create_client().list_custom_agent(req)
            body = resp.body
            if not body:
                return None
            data_list = body.data
            if not data_list:
                return None
            for data in data_list:
                if getattr(data, "name", None) == agent_name:
                    return getattr(data, "id", None)
        except Exception as e:
            raise RuntimeError(e)
        return None

    # 删除专属Agent
    def do_delete_custom_agent(self, custom_agent_id: str):
        try:
            req = rds_ai_20250507_models.DeleteCustomAgentRequest(custom_agent_id=custom_agent_id)
            resp = self.create_client().delete_custom_agent(req)
            return resp.body
        except Exception as e:
            raise RuntimeError(e)

    @staticmethod
    def main() -> None:
        custom_agent = CustomAgentDemo()
        agent_name = "copilot-agent-python"
        create_custom_agent_id = custom_agent.do_create_custom_agent(agent_name=agent_name, prompt="copilot-agent-prompt-demo", enable_tools=True,
                                                                     tools=["describe_db_instances"])
        print("custom_agent_id: " + create_custom_agent_id)

        update_agent_body = custom_agent.do_update_custom_agent(custom_agent_id=create_custom_agent_id, system_prompt="你是一个专门获取指定地域下实例信息列表的专属agent，你可以调用[describe_db_instances]工具来帮你获取信息，并为我生成详细的信息总结。")
        print("custom_agent_id_update: " + update_agent_body.id)

        custom_agent_id_by_query = custom_agent.do_get_custom_agent(agent_name, 1, 30)
        print("custom_agent_id: " + custom_agent_id_by_query)

        chat_messages_response = custom_agent.do_chat_messages(search_query="查询杭州地域有哪些实例即将到期", custom_agent_id=create_custom_agent_id,
                                                               conversation_id=None)
        for chunk in chat_messages_response:
            body = chunk.body
            if body is not None and body.event == 'message':
                print(f"{body.answer}", end="")
                conversation_id = body.conversation_id

        del_custom_agent_body = custom_agent.do_delete_custom_agent(custom_agent_id=create_custom_agent_id)
        print("\n" + "delete_agent_result: " + del_custom_agent_body.result)

if __name__ == '__main__':
    CustomAgentDemo.main()
