import os
import logging
import asyncio
import argparse
from loguru import logger
from dingtalk_stream import AckMessage
import dingtalk_stream

from typing import Callable
from rds_copilot import RdsCopilot, MessageEvent


def define_options():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--client_id",
        dest="client_id",
        default=os.getenv("DINGTALK_APP_CLIENT_ID"),
        help="app_key or suite_key from https://open-dev.digntalk.com",
    )
    parser.add_argument(
        "--client_secret",
        dest="client_secret",
        default=os.getenv("DINGTALK_APP_CLIENT_SECRET"),
        help="app_secret or suite_secret from https://open-dev.digntalk.com",
    )
    options = parser.parse_args()
    return options


async def call_with_stream(request_content: str, callback: Callable[[str], None], rds_copilot: RdsCopilot):
    full_content = ""  # 累积完整响应内容
    
    # 调用 rds_copilot 的 chat 方法，获取流式响应
    for event in rds_copilot.chat(request_content):
        # 只处理 MessageEvent 类型的事件，累积文本内容
        if isinstance(event, MessageEvent) and event.text:
            full_content += event.text
            # 每次收到新的 chunk 就立即调用流式更新接口更新内容
            await callback(full_content)
            logger.info(
                f"调用流式更新接口更新内容：current_length: {len(full_content)}"
            )
        # 其他类型的事件（ToolCallStart, DocumentEvent 等）可以在这里处理
        # 但主要关注文本内容的流式更新
    
    logger.info(
        f"Request Content: {request_content}\nFull response: {full_content}\nFull response length: {len(full_content)}"
    )
    
    return full_content


async def handle_reply_and_update_card(self: dingtalk_stream.ChatbotHandler, incoming_message: dingtalk_stream.ChatbotMessage):
    # 卡片模板 ID
    card_template_id = "8aebdfb9-28f4-4a98-98f5-396c3dde41a0.schema"  # 该模板只用于测试使用，如需投入线上使用，请导入卡片模板 json 到自己的应用下
    content_key = "content"
    card_data = {content_key: ""}
    card_instance = dingtalk_stream.AICardReplier(
        self.dingtalk_client, incoming_message
    )
    # 先投放卡片: https://open.dingtalk.com/document/orgapp/create-and-deliver-cards
    card_instance_id = await card_instance.async_create_and_deliver_card(
        card_template_id, card_data
    )

    # 初始化 RdsCopilot 实例
    rds_copilot = RdsCopilot()

    # 再流式更新卡片: https://open.dingtalk.com/document/isvapp/api-streamingupdate
    async def callback(content_value: str):
        return await card_instance.async_streaming(
            card_instance_id,
            content_key=content_key,
            content_value=content_value,
            append=False,
            finished=False,
            failed=False,
        )

    try:
        full_content_value = await call_with_stream(
            incoming_message.text.content, callback, rds_copilot
        )
        await card_instance.async_streaming(
            card_instance_id,
            content_key=content_key,
            content_value=full_content_value,
            append=False,
            finished=True,
            failed=False,
        )
    except Exception as e:
        self.logger.exception(e)
        await card_instance.async_streaming(
            card_instance_id,
            content_key=content_key,
            content_value="",
            append=False,
            finished=False,
            failed=True,
        )


class CardBotHandler(dingtalk_stream.ChatbotHandler):
    def __init__(self, logger: logging.Logger = logger):
        super(dingtalk_stream.ChatbotHandler, self).__init__()
        if logger:
            self.logger = logger

    async def process(self, callback: dingtalk_stream.CallbackMessage):
        incoming_message = dingtalk_stream.ChatbotMessage.from_dict(callback.data)
        self.logger.info(f"收到消息：{incoming_message}")

        if incoming_message.message_type != "text":
            self.reply_text("俺只看得懂文字喔~", incoming_message)
            return AckMessage.STATUS_OK, "OK"

        asyncio.create_task(handle_reply_and_update_card(self, incoming_message))
        return AckMessage.STATUS_OK, "OK"


def main():
    options = define_options()

    credential = dingtalk_stream.Credential(options.client_id, options.client_secret)
    client = dingtalk_stream.DingTalkStreamClient(credential)
    client.register_callback_handler(
        dingtalk_stream.ChatbotMessage.TOPIC, CardBotHandler()
    )
    client.start_forever()


if __name__ == "__main__":
    main()
