//
//  ChatViewController.swift
//  ChatViewController
//
//  Created by songsong.sss on 2025/3/24.
//

import UIKit
import multimodal_dialog
import SnapKit
import AVFoundation



class ChatViewController: UIViewController, AudioHandlerDelegate{

    
    let videoView = UIView() //视频预览流窗口
    var HOST = ""
    var API_KEY = ""
    var WORKSPACE_ID = ""
    var APP_ID = ""
    var chain = ChainMode.WebSocket
    var isconnected : Bool
    var conversation:MultiModalDialog?
    var isInDialog = false
    var audioManager: AudioManager?
    var image_url: String?
    var myState: DialogState?
    var useKws = false
    var duplexMode = true
    
    
    lazy var titleLabel1: UILabel = createLabel(text: "用户说:")
    lazy var titleLabel2: UILabel = createLabel(text: "AI 说:")
    lazy var titleLabel3: UILabel = createLabel(text: "")
    
    lazy var textView1: UITextView = createTextField()
    lazy var textView2: UITextView = createTextField()
    lazy var textView3: UITextView = createTextField()
    
    private let buttonStack: UIStackView = {
            let stack = UIStackView()
            stack.axis = .horizontal
            stack.spacing = 20
            stack.distribution = .fillEqually
            return stack
        }()
        
    private let recordButton: UIButton = {
            let btn = UIButton(type: .system)
            btn.setTitle("开始收音", for: .normal)
            btn.backgroundColor = .systemBlue
            btn.tintColor = .white
            btn.layer.cornerRadius = 8
            return btn
        }()
        
    private let interruptButton: UIButton = {
            let btn = UIButton(type: .system)
            btn.setTitle("打断播放", for: .normal)
            btn.backgroundColor = .systemGreen
            btn.tintColor = .white
            btn.layer.cornerRadius = 8
            return btn
        }()

    
    init(){
        self.isconnected = false
        super.init(nibName: nil, bundle: nil)
    }
    
    public func updateParam(url: String, apiKey: String , workSpaceId: String, appId: String, chain: String ,image_url:String){
        self.HOST = url
        self.API_KEY = apiKey
        self.image_url = image_url
        self.WORKSPACE_ID = workSpaceId
        self.APP_ID = appId
        
        
        self.audioManager = AudioManager()
        self.audioManager?.setPlayerSampleRate(16000)
        self.audioManager?.setAudioHandlerDelegate(self)
        
        if chain.lowercased() == "websocket" {
            self.chain = ChainMode.WebSocket

        }else{
            self.chain = ChainMode.RTC
        }
        var mode = DialogMode.duplex
        if !duplexMode{
            mode = DialogMode.tap2talk
        }
        
        
        
        self.conversation = MultiModalDialog(url: self.HOST, chainMode: self.chain,  workSpaceId:self.WORKSPACE_ID ,
                                             appId: self.APP_ID, mode: mode)
        //打开语音唤醒功能
        if(useKws){
            self.conversation?.enableKWS(enableKws: true, wakeupOnly:false)
        }
        
        //for debug
        self.conversation?.setModel(model: "multimodal-dialog")
    }
    override func viewDidLoad() {
        super.viewDidLoad()
        print("==================viewDidLoad")
        setupDialogCallbacks()
        
        //视频模式页面配置
        if self.chain == ChainMode.RTC {
            view.addSubview(videoView)
            videoView.backgroundColor = .white
            
            videoView.snp.makeConstraints { make in
                make.edges.equalToSuperview()
            }
            
            videoView.addGestureRecognizer(UITapGestureRecognizer(target: self, action: #selector(videoViewTapped)))
        }else {
        //音频模式页面配置
            setupUI()
            setupActions()
        }
        //启动连接
        connect()
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        TYLogger.shared.debug("==========viewWillDisappear")
        self.isInDialog = false
        self.audioManager?.cleanPlayerBuffer()
        self.audioManager?.stopPlaying()
        self.audioManager?.stopRecording(notify: false)
        self.conversation?.stop()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        TYLogger.shared.debug("==========viewDidDisappear")
    }
    
    func connect(){
        self.conversation?.stop()
        isInDialog = false
        
        var mode = DialogMode.duplex
        if(!duplexMode){
            mode = DialogMode.tap2talk
        }
        
        var promptParam = ["user_name" : "大米"]
        
        var params = MultiModalRequestParam{ multiBuilder in
            multiBuilder.upStream = MultiModalRequestParam.UpStream(builder: { upstreamBuilder in
                upstreamBuilder.mode = mode.rawValue
                upstreamBuilder.type = "AudioAndVideo"
            })
            multiBuilder.clientInfo = MultiModalRequestParam.ClientInfo(builder: {
                clientInfoBuilder in
                clientInfoBuilder.userId = "test-ios-user"
                clientInfoBuilder.device = MultiModalRequestParam.ClientInfo.Device(uuid: "12345")
            })
            multiBuilder.downStream = MultiModalRequestParam.DownStream(builder: {
                downStreamBuilder in
                downStreamBuilder.sampleRate = 16000
                downStreamBuilder.voice="longxiaochun_v2"
            })
            multiBuilder.bizParams = MultiModalRequestParam.BizParams(builder: {
                bizBuilder in
                bizBuilder.userPromptParams = promptParam
            })
        }
        
        self.conversation?.start(apiKey:self.API_KEY, params: params, completion: { success, error in
            if success {
                print("success")
                self.isconnected = true
                
                //唤醒下提前打开录音
                if self.useKws {
                    self.audioManager?.startRecording()
                }
            }else {
                self.isconnected = false
                if let e = error {
                    print("连接失败，错误：\(String(describing: error))")
                } else {
                    print("连接失败，无错误信息")
                }
            }})
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    
    // 必须使用 @objc 标记，因为这个方法需要暴露给 Objective-C 运行时
    @objc func videoViewTapped() {
        TYLogger.shared.debug("videoViewTapped")
    }
    
    deinit{
        self.conversation?.stop()
    }
    
    
    private func setupDialogCallbacks() {
        self.conversation?.onConnected = {
            print("callback: onConnectionReady ")
        }
        
        self.conversation?.onVolumeChanged = { volume , type in
        }
        
        self.conversation?.onConversationStarted = {
            TYLogger.shared.debug("callback: onConversationStarted")
            //RTC 进入视频模式
            if self.chain == ChainMode.RTC {
                self.conversation?.requestToRespond(type: "prompt", text: "", params: self.createVideoChatParams())
            }else{
                self.audioManager?.startRecording()
            }
        }
        
        self.conversation?.onKeywordSpotted = {word, type in
            
            DispatchQueue.main.async {
                TYLogger.shared.debug("onKeywordSpotted::: \(word), \(type)")
                self.titleLabel3.text = "Wakeup:\(word)"
            }
        }
        
        self.conversation?.onConversationEvent = { event in
            switch event{
            case .RespondingStarted:
                self.audioManager?.startPlaying()
                break
            case .RespondingEnded:
                //结束发送tts数据
                self.audioManager?.drainPlayer()
                break
            case .InterruptAccepted:
                TYLogger.shared.debug("InterruptAccepted")
                self.audioManager?.cleanPlayerBuffer()
                break
            case .SpeechStarted:
                break
            case .SpeechEnded:
                if self.chain != ChainMode.RTC  && !self.duplexMode{
                    self.audioManager?.stopRecording(notify: false)
                }
                break
            default:
                break
            }
            print("callback: onConversationEvent ",event)
        }
        
        self.conversation?.onMessageReceived = { message, type in
            let payload = message?["payload"] as? [String: Any]
            let output = payload?["output"] as? [String: Any]
            
            
            let dialogId = output?["dialog_id"] as? String
            var debug_info = ("dialog_id:").appending(dialogId ?? "")
            //用户识别结果
            if type == ResponsetMessageType.speaking {
                let text = output?["text"] as? String
                DispatchQueue.main.async {
                    self.textView1.text = text
                }
                
            }else{
            //大模型回复结果
                let spoken = output?["spoken"] as? String
                let llm_request_id = output?["llm_request_id"] as? String
                let round_id = output?["round_id"] as? String
                
                DispatchQueue.main.async {
                    self.textView2.text = spoken
                }
                let commands = self.handleCommand(output: output)
                
                debug_info = debug_info.appending("\n llm_request_id:").appending(llm_request_id ?? "")
                debug_info = debug_info.appending("\n round_id:").appending(round_id ?? "")
            }
            DispatchQueue.main.async {
                self.textView3.text = debug_info
            }
            TYLogger.shared.debug("onMessageReceived handled over")
        }
        
        self.conversation?.onConversationStatechanged = {state in
            DispatchQueue.main.async {
                self.titleLabel3.text = state.rawValue
            }
            self.myState = state
            switch state {
            case .idle:
                break
            case .listening:
                self.muteInterruptButton()
                print("camera::::", self.conversation?.getCurrentCameraDirection())
                if self.chain != ChainMode.RTC {
                    self.audioManager?.startRecording()
                }
                break
            case .responding:
                self.unMuteInterruptButton()
                break
            case .thinking:
                break
            }
            print("callback: onConversationStatechanged ",state)
        }
        self.conversation?.onErrorReceived = { err in
            print("callback: onErrorReceived: ",err.key, err.message )
            if(err.key.hasPrefix("RTCException.")) {
            }
            let msg = err.key.appending(err.message ?? "")
            self.showAlert(msg:msg)
        }
        
        self.conversation?.onFirstVideoPacketSent = {
            print("callback: onFirstVideoPacketSent")
        }
        
        self.conversation?.onConnectionStatusChanged = {
            state in
            print("onConnectionStatusChanged ::::::" ,state)
            switch state{
            case .failed :
                self.conversation?.stop()
                break
            case .inited:
                break
            case .disconnected:
                break
            case .connected:
                print("==================connected")
                self.isInDialog = true
                //设置视频显示
                if self.chain == ChainMode.RTC {
                    DispatchQueue.main.async {
                        self.conversation?.setupLocalView(self.videoView, config: TYVideoConfig(fps: 12, width: 720, height: 1080, bitrate: 0))
                        self.conversation?.publishLocalVideo()
                    }
                }
                
                break
            case .connecting:
                break
            case .reconnecting:
                break
            @unknown default:
                break
            }
            
        }
        
        self.conversation?.onSynthesizedData = { data, len in
            TYLogger.shared.info("callback tts \(len)")
            let audio = Data(bytes: UnsafeRawPointer(data), count: Int(len))
            self.audioManager?.writeAudioData(audio)
            TYLogger.shared.info("callback tts \(len) write done")
        }
            
    }
    
    //RTC only
    private func createVideoChatParams() -> [String: Any]{
        var video:[String: Any] = [
            "action":"connect",
            "type" : "voicechat_video_channel"
        ]
        var videos = [video]
        
        var updateParam = MultiModalRequestParam{ multiBuilder in
            multiBuilder.bizParams = MultiModalRequestParam.BizParams(builder: {
                bizBuilder in
                bizBuilder.videos = videos
            })
        }
        return updateParam.parameters
    }
    
    //for VQA
    private func createImageParams() -> [String: Any]{
        var imageObject:[String: Any] = [
            "type":"url",
            "value":self.image_url ?? ""
        ]
//        var imageObjectBase64:[String: Any] = [
//                    "type":"base64",
//                    "value":"imagebase64"
//        ]
        var images = [imageObject]
        
        var updateParam = MultiModalRequestParam{ multiBuilder in
            multiBuilder.images = images
        }
        return updateParam.parameters
    }
    
    
    /// callback for AudioManager
    func playerDidStart() {
        self.conversation?.sendLocalRespondingStarted()
    }
    
    func playerDidStop() {
        self.conversation?.sendLocalRespondingEnded()
    }
    
    func didReceiveRecordingData(_ buffer: UnsafeMutablePointer<UInt8>?, length: Int32) {
        if (buffer != nil){
            guard let data = buffer else
            {return}
            self.conversation?.sendAudioData(data: data, length: length)
        }
    }
    
    func didReceivePlayingingData(_ buffer: UnsafeMutablePointer<UInt8>?, length: Int32) {
        if (buffer != nil){
            guard let data = buffer else
            {return}
            //发送ref
            self.conversation?.sendRefData(data: data, length: length)
        }
    }
    
    //handle command
    private func handleCommand(output: [String: Any]?) -> [[String: Any]]?{
        guard let output = output,
                  let extraInfo = output["extra_info"] as? [String: Any],
                  let commandsString = extraInfo["commands"] as? String,
              let commandsData = commandsString.data(using: .utf8) else {
            return nil
        }
        
        do {
                // 解析commands字符串为JSON数组
                if let commands = try JSONSerialization.jsonObject(with: commandsData) as? [[String: Any]] {
                    
                    for command in commands {
                            if let name = command["name"] as? String, name == "visual_qa" {
                                print("before vqa send img")
                                self.conversation?.requestToRespond(type: "prompt", text: "", params: self.createImageParams())
                                print("vqa send img")
                            }
                        }
                    return commands
                }
                return nil
            } catch {
                print("解析commands时出错: \(error)")
                return nil
            }
    }
    
    
    // 创建标签的辅助方法
    private func createLabel(text: String) -> UILabel {
        let label = UILabel()
        label.text = text
        label.font = UIFont.systemFont(ofSize: 16, weight: .medium)
        label.textAlignment = .right
        label.setContentHuggingPriority(.defaultHigh, for: .horizontal)
        label.translatesAutoresizingMaskIntoConstraints = false
        return label
    }
    
    // 创建文本框的辅助方法
    private func createTextField() -> UITextView {
        let tv = UITextView()
        tv.layer.cornerRadius = 8
        tv.layer.borderColor = UIColor.systemGray4.cgColor
        tv.layer.borderWidth = 1
        tv.font = UIFont.systemFont(ofSize: 14)
        tv.isScrollEnabled = false
        tv.autocorrectionType = .no
        return tv
    }
    
    private func setupUI() {
            view.backgroundColor = .white
            view.addSubview(titleLabel1)
            view.addSubview(textView1)
            view.addSubview(titleLabel2)
            view.addSubview(textView2)
            view.addSubview(textView3)
            view.addSubview(titleLabel3)
            view.addSubview(buttonStack)
            
//            buttonStack.addArrangedSubview(recordButton)
            buttonStack.addArrangedSubview(interruptButton)
            
            // AutoLayout 约束
            let safeArea = view.safeAreaLayoutGuide
            let padding: CGFloat = 20
            
            titleLabel1.translatesAutoresizingMaskIntoConstraints = false
            textView1.translatesAutoresizingMaskIntoConstraints = false
            titleLabel2.translatesAutoresizingMaskIntoConstraints = false
            textView2.translatesAutoresizingMaskIntoConstraints = false
            textView3.translatesAutoresizingMaskIntoConstraints = false
            titleLabel3.translatesAutoresizingMaskIntoConstraints = false
            buttonStack.translatesAutoresizingMaskIntoConstraints = false
            
            NSLayoutConstraint.activate([
                titleLabel1.topAnchor.constraint(equalTo: safeArea.topAnchor, constant: padding),
                titleLabel1.leadingAnchor.constraint(equalTo: safeArea.leadingAnchor, constant: padding),
                titleLabel1.trailingAnchor.constraint(equalTo: safeArea.trailingAnchor, constant: -padding),
                
                textView1.topAnchor.constraint(equalTo: titleLabel1.bottomAnchor, constant: 8),
                textView1.leadingAnchor.constraint(equalTo: safeArea.leadingAnchor, constant: padding),
                textView1.trailingAnchor.constraint(equalTo: safeArea.trailingAnchor, constant: -padding),
                textView1.heightAnchor.constraint(greaterThanOrEqualToConstant: 100),
                
                titleLabel2.topAnchor.constraint(equalTo: textView1.bottomAnchor, constant: padding),
                titleLabel2.leadingAnchor.constraint(equalTo: safeArea.leadingAnchor, constant: padding),
                titleLabel2.trailingAnchor.constraint(equalTo: safeArea.trailingAnchor, constant: -padding),
                
                textView2.topAnchor.constraint(equalTo: titleLabel2.bottomAnchor, constant: 8),
                textView2.leadingAnchor.constraint(equalTo: safeArea.leadingAnchor, constant: padding),
                textView2.trailingAnchor.constraint(equalTo: safeArea.trailingAnchor, constant: -padding),
                textView2.heightAnchor.constraint(greaterThanOrEqualToConstant: 100),
                
                textView3.topAnchor.constraint(equalTo: textView2.bottomAnchor, constant: 8),
                textView3.leadingAnchor.constraint(equalTo: safeArea.leadingAnchor, constant: padding),
                textView3.trailingAnchor.constraint(equalTo: safeArea.trailingAnchor, constant: -padding),
                textView3.heightAnchor.constraint(greaterThanOrEqualToConstant: 100),
                
                titleLabel3.topAnchor.constraint(equalTo: textView3.bottomAnchor, constant: padding),
                titleLabel3.leadingAnchor.constraint(equalTo: safeArea.leadingAnchor, constant: padding),
                titleLabel3.trailingAnchor.constraint(equalTo: safeArea.trailingAnchor, constant: -padding),
                
                
                buttonStack.bottomAnchor.constraint(equalTo: safeArea.bottomAnchor, constant: -padding),
                buttonStack.leadingAnchor.constraint(equalTo: safeArea.leadingAnchor, constant: padding),
                buttonStack.trailingAnchor.constraint(equalTo: safeArea.trailingAnchor, constant: -padding),
                buttonStack.heightAnchor.constraint(equalToConstant: 50)
            ])
        }
    private func setupActions(){
        // 为按钮添加点击事件
        recordButton.addTarget(self, action: #selector(startButtonTapped), for: .touchUpInside)
        interruptButton.addTarget(self, action: #selector(interruptTapped), for: .touchUpInside)
    
    }
    
    private func showAlert(msg: String) {
        let alert = UIAlertController(
            title: "error",
            message: msg,
            preferredStyle: .alert
        )
        let okAction = UIAlertAction(title: "确定", style: .default, handler: nil)
        alert.addAction(okAction)
        DispatchQueue.main.async {
            // 确保在顶层视图控制器显示
            if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene,
                       let sceneDelegate = windowScene.delegate as? SceneDelegate,
                       let window = sceneDelegate.window,
                       let rootVC = window.rootViewController {
                        
                        // 获取当前最顶层的视图控制器
                        var topController = rootVC
                        while let presentedController = topController.presentedViewController {
                            topController = presentedController
                        }
                        
                        topController.present(alert, animated: true)
            }
        }
        
        
    }
    
    // MARK: - Action Handlers
        
        @objc private func startButtonTapped() {
            if recordButton.title(for: .normal) == "开始收音" {
                startSpeech()
                recordButton.setTitle("停止收音", for: .normal)
                recordButton.backgroundColor = .systemRed
//                statusLabel.text = "正在录音..."
                
            } else {
                stopSpeech()
                recordButton.setTitle("开始收音", for: .normal)
                recordButton.backgroundColor = .systemBlue
//                statusLabel.text = "录音已停止"
            }
        }
    
    @objc private func interruptTapped() {
        if(myState == DialogState.responding) {
//            self.audioPlayer?.stop()
            TYLogger.shared.info("interruptTapped")
            interruptButton.setTitle("点击打断", for: .normal)
            self.interrupt()
        }
    }
    
    private func startSpeech(){
        self.conversation?.startSpeech()
    }
    
    private func stopSpeech(){
        self.conversation?.stopSpeech()
    }
    
    private func interrupt(){
        self.conversation?.interrupt()
    }
    
    private func muteInterruptButton(){
        DispatchQueue.main.async {
            self.interruptButton.isEnabled = false
            self.interruptButton.backgroundColor = .systemGray
        }
    }
    
    private func unMuteInterruptButton(){
        DispatchQueue.main.async {
            self.interruptButton.isEnabled = true
            self.interruptButton.backgroundColor = .systemBlue
        }
    }
}



extension [String: Any] {
    func toUTF8String() -> String? {
        do {
            let data = try JSONSerialization.data(withJSONObject: self, options: .prettyPrinted)
            return String(data: data, encoding: .utf8)
        } catch {
            return nil
        }
    }
}

