import React from "react";

type ReturnResult = Required<Pick<React.CSSProperties, "top" | "left">> &
  Pick<React.CSSProperties, "position" | "zIndex">;

const getNearestFixedParent = (container: HTMLElement) => {
  let element = container.parentElement;

  while (element) {
    const { position } = getComputedStyle(element);
    if (position === "fixed" || element === document.body) {
      return element;
    }
    element = element.parentElement;
  }
  return undefined;
};

export const getDropdownPosition = (
  mouseX: number,
  _mouseY: number,
  rangeRect: DOMRect,
  container: HTMLElement,
  documentContainer: HTMLElement,
  dropdownWidth: number,
  dropdownHeight: number,
  dropdownMargin: number,
  isCreatedByFixParagraph = true,
  unfixParagraphHeight = 0
): ReturnResult => {
  // console.log('debug mouseY', mouseY);

  const containerRect = container.getBoundingClientRect();
  const documentContainerRect = documentContainer.getBoundingClientRect();

  const minX = Math.max(rangeRect.left - containerRect.left - dropdownWidth, 22);
  const maxX = rangeRect.left + rangeRect.width - containerRect.left;

  const calcX = Math.min(maxX, Math.max(minX, mouseX));

  let top = rangeRect.bottom - containerRect.top + dropdownMargin + container.scrollTop;
  let left = calcX + dropdownMargin;

  if (left > container.scrollWidth - dropdownWidth) {
    left = mouseX - containerRect.left - dropdownWidth - dropdownMargin;
  }

  left = Math.max(dropdownMargin, left);

  top = Math.min(top, container.scrollHeight - dropdownHeight - dropdownMargin);

  const maxScreenTop = container.scrollTop + containerRect.height - dropdownHeight - dropdownMargin;
  if (top > maxScreenTop) {
    top = maxScreenTop;
  }

  top = Math.max(documentContainerRect.top + container.scrollTop - containerRect.top, top);

  if (isCreatedByFixParagraph && unfixParagraphHeight) {
    const containerInner = container.querySelector(".transcriptionContent");
    const childHeight = containerInner?.clientHeight || 0;

    const containerHeight = Math.min(childHeight, container.scrollHeight);
    top = Math.min(
      top,
      Math.max(0, containerHeight - dropdownHeight - unfixParagraphHeight - dropdownMargin)
    );
  }

  if (!isCreatedByFixParagraph) {
    // unfix
    top = top - container.scrollTop + containerRect.top;
    left = left - container.scrollLeft + containerRect.left;
    const fixedParent = getNearestFixedParent(container);

    if (fixedParent) {
      const fixedParentRect = fixedParent.getBoundingClientRect();
      top -= fixedParentRect.top;
      left -= fixedParentRect.left;
    }
  }

  const extStyles: Pick<React.CSSProperties, "position" | "zIndex"> = isCreatedByFixParagraph
    ? {
        position: "absolute",
        zIndex: 100,
      }
    : {
        position: "fixed",
        zIndex: 300,
      };

  return {
    top,
    left,
    ...extStyles,
  };
};
