import React, { useMemo } from "react";
import styled, { css } from "styled-components";
import { config } from "../../config";
import { WordsStyled } from "../wordsStyled";
import { PureCaret } from "./caret/pureCaret";
import { Controller } from "@tingwu/core";

const { textEditor } = config.theme;

export const ComposingDataWrapper = styled.span<{
  isHighlight: boolean;
}>`
  text-decoration: underline;
  color: ${props => (props.isHighlight ? "#fff" : "rgba(39,38,77,0.85)")};
  position: relative;
`;

export interface ComposingDataProps {
  children: React.ReactNode;
  controller: Controller;
  isHighlight: boolean;
}
export const ComposingData: React.FC<ComposingDataProps> = props => {
  const { children, controller, isHighlight } = props;
  const pureCaretStyle = useMemo<React.CSSProperties>(() => {
    return {
      display: "inline-block",
      position: "relative",
      top: 3,
      height: 16,
    };
  }, []);
  return (
    <ComposingDataWrapper className="ignore" isHighlight={isHighlight}>
      {children}
      <PureCaret controller={controller} style={pureCaretStyle} />
    </ComposingDataWrapper>
  );
};

export const Wrapper = styled(WordsStyled)<{
  isEmptyContent: boolean;
  isHighlight?: boolean;
  readonly?: boolean;
  isUnfix?: boolean;
  isPlaying?: boolean;
  modifyDefaultFontColor?: string;
}>`
  display: ${props => {
    if (!props.isUnfix) {
      return "block";
    }
    return props.isEmptyContent ? "inline-block" : "inline";
  }};
  min-width: 20px;
  overflow-wrap: anywhere;
  outline: none;
  font-size: ${textEditor.fontSize}px;
  line-height: ${textEditor.lineHeight}px;
  min-height: ${textEditor.lineHeight}px;
  font-family: ${props => props.theme.editorFontFamily};
  cursor: ${props => (props.readonly ? "default" : "text")};
  white-space: pre-wrap;

  ${({ modifyDefaultFontColor }) =>
    modifyDefaultFontColor &&
    css`
      color: ${modifyDefaultFontColor};
    `}
  user-select: none;
  cursor: text;
  position: relative;

  span::selection {
    background-color: var(--Line_Primary);
    color: ${props => (props.isHighlight ? "var(--Text_Brand_Primary)" : "var(--Text_Caption)")};
  }

  span::-moz-selection {
    background-color: var(--Line_Primary);
    color: ${props => (props.isHighlight ? "var(--Text_Primary)" : "var(--Text_Caption)")};
  }

  ${props =>
    props.isPlaying &&
    css`
      .word-blue,
      .word-yellow,
      .word-red,
      .qna {
        color: var(--Text_Caption);
        &.played {
          color: var(--Text_Caption);
        }
      }
      .base {
        color: var(--Text_Disabled);
      }
      .played {
        color: ${props.isHighlight ? "var(--Text_Primary)" : "var(--Text_Primary)"};
      }
    `}
`;

export const ViewWordsStyled = styled.div<{ isHighlight: boolean }>`
  display: inline;
  color: ${props => (props.isHighlight ? "var(--Text_Primary)" : "var(--Text_Caption)")};
  *:selection {
    background-color: transparent;
  }
`;
