import React, { useEffect, useRef, useState } from 'react';
import { TipsAgendaContext } from '../../../WuLab/Agenda/TipsAgenda';

import { iMediaAgendaProps } from '../../TWMedia.type';
import './AudioTimeTooltip.less';

// 时间提示信息
export interface iAudioTimeTooltipContent {
  displayTime: string; // 显示时间
  mediaAgenda?: iMediaAgendaProps; // 当前章节摘要
  offsetX: number;
  rectWidth: number;
  minLeft?: number;
}

const audioLeft = 238;
const audioTipRight = 132;

const AudioTimeToolTip: React.FC<{
  tipContent?: iAudioTimeTooltipContent;
}> = (props) => {
  // 容器数据
  const timeTooltipRef = useRef<HTMLDivElement>(null);
  const [timeTooltipPos, setTimeTooltipPos] = useState({
    left: 0,
    arrowLeft: 0,
  });

  const { tipContent } = props;
  useEffect(() => {
    if (!timeTooltipRef.current || !tipContent) return;

    const { offsetX, rectWidth, minLeft } = tipContent;
    if (offsetX === undefined || rectWidth === undefined) return;

    const dom = timeTooltipRef.current;
    const { width } = dom.getBoundingClientRect();
    let left = offsetX - width / 3;
    const bodyWidth = document.body.clientWidth / 2;
    // 计算 tooltip 的基础位置
    if (tipContent.mediaAgenda?.title && tipContent.mediaAgenda?.content) {
      if (bodyWidth - (left + audioLeft) < width + audioTipRight) {
        left = bodyWidth - width - audioTipRight - audioLeft;
      }
    }
    if (minLeft && left < minLeft) {
      left = minLeft;
    }
    setTimeTooltipPos({
      left,
      arrowLeft: calculateArrowPosition(offsetX, left, width),
    });
  }, [tipContent]);

  const calculateArrowPosition = (
    offsetX: number,
    tooltipLeft: number,
    tooltipWidth: number,
  ): number => {
    const minArrowDistance = 20; // 距离边缘的最小距离
    let arrowLeft = offsetX - tooltipLeft;

    // 限制箭头不要太靠近边缘
    if (arrowLeft < minArrowDistance) {
      arrowLeft = minArrowDistance;
    } else if (arrowLeft > tooltipWidth - minArrowDistance) {
      arrowLeft = tooltipWidth - minArrowDistance;
    }

    return arrowLeft;
  };

  // 1. 清空不显示
  if (!tipContent) return null;

  const { displayTime, offsetX } = tipContent;
  const lineLeft = offsetX - 1 || 0;
  // 2. 仅显示时间
  if (!tipContent.mediaAgenda) {
    return (
      <div className="AudioTimeToolTip">
        <div
          className="timeTooltipBox"
          ref={timeTooltipRef}
          style={{
            left: timeTooltipPos.left,
          }}
        >
          <div className="textBox">
            <div className="timeBox">{displayTime}</div>
          </div>
        </div>
        <div
          style={{
            left: lineLeft,
          }}
          className="line"
        />
      </div>
    );
  }

  const { title, content } = tipContent.mediaAgenda;
  // 3. 章节速览Hover:无摘要
  if (title && !content) {
    return (
      <div className="AudioTimeToolTip">
        <div
          className="timeTooltipBox"
          ref={timeTooltipRef}
          style={{
            left: timeTooltipPos.left,
          }}
        >
          <div className="arrow" />
          <div className={`textBox`}>
            <div className="highlightText">
              {title?.length > 15 ? `${title.slice(0, 15)}...` : title || ''}
            </div>
          </div>
        </div>
        <div
          style={{
            left: offsetX - 1,
          }}
          className="line"
        />
      </div>
    );
  }

  // 4. 章节速览hover: 有摘要
  if (title && content) {
    return (
      <div className="AudioTimeToolTip">
        <div
          ref={timeTooltipRef}
          className="agendaSummaryWrap"
          style={{
            left: timeTooltipPos.left,
          }}
        >
          <TipsAgendaContext
            style={{ opacity: 0 }}
            info={
              {
                agendaHeading: title,
                content,
              } as any
            }
            time={displayTime}
            maxHeight={document.body.clientHeight * 0.8}
            arrowLeft={timeTooltipPos.arrowLeft}
          />
          <TipsAgendaContext
            style={{ position: 'absolute', top: 0, left: 0 }}
            info={
              {
                agendaHeading: title,
                content,
              } as any
            }
            time={displayTime}
            maxHeight={document.body.clientHeight * 0.8}
            arrowLeft={timeTooltipPos.arrowLeft}
          />
        </div>
        <div
          style={{
            left: lineLeft,
          }}
          className="line"
        />
      </div>
    );
  }

  return null;
};

export { AudioTimeToolTip };
