/**
 *  *************************************************************
 *  ****************  大模型数据 Model 类型定义  *******************
 *  *************************************************************
 */

import { iDocIDType } from '../document.baseType';

/**
 * @description: 大模型数据集合
 * <key> 文档🆔
 * <value> 大模型数据
 */
export type iDocBigModelDataMap = Record<
  iDocIDType,
  Readonly<iDocBigModelData>
>;

/**
 * 大模型数据类型定义
 */
export interface iDocBigModelData
  extends iDocBigModelDataBase,
    iDocBigModelDataLoading,
    iDocBigModelDataStatus {}

export interface iDocBigModelDataBase {
  keyWordInfo?: iKeyWordInfo[]; // 关键词信息
  fullTextSummary?: iFullTextSummary; // 全文概要
  chapterOverview?: iChapterOverview[]; // 章节速览
  chapterOverviewFeedback: iFeedbackDetail[]; // 章节速览评价信息列表
  speakSummary?: iSpeakSummary[]; // 发言总结
  keyPointsReview?: iKeyPointsReview[]; // 要点、问答回顾
  pptTitleInfos?: iPPTTitleInfo[]; // PPT标题信息
  pptSummaryInfos?: iPPTSummaryInfos; // PPT概要信息
  bigModelSceneInfo?: iBigModelSceneInfo; // 场景信息
  bigModelXmindInfo?: iBigModelXmindInfo; // 脑图信息
  todoList?: iTodo[]; // 待办事项
  customPrompt?: any; // 自定义Prompt
  allLoadFinished?: boolean;
}

/**
 * 大模型相关数据加载中标识
 */
export interface iDocBigModelDataLoading {
  labInfoQueued?: boolean; // 是否处于排队中
  normalLabInfoLoading?: boolean; // 常规大模型数据加载中标识
  pptLoading?: boolean; // PPT信息加载中标识
  pptSummaryLoading?: boolean; // PPT摘要加载中标识
  allLoadFinished?: boolean; // 所有都加载完成
}

/**
 * 大模型数据状态
 */
export interface iDocBigModelDataStatus {
  keyWordInfoStatus?: eLabInfoStatus; // 关键词信息加载状态
  fullTextSummaryStatus?: eLabInfoStatus; // 全文概要加载状态
  chapterOverviewStatus?: eLabInfoStatus; // 章节速览加载状态
  speakSummaryStatus?: eLabInfoStatus; // 发言总结加载状态
  keyPointsReviewStatus?: eLabInfoStatus; // 要点回顾加载状态
  todoListStatus?: eLabInfoStatus; // 待办事项加载状态
  bigModelSceneInfoStatus?: eLabInfoStatus; // 场景信息加载状态
  pptTitleInfosStatus?: eLabInfoStatus; // PPT标题信息加载状态
  bigModelXmindInfoStatus?: eLabInfoStatus; // PPT脑图信息加载状态
  pptSummaryInfosStatus?: ePPTStatus; // PPT概要信息加载状态
}

/**
 * @description: 关键词信息类型定义
 */
export interface iKeyWordInfo {
  id: number; // 关键词ID
  value: string; // 关键词
}

/**
 * @description: 全文概要类型定义
 */
export interface iFullTextSummary {
  id: number; // 全文概要ID
  value: string; // 全文概要
}

/**
 * @description: 章节速览类型定义
 */
export interface iChapterOverview {
  id: number; // 章节速览ID
  title: string; // 章节速览标题
  content: string; // 章节速览内容
  time: number; // 章节速览开始时间戳
  endTime: number; // 章节速览结束时间戳
  extensions: Array<{ startTime: number; endTime: number; sId: number }>;
  summary?: string; // 实时使用这个数据
}

/**
 * @description: 评价信息类型定义
 */
export interface iFeedbackDetail {
  id: number;
  userId: number;
  transId: number;
  labShowKey: string;
  bindId: number;
  feedbackChoice: number;
  feedbackInfo: string; // 注意📢 次此处是json 字符串
}

/**
 * @description: 发言总结类型定义
 */
export interface iSpeakSummary {
  id: number; // 发言总结ID
  value: string; // 发言总结
  speaker: string; // 发言人🆔
}

/**
 * @description: 要点回顾类型定义
 */
export interface iKeyPointsReview {
  id: number; // 要点回顾ID
  title: string; // 要点回顾标题
  value: string; // 要点回顾内容
  extensions: Array<{
    sentenceIdsOfAnswer: number[];
    sentenceIdsOfQuestion: number[];
    sentenceInfoOfAnswer: iSentenceInfo[];
    sentenceInfoOfQuestion: iSentenceInfo[];
  }>;
}

/**
 * @description: 句子信息类型定义
 */
export interface iSentenceInfo {
  beginTime: number; // 开始时间
  endTime: number; // 结束时间
  paragraphId: string; // 段落ID
  sentenceId: number; // 句子ID
  ui: string; // 发言人🆔
}

/**
 * @description: PPT标题信息类型定义
 */
export type iPPTTitleInfo = Omit<iChapterOverview, 'content'>;

/**
 * @description: PPT信息类型定义
 */
export interface iPPTSummaryInfos {
  modelType: 'simple' | 'detail';
  hasPic: boolean;
  contents: iPPTInfoContent[];
}

export interface iPPTInfoContent {
  id: number; // PPT 🆔 标识
  beginTime: number; // PPT开始时间(时间戳)
  isFailed: boolean; // 是否失败
  isLoadingContent: boolean;
  content?: string[]; // PPT摘要
  isTextTooShort?: boolean; // 是否文本太短
  index: number; // 索引标识
  pictureKey: string; // 高清图像 oss 在线链接
  pictureOssLink: string; // 常规图 oss 在线链接
  pictureThumbnailOssLink: string; // 缩略图 oss 在线链接
}

/** 场景信息 */
export interface iBigModelSceneInfo {
  value: string;
  labShowKeys: Array<
    | 'keyWordsExtractor' // 关键词
    | 'fullSummary' // 全文概要
    | 'titleGenerator' // 章节速览
    | 'agendaSummary' // 章节速览摘要
    | 'roleSummary' // 发言总结摘要
    | 'pptExtractor' // PPT提取
    | 'pptSummary' // PPT摘要
    | 'qaReview' // 问答回顾
    | 'qaKeyPoint' // 要点回顾
    | 'actionExtractor' // 待办事项
  >;
}

/** 脑图信息 */

export interface iBigModelXmindInfoChildren {
  content: string;
  children?: iBigModelXmindInfoChildren[];
}
export interface iBigModelXmindInfo {
  id: number;
  json: iBigModelXmindInfoChildren;
}

/** 待办事项 */
export interface iTodo {
  id: number; // 🆔 标识
  time: number; // 时间戳
  value: string; // 待办内容
}

// 常规大模型信息状态枚举（或者说目前除了PPT 信息外）
export enum eLabInfoStatus {
  FINISH, // 已完成
  LOADING, // 进行中
  FAILED, // 错误
  TEXT_TOO_SHORT, // 文本太短
}

// ppt 状态枚举
export enum ePPTStatus {
  FINISH = 0, // 已完成
  LOADING = 1, // 进行中
  FAILED = 2, // 错误
  TEXT_TOO_SHORT = 3, // 文本太短
  NONE_PPT = 4, // 无ppt
  SUMMARY_PROCESSING = 5, //  ppt 摘要处理中
  SUMMARY_FAILED = 7, //  ppt 摘要处理失败
  UNKNOWN = 8, //  ppt 未知错误，失败
}
