import React, { memo, useCallback, useMemo, useState } from "react";
import { StarRating } from "../starRating";
import { Summary } from "../../spheres/agendaSummary";
import {
  Wrapper,
  Content,
  TopBar,
  Time,
  Line,
  LineOfTop,
  LineOfBottom,
  LineOfDot,
  Agenda,
  AgendaHeading,
  AgendaSummary,
} from "./styled";
import { getFlexibleTime } from "@tingwu/common";

export interface AgendaSummaryCardProps {
  summary: Summary;
  firstSummary?: boolean;
  lastSummary?: boolean;
  readonly?: boolean;
  star?: number;
  onRate?: (star: number) => void;
  isThreeUnitTime?: boolean;
  displayTimeWidth?: number;
}

export const AgendaSummaryCard: React.FC<AgendaSummaryCardProps> = memo(props => {
  const {
    summary,
    firstSummary = false,
    lastSummary = false,
    readonly = false,
    star,
    onRate,
    isThreeUnitTime = false,
    displayTimeWidth = 68,
  } = props;
  const [isMouseOnCard, setIsMouseOnCard] = useState(false);

  const handleMouseEnter = useCallback(() => {
    setIsMouseOnCard(true);
  }, []);

  const handleMouseLeave = useCallback(() => {
    setIsMouseOnCard(false);
  }, []);

  const starRatingStyle = useMemo(() => {
    return {
      opacity: isMouseOnCard ? 1 : 0,
      transition: "all 0.25s",
    };
  }, [isMouseOnCard]);

  const timeFormat = useMemo(() => {
    return getFlexibleTime(summary.beginTime);
  }, [summary]);

  return (
    <Wrapper onMouseEnter={handleMouseEnter} onMouseLeave={handleMouseLeave}>
      <TopBar>
        {!readonly && <StarRating style={starRatingStyle} value={star} onChange={onRate} />}
      </TopBar>
      <Content displayTimeWidth={displayTimeWidth}>
        <Time isThreeUnitTime={isThreeUnitTime}>{timeFormat}</Time>
        <Line>
          <LineOfTop visible={!firstSummary} />
          <LineOfDot />
          {lastSummary ? null : (
            // eslint-disable-next-line react/jsx-indent
            <LineOfBottom>
              <img
                style={{
                  display: "inline-block",
                  width: 1,
                  height: 803,
                }}
                src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAMoCAYAAAAOXorqAAAAAXNSR0IArs4c6QAAAERlWElmTU0AKgAAAAgAAYdpAAQAAAABAAAAGgAAAAAAA6ABAAMAAAABAAEAAKACAAQAAAABAAAAAaADAAQAAAABAAADKAAAAACzZ6IlAAAAQUlEQVRIDWNIjH3Ww4AAo9zR0ICnhtHEMJoYRhMDJARG88JoXhjNC6N5ARgCo0XBaFEwWhSMFgWjRcFoSThYSkIA+w5Ov8fAuV8AAAAASUVORK5CYII="
              />
            </LineOfBottom>
          )}
          {/* <LineOfBottom visible={!lastSummary} /> */}
        </Line>
        <Agenda className="agenda">
          <AgendaHeading>{summary.agendaHeading}</AgendaHeading>
          <AgendaSummary>{summary.content}</AgendaSummary>
        </Agenda>
      </Content>
    </Wrapper>
  );
});
