// Mock axios before importing the module
const mockAxiosInstance = {
  get: jest.fn(),
  post: jest.fn(),
};

jest.mock('axios', () => {
  const actualAxios = jest.requireActual('axios');
  return {
    ...actualAxios,
    default: {
      ...actualAxios.default,
      create: jest.fn(() => mockAxiosInstance),
    },
    create: jest.fn(() => mockAxiosInstance),
  };
});

import request, { getResultRequest, websocketUrl, API_KEY, APP_ID } from '../service/request';

describe('@tingwu/service - request', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('request.get', () => {
    it('should make GET request successfully', async () => {
      const mockData = { result: 'success' };
      mockAxiosInstance.get.mockResolvedValue({ data: mockData });

      const result = await request.get({ param: 'value' });
      expect(result).toEqual(mockData);
      expect(mockAxiosInstance.get).toHaveBeenCalledWith('/api/services/aigc/multimodal-generation/generation', { params: { param: 'value' } });
    });

    it('should handle GET request error', async () => {
      const mockError = new Error('Network error');
      mockAxiosInstance.get.mockRejectedValue(mockError);

      // request.get 会捕获错误并返回，不会抛出
      const result = await request.get({ param: 'value' });
      expect(result).toBeInstanceOf(Error);
      expect((result as Error).message).toBe('Network error');
    });
  });

  describe('request.post', () => {
    it('should make POST request successfully', async () => {
      const mockData = { result: 'success' };
      mockAxiosInstance.post.mockResolvedValue({ data: mockData });

      const result = await request.post({ key: 'value' });
      expect(result).toEqual(mockData);
      expect(mockAxiosInstance.post).toHaveBeenCalledWith('/api/services/aigc/multimodal-generation/generation', { key: 'value' });
    });

    it('should make POST request with custom URL', async () => {
      const mockData = { result: 'success' };
      const customUrl = '/custom/endpoint';
      mockAxiosInstance.post.mockResolvedValue({ data: mockData });

      const result = await request.post({ key: 'value' }, customUrl);
      expect(result).toEqual(mockData);
      expect(mockAxiosInstance.post).toHaveBeenCalledWith(customUrl, { key: 'value' });
    });

    it('should handle POST request error', async () => {
      const mockError = new Error('Network error');
      mockAxiosInstance.post.mockRejectedValue(mockError);

      // request.post 会捕获错误并返回，不会抛出
      const result = await request.post({ key: 'value' });
      expect(result).toBeInstanceOf(Error);
      expect((result as Error).message).toBe('Network error');
    });
  });

  describe('getResultRequest', () => {
    it('should make GET request to provided URL', async () => {
      const mockData = { result: 'success' };
      const url = 'https://example.com/api/data';
      mockAxiosInstance.get.mockResolvedValue({ data: mockData });

      const result = await getResultRequest(url);
      expect(result).toEqual(mockData);
      expect(mockAxiosInstance.get).toHaveBeenCalledWith(url);
    });

    it('should handle error in getResultRequest', async () => {
      const mockError = new Error('Request failed');
      const url = 'https://example.com/api/data';
      mockAxiosInstance.get.mockRejectedValue(mockError);

      const result = await getResultRequest(url);
      expect(result).toEqual(mockError);
    });
  });

  describe('constants', () => {
    it('should export websocketUrl', () => {
      expect(websocketUrl).toBe('wss://dashscope.aliyuncs.com/api-ws/v1/inference');
    });

    it('should export API_KEY and APP_ID', () => {
      expect(typeof API_KEY).toBe('string');
      expect(typeof APP_ID).toBe('string');
    });
  });
});

