import { SID } from "../persist";
import { AbstractCmdController } from "./abstractCmdController";

export interface TodoListCheckpoint {
  id: number;
  value: string;
  time?: number;
  endTime?: number;
  extensions?: Array<{
    sentenceId: any;
    startTime: number;
    endTime: number;
    // sId: number;
  }>;
}

export interface TodoList {
  id: string;
  title: string;
  beginTime: number;
  endTime: number;
  highlightContents: Array<{
    beginTime: number;
    endTime: number;
    sid: SID;
  }>;
}

export class TodoListController extends AbstractCmdController {
  private todoLists: TodoList[] = [];

  private activetodoListIndex: number | undefined;

  private enableActivetodoList = false;

  getEnableActivetodoList() {
    return this.enableActivetodoList;
  }

  settodoLists(todoLists: TodoList[]) {
    this.todoLists = todoLists;
    this.controller.emit("todoListChange", { todoLists });
  }

  gettodoLists() {
    return this.todoLists;
  }

  fromJSON(todoListCheckpoints: TodoListCheckpoint[]) {
    if (!todoListCheckpoints || todoListCheckpoints.length === 0) {
      this.settodoLists([]);
    }

    const todoLists: TodoList[] = todoListCheckpoints.map(cp => {
      return {
        id: String(cp.id),
        title: cp.value,
        beginTime: cp.time || 0,
        endTime: cp.endTime || 0,
        highlightContents: (cp.extensions || []).map(ext => {
          return {
            beginTime: ext.startTime,
            endTime: ext.endTime,
            sid: String(ext.sentenceId),
          };
        }),
      };
    });

    this.settodoLists(todoLists);
    this.activetodoListIndex = undefined;
  }

  getActivetodoList() {
    if (this.activetodoListIndex == null || !this.todoLists) {
      return undefined;
    }
    return this.todoLists[this.activetodoListIndex];
  }

  setActivetodoListById(id: string) {
    if (!this.todoLists) {
      return;
    }
    const { length } = this.todoLists;
    for (let i = 0; i < length; i++) {
      const todoList = this.todoLists[i];
      if (todoList.id === id) {
        if (this.activetodoListIndex === i) {
          return;
        }
        this.enableActivetodoList = true;
        this.activetodoListIndex = i;
        this.controller.emit("activetodoListChange", {
          todoList: this.todoLists[i],
        });
        return;
      }
    }
    this.clearActivetodoList();
  }

  clearActivetodoList() {
    this.enableActivetodoList = false;
    this.activetodoListIndex = undefined;

    this.controller.emit("activetodoListChange", {
      todoList: undefined,
    });
  }

  setActivetodoListByTime(timestamp: number) {
    if (!this.todoLists) {
      return;
    }
    this.todoLists.some(todoList => {
      if (todoList) {
        if (timestamp >= todoList.beginTime && timestamp <= todoList.endTime) {
          this.setActivetodoListById(todoList.id);
          return true;
        }
      }
      return false;
    });
  }

  getActivetodoListHighlightSids() {
    if (
      this.activetodoListIndex == null ||
      !this.todoLists ||
      !this.todoLists[this.activetodoListIndex]
    ) {
      return [];
    }

    return this.todoLists[this.activetodoListIndex].highlightContents.map(c => c.sid);
  }

  reset() {
    this.todoLists = [];
    this.activetodoListIndex = undefined;
    this.enableActivetodoList = false;
  }
}
