import { PID, UID, Word } from "../persist";
import { AbstractCmdController } from "./abstractCmdController";

export interface ExtractContent {
  pid: PID;
  beginTime: number;
  uid: UID;
  words: Word[];
  textContent: string;
}

export class ExtractController extends AbstractCmdController {
  getAllExtractContent() {
    const { controller } = this;
    const extraContents: ExtractContent[] = [];
    controller.getPids().forEach(pid => {
      const words = controller.getWords(pid);
      const textContent = controller.getTextContent(pid);
      const paragraph = controller.getParagraph(pid);
      if (words.length > 0) {
        extraContents.push({
          pid,
          beginTime: words[0].beginTime,
          uid: paragraph.uid!,
          words,
          textContent,
        });
      }
    });
    return extraContents;
  }

  getExtraContentByTag(tag?: string) {
    const { controller } = this;
    const extraContents: ExtractContent[] = [];
    controller.getPids().forEach(pid => {
      const paragraph = controller.getParagraph(pid);
      const words = controller.getWords(pid);
      let foundTag: string | undefined;

      words.forEach(word => {
        if (!foundTag || word.tag !== foundTag) {
          if (word.tag && (tag == null || word.tag === tag)) {
            foundTag = word.tag;

            extraContents.push({
              pid,
              beginTime: word.beginTime,
              uid: paragraph.uid!,
              words: [],
              textContent: "",
            });
          } else {
            foundTag = undefined;
          }
        }
        if (foundTag && word.tag === foundTag) {
          const lastExtraContent = extraContents[extraContents.length - 1];
          if (lastExtraContent) {
            lastExtraContent.words.push(word);
            lastExtraContent.textContent = lastExtraContent.words.map(w => w.text).join("");
          }
        }
      });
    });
    return extraContents;
  }

  // 获取全部的翻译数据
  getAllExtractContentByTranslate() {
    const { controller } = this;
    const extraContents: ExtractContent[] = [];
    controller.getPids().forEach(pid => {
      const { translateResult: words } = controller.getTranslate(pid) || {};
      if (words && words.length) {
        const paragraph = controller.getParagraph(pid);
        extraContents.push({
          pid,
          beginTime: words[0].beginTime,
          uid: paragraph.uid!,
          words,
          textContent: words.map(({ text }) => text).join(""),
        });
      }
    });
    return extraContents;
  }

  // 获取全部的改写数据
  getAllExtractContentByTextPolish() {
    const { controller } = this;
    const extraContents: ExtractContent[] = [];
    controller.getPids().forEach(pid => {
      const { textPolishResult } = controller.getTextPolish(pid) || {};
      const paragraph = controller.getParagraph(pid);
      extraContents.push({
        pid,
        beginTime: textPolishResult.et,
        uid: paragraph.uid!,
        words: [],
        textContent: textPolishResult.formal_paragraph,
      });
    });
    return extraContents;
  }
}
