import { TYIcon } from "@tingwu/design";
import classnames from "classnames";
import React, { useEffect, useRef, useState } from "react";
import { TWXmind } from "../tongyi-xmind";
import MyTooltip from "./MyTooltip";
import { XMindStyleText, XMindWrapper } from "./styled";

interface XMindProps {
  xmindData: string; // 思维导图数据。 json 字符, json 对象 均可
  height: number | string; // 思维导图 高度
  maxWidth?: number; // 思维导图 节点 文本最大宽度： 建议 中文 300  英文 600， 默认 300
  // Feedback?: () => JSX.Element; // 反馈
  exportFile?: (arg0: string) => { success: boolean; errorMsg?: string }; // 导出文件回调
  copyImg?: {
    // 复制图片
    url: string;
    onCopySuccess?: () => void;
  };
  spacingVertical?: number; // 行间距
  isAnimation?: boolean; // 初始化开启动画，后续是否开启动画
  initialExpandLevel?: number; // 默认展开几级
  isResize?: boolean; // 全屏查看或者恢复正常 时候 是否需要自适应画布； 默认 需要自适应
  showAllNodes?: boolean; // 全屏时候是否展示所有节点
  isBottom?: boolean;
}

const XMind: React.FC<XMindProps> = props => {
  const {
    showAllNodes = true,
    isResize = true,
    initialExpandLevel = 4,
    xmindData,
    height,
    isBottom = true,
    maxWidth = 300,
    spacingVertical = 10,
    isAnimation = true,
  } = props;
  const [isFullscreen, setIsFullscreen] = useState<boolean>(false);
  const twXmindRef = useRef<TWXmind>();
  const [oneShowAllNodes, setOneShowAllNodes] = useState<boolean>(true);

  useEffect(() => {
    const wrapper = document.querySelector(".wrapperXmind") as HTMLElement;
    if (!wrapper) return;
    if (wrapper.querySelector(".markmap")) {
      // 是否已存在 脑图
      wrapper.innerHTML = "";
    }
    const xmindInst = new TWXmind({
      wrapper,
      xmindOptions: {
        data: xmindData, // 获取数据
        dataType: "json",
        initialExpandLevel,
        spacingVertical,
        maxWidth,
        colorList: ["#605CE5"],
        isAnimation,
        style: () => XMindStyleText,
      },
    });
    twXmindRef.current = xmindInst;
  }, [xmindData]);

  const resetSize = () => twXmindRef.current && twXmindRef.current.resetSize();
  /** 一键展开 / 收起 */
  const oneNodes = () => {
    if (!twXmindRef.current) return;
    if (oneShowAllNodes) {
      twXmindRef.current.expendAllNodes(true);
    } else {
      twXmindRef.current.collapseNodes(true);
    }
    setOneShowAllNodes(!oneShowAllNodes);
  };
  return (
    <XMindWrapper isBottom={isBottom}>
      <div
        id="twXmindWrapper"
        style={{
          height: isFullscreen ? "100vh" : height,
        }}
        className={isFullscreen ? "fullscreenContent" : "mindMapContent"}
      >
        <div
          className="wrapperXmind"
          // ref={wrapperRef}
          style={{ height: isFullscreen ? "100vh" : "100%" }}
        />
        <div className="operateContent">
          <div className={classnames("wrapCom pad68", isFullscreen ? "fullStyle" : "")}>
            <MyTooltip title="适应画布" type="icon-zishiying-layout-scale" fn={resetSize} />
            <div className="line" />
            <MyTooltip
              title={oneShowAllNodes ? "一键全部展开" : "恢复默认"}
              type={oneShowAllNodes ? "icon-indent-increase" : "icon-indent-decrease"}
              fn={oneNodes}
            />
            <div className="line" />
            <MyTooltip
              title="缩小"
              type="icon-dash"
              fn={() => twXmindRef.current && twXmindRef.current.updateZoom(0.8)}
            />
            <MyTooltip
              title="放大"
              type="icon-jia-plus-line"
              fn={() => twXmindRef.current && twXmindRef.current.updateZoom(1.25)}
            />
          </div>
          {!isFullscreen ? (
            <div className="fullScreen wrapCom pad66">
              <MyTooltip
                title="全屏查看"
                type="icon-enlarge-line"
                fn={() => {
                  showAllNodes && twXmindRef.current && twXmindRef.current.expendAllNodes();
                  setTimeout(() => isResize && resetSize(), 10);
                  setIsFullscreen(true);
                }}
              />
            </div>
          ) : null}
        </div>
        {isFullscreen ? (
          <div className="fullScreenWrap">
            <div
              className="btnIcon"
              onClick={() => {
                setTimeout(() => isResize && resetSize(), 10);
                setIsFullscreen(false);
                showAllNodes && twXmindRef.current && twXmindRef.current.collapseNodes();
              }}
            >
              <TYIcon type="icon-x-01" className="operateItem" />
            </div>
          </div>
        ) : null}
      </div>
    </XMindWrapper>
  );
};

export { XMind };
