import { useEffect, useState } from "react";
import { eMicrophonePermissionState, MicrophonePermissionHookReturn } from "./index.types";

/**
 * 自定义hook，用于检测和请求麦克风权限
 */
export const useMicrophonePermission = (): MicrophonePermissionHookReturn => {
  const [permission, setPermission] = useState<eMicrophonePermissionState>(
    eMicrophonePermissionState.UNKNOWN
  );
  const [isRequesting, setIsRequesting] = useState(false);
  const [error, setError] = useState<string | null>(null);

  /**
   * 检测当前麦克风权限状态
   */
  const checkPermission = async (): Promise<eMicrophonePermissionState> => {
    try {
      // 检查浏览器是否支持Permissions API
      if (!navigator.permissions) {
        setPermission(eMicrophonePermissionState.UNSUPPORTED);
        return eMicrophonePermissionState.UNSUPPORTED;
      }

      // 尝试使用Permissions API查询麦克风权限
      const permissionObj = await navigator.permissions.query({
        name: "microphone" as PermissionName,
      });

      // 设置当前权限状态
      const state = permissionObj.state as eMicrophonePermissionState;
      setPermission(state);

      // 监听权限状态变化
      permissionObj.onchange = () => {
        setPermission(permissionObj.state as eMicrophonePermissionState);
      };

      return state;
    } catch (err) {
      // 在某些浏览器中Permissions API可能不支持'microphone'参数
      // 这种情况下我们返回'unknown'，需要通过实际请求权限来确定状态
      setPermission(eMicrophonePermissionState.UNKNOWN);
      return eMicrophonePermissionState.UNKNOWN;
    }
  };

  /**
   * 请求麦克风权限
   */
  const requestPermission = async (): Promise<eMicrophonePermissionState> => {
    setIsRequesting(true);
    setError(null);

    try {
      // 检查浏览器是否支持mediaDevices API
      if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
        throw new Error("浏览器不支持麦克风访问");
      }

      // 尝试请求麦克风权限
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true });

      // 成功获取权限后，关闭所有轨道以避免持续占用麦克风
      stream.getTracks().forEach(track => track.stop());

      // 更新权限状态为已授予
      setPermission(eMicrophonePermissionState.GRANTED);
      setIsRequesting(false);

      return eMicrophonePermissionState.GRANTED;
    } catch (err) {
      setIsRequesting(false);

      // 处理不同类型的错误
      if (err instanceof Error) {
        setError(err.message);

        // 根据错误类型判断权限状态
        switch (err.name) {
          case "NotAllowedError":
            // 用户拒绝了权限请求
            setPermission(eMicrophonePermissionState.DENIED);
            return eMicrophonePermissionState.DENIED;
          case "NotFoundError":
            // 未找到麦克风设备
            setError("未检测到麦克风设备");
            setPermission(eMicrophonePermissionState.DENIED);
            return eMicrophonePermissionState.DENIED;
          case "NotReadableError":
            // 麦克风设备被占用
            setError("麦克风设备被占用，请稍后重试");
            setPermission(eMicrophonePermissionState.DENIED);
            return eMicrophonePermissionState.DENIED;
          default:
            // 其他错误情况
            setError(`获取麦克风权限失败: ${err.message}`);
            setPermission(eMicrophonePermissionState.DENIED);
            return eMicrophonePermissionState.DENIED;
        }
      }

      // 处理非Error对象的错误情况
      const errorMessage = String(err);
      setError(errorMessage);
      setPermission(eMicrophonePermissionState.DENIED);
      return eMicrophonePermissionState.DENIED;
    }
  };

  /**
   * 初始化时检测权限状态
   */
  useEffect(() => {
    checkPermission();
  }, []);

  return {
    permission,
    isRequesting,
    error,
    checkPermission,
    requestPermission,
  };
};
