/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.aliyun.lindorm.ldspark.examples.thrift

import java.sql.DriverManager
import java.util.UUID

import scala.collection.JavaConverters._

import org.apache.hive.service.cli.thrift.ThriftCLIService
import org.apache.spark.SparkConf
import org.apache.spark.sql.SparkSession
import org.apache.spark.sql.hive.thriftserver.HiveThriftServer2

/**
 * Shows how to incorperate an embedded mini Spark thrift-server inside a project.
 * Please reference pom.xml for Spark and Iceberg dependency.
 */
object EmbeddedMiniThriftSqlServer {

  def main(args:Array[String]): Unit = {
    val conf = new SparkConf()

    // Create a local Spark cluster with 2 cores
    conf.setMaster("local[2]")

    // Define a catalog -- 'lindorm_columnar',
    // which is based on file system bye path of '$PWD/warehouse'
    conf.set("spark.sql.catalog.lindorm_columnar", "org.apache.iceberg.spark.SparkCatalog")
    conf.set("spark.sql.catalog.lindorm_columnar.type", "hadoop")
    conf.set("spark.sql.catalog.lindorm_columnar.warehouse", "$PWD/warehouse")

    val spark = SparkSession.builder()
      .config(conf)
      .appName("my-thrift")
      .getOrCreate()
    spark.sparkContext.setLogLevel("FATAL")

    val thriftServer = HiveThriftServer2.startWithContext(spark.sqlContext)
    val port = thriftServer.getServices.asScala.find(_.isInstanceOf[ThriftCLIService])
      .orElse(throw new Exception("Thrift server failed."))
      .get.asInstanceOf[ThriftCLIService].getPortNumber

    val jdbcUri = s"jdbc:hive2://localhost:$port/"
    val conn = DriverManager.getConnection(jdbcUri, "root", "root")
    val stmt = conn.createStatement()
    val dbName = UUID.randomUUID().toString.replace("-", "") + "_db"

    // Create a tmp database
    stmt.execute(
      s"""
         |create database lindorm_columnar.${dbName}
         |""".stripMargin)

    // Create a tmp table
    stmt.execute(
      s"""
         |create table lindorm_columnar.${dbName}.mytbl (id int, name string) using iceberg
         |""".stripMargin)

    // Insert values
    stmt.execute(
      s"""
         |insert into lindorm_columnar.${dbName}.mytbl values
         |(0, 'zhang3')
         |""".stripMargin)

    // Query from tmp table
    val resultSet = stmt.executeQuery(
      s"""
         |select * from lindorm_columnar.mydb.mytbl
         |""".stripMargin)
    if (resultSet.next()) {
      println(resultSet.getInt(1))
      println(resultSet.getString(2))
    }

    // Cleanup on file system.
    stmt.execute(
      s"""
         |drop table lindorm_columnar.${dbName}.mytbl
         |""".stripMargin)
    stmt.execute(
      s"""
         |drop database lindorm_columnar.${dbName}
         |""".stripMargin)

    // Good-bye
    stmt.close()
    conn.close()
    thriftServer.stop()
  }
}
