#include <pthread.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include <fstream>
#include <iostream>

#include "conversation.h"
#include "conversation_utils.h"
#include "json/json.h"

using namespace convsdk;

#define OPERATION_TIMEOUT_S 2
#define SEND_AUDIO_FILE_S 30

// 自定义事件回调参数
struct ParamCallBack {
 public:
  explicit ParamCallBack() {
    memset(userInfo, 0, 8);
    pthread_mutex_init(&mtxWord, NULL);
    pthread_cond_init(&cvWord, NULL);

    conversation = NULL;
    task_id = "";
    asr_binary_index = 0;
    binary_index = 0;
    binary_bytes = 0;
    listening = false;
    exit = false;
    bidirectional_speech = false;
    thread_id = 0;
    cur_event_type = 0;
  };
  ~ParamCallBack() {
    pthread_mutex_destroy(&mtxWord);
    pthread_cond_destroy(&cvWord);
  };

  char userInfo[8];
  pthread_mutex_t mtxWord;
  pthread_cond_t cvWord;

  struct timeval connectTv;
  struct timeval disconnectTv;

  Conversation* conversation;
  std::string task_id;

  int asr_binary_index; /* 此轮asr发送帧索引号 */

  int binary_index;            /* 此轮接收到的tts帧索引号 */
  size_t binary_bytes;         /* 此轮接收到的tts字节数 */
  struct timeval output_start; /* output_start时间戳 */

  bool listening;
  bool exit;
  bool bidirectional_speech;
  pthread_t thread_id;
  int cur_event_type;
};

std::string g_log_level = "verbose"; /* version, debug, info, warn, error */
std::string g_mode = "duplex"; /* tap2talk, push2talk, duplex, kws_duplex */
std::string g_debug_mode = "normal";
std::string g_url = "wss://dashscope.aliyuncs.com/api-ws/v1/inference";
std::string g_apikey = "";

int g_in_frame_len =
    -1; /* 发送数据的帧长，单位样本数，比如默认320表示320样本即640字节 */
int g_sr = -1;                           /* upstream.sample_rate */
std::string g_format = "opus";           /* upstream.audio_format */
std::string g_downstream_format = "pcm"; /* downstream.audio_format */
int g_downstream_sr = 24000;             /* downstream.sample_rate */
int g_downstream_bit_rate = 0;           /* downstream.sample_rate */
std::string g_voice = "longxiaochun_v2"; /* downstream.voice */
int g_frame_size = -1;                   /* downstream.frame_size */
int g_transmit_rate_limit = -1;          /* downstream.transmit_rate_limit */
bool g_enable_external_aec_module =
    false; /* acoustic_echo_cancelling_attributes.enable_external_aec_module */
bool g_enable_external_vad_module =
    false; /* voice_activity_detection_attributes.enable_external_vad_module */
bool g_enable_vad_module =
    true; /* voice_activity_detection_attributes.enable_vad_module */

std::string g_app_id = "empty_app_id";
std::string g_workspace_id = "empty_workspace_id";
std::string g_model = "multimodal-dialog";
std::string g_workspace = "";
std::string g_debug_path = "";
std::string g_output_path = "";
std::string g_input_path = "";
std::string g_file_path = "";
std::string g_jpeg_file_path = "";
std::string g_task_id = "";
bool g_simple_log = false;

int g_msg_flag = 0;
int g_threads = 1;
bool g_exit = false;
ConversationCallbackMethod on_message_callback = NULL;
EventTrackCallbackMethod on_et_callback = NULL;

void signal_handler_int(int signo) {
  std::cout << "\nget interrupt mesg\n" << std::endl;
  g_exit = true;
}
void signal_handler_quit(int signo) {
  std::cout << "\nget quit mesg\n" << std::endl;
  g_exit = true;
}

int respond_jpeg_data(std::string path, Conversation* conversation) {
  if (path.empty()) {
    std::cout << "jpeg path is empty" << std::endl;
    return -1;
  }

  ConversationUtils utils;
  std::string base64_content = utils.Base64EncodeFromFilePath(path);
  std::cout << "base64_content has " << base64_content.length() << "bytes"
            << std::endl;
  if (!base64_content.empty()) {
    Json::Value root;
    root["type"] = "prompt";
    root["text"] = "看看图片里有什么？";
    Json::Value parameters;
    parameters["biz_params"] = Json::objectValue;

    Json::Value client_info;
    Json::Value status;
    Json::Value bluetooth_announcement;
    bluetooth_announcement["status"] = "stopped";
    status["bluetooth_announcement"] = bluetooth_announcement;
    client_info["status"] = status;
    parameters["client_info"] = client_info;

    Json::Value image;
    image["type"] = "base64";
    image["value"] = base64_content;
    image["width"] = 480;
    image["height"] = 720;
    Json::Value images(Json::arrayValue);
    images.append(image);
    parameters["images"] = images;
    root["parameters"] = parameters;

    Json::StreamWriterBuilder writer;
    writer["indentation"] = "";
    conversation->SendResponseData(Json::writeString(writer, root).c_str());
  }

  return 0;
}

int send_jpeg_data(std::string path, Conversation* conversation) {
  if (path.empty()) {
    std::cout << "jpeg path is empty" << std::endl;
    return -1;
  }

  ConversationUtils utils;
  std::string base64_content = utils.Base64EncodeFromFilePath(path);
  if (!base64_content.empty()) {
    Json::Value root;
    root["type"] = "update_info";
    Json::Value parameters;
    parameters["biz_params"] = Json::objectValue;

    Json::Value client_info;
    Json::Value status;
    Json::Value bluetooth_announcement;
    bluetooth_announcement["status"] = "stopped";
    status["bluetooth_announcement"] = bluetooth_announcement;
    client_info["status"] = status;
    parameters["client_info"] = client_info;

    Json::Value image;
    image["type"] = "base64";
    image["value"] = base64_content;
    image["width"] = 480;
    image["height"] = 720;
    Json::Value images(Json::arrayValue);
    images.append(image);
    parameters["images"] = images;
    root["parameters"] = parameters;

    Json::StreamWriterBuilder writer;
    conversation->UpdateMessage(Json::writeString(writer, root).c_str());
  }

  return 0;
}

int update_info_languages(Conversation* conversation) {
  Json::FastWriter writer;
  Json::Value root;
  Json::Value new_agent_chat;
  new_agent_chat["language_type"] = "zn";        /* 以文档为准 */
  new_agent_chat["answer_language_type"] = "ja"; /* 以文档为准 */
  root["SendSpeech_agent_chat"] = new_agent_chat;
  root["type"] = "update_nls_parameters";
  return conversation->UpdateMessage(writer.write(root).c_str());
}

int update_info_example(Conversation* conversation) {
  Json::FastWriter writer;
  Json::Value root, parameters;
  Json::Value biz_params, user_prompt_params, tool_bfad3a10;
  parameters["new_params"] = "hello world";
  tool_bfad3a10["appKey"] = "xxxxxxx#";
  tool_bfad3a10["siteKey"] = "OTT_B2C_V3";
  tool_bfad3a10["userGroup"] =
      "OTT_GROUP_BSTY$TerOut_4$$011$BesTV_LiteI_HBYD_7.6.2007.6";
  tool_bfad3a10["userId"] = "023909999999999";
  user_prompt_params["tool_bfad3a10"] = tool_bfad3a10;
  biz_params["user_prompt_params"] = user_prompt_params;
  parameters["biz_params"] = biz_params;
  root["type"] = "update_info";
  root["parameters"] = parameters;
  return conversation->UpdateMessage(writer.write(root).c_str());
}

int update_custom_params_example(Conversation* conversation) {
  /* 万能参数接口演示
    {
      "nls_rule": {
        "payload": ["SendSpeech_1234567"]
      },
      "nls_params": {
        "SendSpeech_1234567": {
          "test": 987654321
        }
      }
    }
  */

  Json::FastWriter writer;
  Json::Value others;
  Json::Value nls_rule, nls_params, nls_params0;
  Json::Value nls_rule_payload(Json::arrayValue);
  nls_rule_payload.append("SendSpeech_1234567");
  nls_rule["payload"] = nls_rule_payload;
  nls_params0["test"] = 987654321;
  nls_params["SendSpeech_1234567"] = nls_params0;
  others["nls_rule"] = nls_rule;
  others["nls_params"] = nls_params;
  others["type"] = "update_custom_nls_parameters";
  return conversation->UpdateMessage(writer.write(others).c_str());
}

void* asr_thread(void* arg) {
  ParamCallBack* tst = static_cast<ParamCallBack*>(arg);
  while (!tst->exit) {
    if (tst->conversation) {
      if (tst->listening) {
        // start to push audio
        std::ifstream fs;
        fs.open(g_file_path, std::ios::binary | std::ios::in);
        if (fs) {
          if (g_mode == "push2talk" || g_mode == "push_to_talk") {
            // update_info_languages(tst->conversation);
            // update_custom_params_example(tst->conversation);
            // update_info_example(tst->conversation);

            if (!g_jpeg_file_path.empty()) {
              respond_jpeg_data(g_jpeg_file_path, tst->conversation);
              usleep(30 * 1000);
            }

            tst->conversation->SetAction(kStartHumanSpeech);
          }

          // update_nls_parameters setting!
          Json::StreamWriterBuilder extra_info_writer;
          Json::Value extra_info_root, prompt, extra_info;
          extra_info_writer["indentation"] = "";
          extra_info["prompt"] = "123";
          extra_info_root["SendSpeech_extra_info"] =
              Json::writeString(extra_info_writer, extra_info);
          extra_info_root["type"] = "update_nls_parameters";
          tst->conversation->UpdateMessage(
              Json::writeString(extra_info_writer, extra_info_root).c_str());

          // update_custom_nls_parameters setting!
          Json::Value nls_rule, nls_params, extra_info_root3, custom3;
          nls_rule["payload"].append("SendSpeech_extra_info3");
          nls_rule["payload"].append("SendSpeech_custom3");
          nls_params["SendSpeech_extra_info3"] =
              Json::writeString(extra_info_writer, extra_info);
          custom3["test"] = 1;
          nls_params["SendSpeech_custom3"] = custom3;
          extra_info_root3["type"] = "update_custom_nls_parameters";
          extra_info_root3["nls_rule"] = nls_rule;
          extra_info_root3["nls_params"] = nls_params;
          tst->conversation->UpdateMessage(
              Json::writeString(extra_info_writer, extra_info_root3).c_str());

          uint8_t data[3200];
          uint32_t send_count = 0;
          while (!fs.eof() && tst->listening && !tst->exit && !g_exit) {
            memset(data, 0, 3200);
            fs.read((char*)data, sizeof(uint8_t) * 3200);
            size_t nlen = fs.gcount();
            if (nlen == 0) {
              break;
            }
            tst->conversation->SendAudioData(data, 3200);
            usleep(100 * 1000);
            if (send_count++ % 5 == 0) {
              send_jpeg_data(g_jpeg_file_path, tst->conversation);
            }
          }  // while
          fs.close();

          if (g_mode == "push2talk" || g_mode == "push_to_talk") {
            tst->conversation->SetAction(kStopHumanSpeech);
            // wait sentenceEnd
            while (tst->listening && !tst->exit && !g_exit) {
              usleep(10 * 1000);
            }
          } else {
            // wait sentenceEnd
            while (tst->listening && !tst->exit && !g_exit) {
              memset(data, 0, 3200);
              tst->conversation->SendAudioData(data, 3200);
              usleep(100 * 1000);
            }
          }
        }
      } else {
        if (tst->bidirectional_speech) {
          uint8_t data[3200] = {0};
          tst->conversation->SendAudioData(data, 3200);
          usleep(100 * 1000);
        } else {
          usleep(50 * 1000);
        }
      }
    } else {
      // tst->conversation is nullptr
      usleep(50 * 1000);
    }
    if (g_exit) {
      tst->exit = true;
    }
  }  // while
  tst->thread_id = 0;
}

void* raw_asr_thread(void* arg) {
  ParamCallBack* tst = static_cast<ParamCallBack*>(arg);
  while (!tst->exit) {
    if (tst->conversation && tst->listening) {
      // start to push audio
      std::ifstream fs;
      std::string file_path = g_input_path + "/" +
                              std::to_string(tst->asr_binary_index) + "." +
                              g_format;
      fs.open(file_path, std::ios::binary | std::ios::in);
      if (fs) {
        if (g_mode == "push2talk" || g_mode == "push_to_talk") {
          tst->conversation->SetAction(kStartHumanSpeech);
        }

        // update_nls_parameters setting!
        Json::StreamWriterBuilder extra_info_writer;
        Json::Value extra_info_root, prompt, extra_info;
        extra_info_writer["indentation"] = "";
        extra_info["prompt"] = "123";
        extra_info_root["SendSpeech_extra_info"] =
            Json::writeString(extra_info_writer, extra_info);
        extra_info_root["type"] = "update_nls_parameters";
        tst->conversation->UpdateMessage(
            Json::writeString(extra_info_writer, extra_info_root).c_str());

        // update_custom_nls_parameters setting!
        Json::Value nls_rule, nls_params, extra_info_root3, custom3;
        nls_rule["payload"].append("SendSpeech_extra_info3");
        nls_rule["payload"].append("SendSpeech_custom3");
        nls_params["SendSpeech_extra_info3"] =
            Json::writeString(extra_info_writer, extra_info);
        custom3["test"] = 1;
        nls_params["SendSpeech_custom3"] = custom3;
        extra_info_root3["type"] = "update_custom_nls_parameters";
        extra_info_root3["nls_rule"] = nls_rule;
        extra_info_root3["nls_params"] = nls_params;
        tst->conversation->UpdateMessage(
            Json::writeString(extra_info_writer, extra_info_root3).c_str());

        uint8_t data[640];
        memset(data, 0, 640);
        fs.read((char*)data, sizeof(uint8_t) * 640);
        size_t nlen = fs.gcount();
        if (nlen == 0) {
          break;
        }
        tst->conversation->SendAudioData(data, nlen);
        usleep(10 * 1000);

        fs.close();
        tst->asr_binary_index++;

        if (g_mode == "push2talk" || g_mode == "push_to_talk") {
          tst->conversation->SetAction(kStopHumanSpeech);
          // wait sentenceEnd
          while (tst->listening && !tst->exit && !g_exit) {
            usleep(10 * 1000);
          }
        }
      }
    } else {
      usleep(50 * 1000);
    }
    if (g_exit) {
      tst->exit = true;
    }
  }  // while
  tst->thread_id = 0;
}

void* action_thread(void* arg) {
  ParamCallBack* tst = static_cast<ParamCallBack*>(arg);
  if (tst == NULL || tst->conversation == NULL) {
    std::cerr << "conversation is nullptr." << std::endl;
    return NULL;
  }

  int event_type = tst->cur_event_type;
  switch (event_type) {
    case ConvEvent::kDataOutputStarted:
      std::cout << "conversation set kDataOutputStarted ->" << std::endl;
      tst->conversation->SetAction(kPlayerStarted);
      break;
    case ConvEvent::kDataOutputCompleted:
      std::cout << "conversation set kPlayerStopped ->" << std::endl;
      tst->conversation->SetAction(kPlayerStopped);
      break;
  }
}

void onMessage(ConvEvent* event, void* param) {
  ParamCallBack* tmpParam = static_cast<ParamCallBack*>(param);
  pthread_t thread_id = 0;
  ConvEvent::ConvEventType event_type = event->GetMsgType();
  int dialog_state = event->GetDialogStateChanged();
  uint64_t output_duration_ms = 0;
  uint64_t output_bytes_per_s = 0;

  if (event_type != ConvEvent::kSoundLevel &&
      event_type != ConvEvent::kBinary) {
    if (g_simple_log) {
      std::cout << event->GetMsgTypeString() << std::endl;
    } else {
      std::cout << "trigger onMessage -->>\n"
                << "  dialog id: " << event->GetDialogId() << "\n"
                << "  round id: " << event->GetRoundId() << "\n"
                << "  messge type: " << event->GetMsgTypeString() << "\n"
                << "  dialog state: " << dialog_state << "\n"
                << "  response: " << event->GetAllResponse() << std::endl;
    }
  }

  switch (event_type) {
    case ConvEvent::kConversationFailed:
      pthread_mutex_lock(&(tmpParam->mtxWord));
      pthread_cond_signal(&(tmpParam->cvWord));
      pthread_mutex_unlock(&(tmpParam->mtxWord));
      tmpParam->exit = true;
      break;
    case ConvEvent::kConversationInitialized:
      break;
    case ConvEvent::kConversationStarted:
      break;
    case ConvEvent::kConversationCompleted:
      break;
    case ConvEvent::kSentenceBegin:
      break;
    case ConvEvent::kSentenceEnd:
      tmpParam->listening = false;
      break;
    case ConvEvent::kDataOutputStarted:
      tmpParam->binary_bytes = 0;
      gettimeofday(&(tmpParam->output_start), NULL);

      tmpParam->cur_event_type = (int)ConvEvent::kDataOutputStarted;
      pthread_create(&thread_id, NULL, &action_thread, tmpParam);
      pthread_detach(thread_id);
      break;
    case ConvEvent::kDataOutputCompleted:
      struct timeval output_end;
      gettimeofday(&output_end, NULL);
      output_duration_ms = output_end.tv_sec * 1000 +
                           output_end.tv_usec / 1000 -
                           tmpParam->output_start.tv_sec * 1000 -
                           tmpParam->output_start.tv_usec / 1000;
      output_bytes_per_s = tmpParam->binary_bytes * 1000 / output_duration_ms;
      std::cout << " --- Recv " << tmpParam->binary_bytes << "bytes during "
                << output_duration_ms << "ms, and " << output_bytes_per_s
                << "bytes/sec ---" << std::endl;

      tmpParam->cur_event_type = (int)ConvEvent::kDataOutputCompleted;
      pthread_create(&thread_id, NULL, &action_thread, tmpParam);
      pthread_detach(thread_id);
      break;
    case ConvEvent::kBinary:
      if (!g_simple_log) {
        std::cout << " --- Recv " << event->GetBinaryDataSize() << "bytes --- "
                  << std::endl;
      }
      tmpParam->binary_bytes += event->GetBinaryDataSize();
      if (!g_output_path.empty()) {
        std::string task_id_dir = g_output_path + "/" + tmpParam->task_id;
        std::string stream_binary = task_id_dir + "/" +
                                    std::to_string(tmpParam->binary_index) +
                                    "." + g_downstream_format;
        std::string total_binary =
            task_id_dir + "/" + tmpParam->task_id + "." + g_downstream_format;
        std::ofstream stream_binary_file(stream_binary,
                                         std::ios::binary | std::ios::out);
        std::ofstream total_binary_file(total_binary,
                                        std::ios::binary | std::ios::app);
        if (stream_binary_file.is_open()) {
          stream_binary_file.write(
              reinterpret_cast<const char*>(event->GetBinaryData().data()),
              event->GetBinaryData().size());
          stream_binary_file.close();
        }
        if (total_binary_file.is_open()) {
          total_binary_file.write(
              reinterpret_cast<const char*>(event->GetBinaryData().data()),
              event->GetBinaryData().size());
          total_binary_file.close();
        }
        tmpParam->binary_index++;
      }
      break;
    case ConvEvent::kSoundLevel:
      break;
    case ConvEvent::kDialogStateChanged:
      if (dialog_state == ConvDialogState::kDialogListening) {
        tmpParam->listening = true;
        tmpParam->asr_binary_index = 0;
      } else if (dialog_state == ConvDialogState::kDialogIdle) {
        if (g_mode == "push2talk" || g_mode == "push_to_talk") {
          tmpParam->listening = true;
        } else {
          tmpParam->listening = false;
        }
      } else {
        tmpParam->listening = false;
      }
      if (event->GetTaskId() && strlen(event->GetTaskId()) > 0) {
        tmpParam->task_id.assign(event->GetTaskId());

        if (!g_output_path.empty()) {
          std::string task_id_dir = g_output_path + "/" + tmpParam->task_id;
          mkdir(task_id_dir.c_str(), 0755);
          tmpParam->binary_index = 0;
        }
      }
      break;
    case ConvEvent::kInterruptAccepted:
      break;
    case ConvEvent::kInterruptDenied:
      break;
    case ConvEvent::kVoiceInterruptAccepted:
      break;
    case ConvEvent::kVoiceInterruptDenied:
      break;
    case ConvEvent::kHumanSpeakingDetail:
      break;
    case ConvEvent::kRespondingDetail:
      break;
  }
}

void onEtMessage(ConvLogLevel level, const char* log, void* user_data) {
  std::cout << " ==>> [" << level << "] " << log << std::endl;
}

int invalid_argv(int index, int argc) {
  if (index >= argc) {
    std::cout << "invalid params..." << std::endl;
    return 1;
  }
  return 0;
}

int parse_argv(int argc, char* argv[]) {
  int index = 1;
  while (index < argc) {
    if (!strcmp(argv[index], "--apikey")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_apikey = argv[index];
    } else if (!strcmp(argv[index], "--url")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_url = argv[index];
    } else if (!strcmp(argv[index], "--mode")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_mode = argv[index];
    } else if (!strcmp(argv[index], "--model")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_model = argv[index];
    } else if (!strcmp(argv[index], "--app_id")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_app_id = argv[index];
    } else if (!strcmp(argv[index], "--workspace_id")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_workspace_id = argv[index];
    } else if (!strcmp(argv[index], "--task_id")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_task_id = argv[index];
    } else if (!strcmp(argv[index], "--workspace")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_workspace = argv[index];
    } else if (!strcmp(argv[index], "--debug_path")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_debug_path = argv[index];
    } else if (!strcmp(argv[index], "--output_path")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_output_path = argv[index];
      mkdir(g_output_path.c_str(), 0755);
    } else if (!strcmp(argv[index], "--input_path")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_input_path = argv[index];
    } else if (!strcmp(argv[index], "--debug_mode")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_debug_mode = argv[index];
    } else if (!strcmp(argv[index], "--file")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_file_path = argv[index];
    } else if (!strcmp(argv[index], "--jpeg_file")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_jpeg_file_path = argv[index];
    } else if (!strcmp(argv[index], "--voice")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_voice = argv[index];
    } else if (!strcmp(argv[index], "--audio_format")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_format = argv[index];
    } else if (!strcmp(argv[index], "--sr")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_sr = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--downstream_audio_format")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_downstream_format = argv[index];
    } else if (!strcmp(argv[index], "--downstream_sr")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_downstream_sr = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--downstream_bitrate")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_downstream_bit_rate = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--downstream_frame_size")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_frame_size = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--transmit_rate_limit")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_transmit_rate_limit = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--frame_len")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_in_frame_len = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--log_level")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_log_level = argv[index];
    } else if (!strcmp(argv[index], "--message")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_msg_flag = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--threads")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_threads = atoi(argv[index]);
    } else if (!strcmp(argv[index], "--simpleLog")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_simple_log = atoi(argv[index]) == 0 ? false : true;
    } else if (!strcmp(argv[index], "--enable_external_aec_module")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_enable_external_aec_module = atoi(argv[index]) == 0 ? false : true;
    } else if (!strcmp(argv[index], "--enable_external_vad_module")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_enable_external_vad_module = atoi(argv[index]) == 0 ? false : true;
    } else if (!strcmp(argv[index], "--enable_vad_module")) {
      index++;
      if (invalid_argv(index, argc)) return 1;
      g_enable_vad_module = atoi(argv[index]) == 0 ? false : true;
    }
    index++;
  }

  if ((g_file_path.empty() && g_input_path.empty()) || g_apikey.empty()) {
    return -1;
  }
  return 0;
}

std::string gen_init_params(ParamCallBack* tst) {
  Json::FastWriter writer;
  Json::Value root;
  Json::Value agent_chat;
  std::string app_code = tst->conversation->GetParameter("app_code");
  root["mode"] = g_mode;
  root["chain_mode"] = "ws";
  root["ws_version"] = 3;

  root["debug_mode"] = g_debug_mode;
  if (app_code != "02C" && !g_workspace.empty()) {
    root["workspace"] = g_workspace;
  }
  if (g_debug_path.empty()) {
    root["debug_path"] = g_workspace;
  } else {
    root["debug_path"] = g_debug_path;
    root["save_log"] = true;
    root["save_wav"] = true;
  }
  if (g_simple_log) {
    root["log_level"] = "none";
  } else {
    root["log_level"] = g_log_level;
  }

  root["url"] = g_url;
  root["apikey"] = g_apikey;
  root["app_id"] = g_app_id;
  root["workspace_id"] = g_workspace_id;

  if (g_task_id.empty()) {
    ConversationUtils utils;
    g_task_id = utils.RandomUUID();
  }
  root["task_id"] = g_task_id;

  if (g_in_frame_len > 0) {
    // 不推荐进行设置修改！
    root["inbound_frame_len"] = g_in_frame_len;
  }

  Json::Value upstream;
  if (g_jpeg_file_path.empty()) {
    upstream["type"] = "AudioOnly";
  } else {
    upstream["type"] = "AudioAndVideo";
  }
  upstream["audio_format"] = g_format;  // asr格式，支持pcm,opus
  if (g_sr > 0) {
    upstream["sample_rate"] = g_sr;
  }
  root["upstream"] = upstream;

  Json::Value downstream;
  downstream["type"] = "Audio";
  downstream["voice"] = g_voice;
  downstream["sample_rate"] =
      g_downstream_sr;  // default tts sample_rate is 24000
  downstream["audio_format"] =
      g_downstream_format;  // 下发的音频编码格式，支持opu,pcm,opus
  downstream["intermediate_text"] = "transcript,dialog";
  downstream["debug"] = true;
  if (g_frame_size > 0) {
    downstream["frame_size"] = g_frame_size;
  }
  if (g_downstream_bit_rate > 0) {
    downstream["bit_rate"] = g_downstream_bit_rate;
  }
  downstream["enable_cbr"] = false;
  if (g_transmit_rate_limit > 0) {
    downstream["transmit_rate_limit"] =
        g_transmit_rate_limit;  // 下行音频发送速率限制,单位:字节每秒
  }
  root["downstream"] = downstream;

  Json::Value client_info; /* 额外扩充的用户信息 */
  client_info["user_id"] = "bin23439207";
  root["client_info"] = client_info;

  Json::Value biz_params;
  Json::Value user_prompt_params;
  user_prompt_params["user_name"] = "大米";
  biz_params["user_prompt_params"] = user_prompt_params;
  root["biz_params"] = biz_params;

  // 02C版本为纯云端能力，不需要设置VAD&AEC&KWS相关参数
  if (app_code != "02C") {
    Json::Value acoustic_echo_cancelling_attributes;
    acoustic_echo_cancelling_attributes["enable_external_aec_module"] =
        g_enable_external_aec_module;
    root["acoustic_echo_cancelling_attributes"] =
        acoustic_echo_cancelling_attributes;

    Json::Value voice_activity_detection_attributes;
    voice_activity_detection_attributes["enable_external_vad_module"] =
        g_enable_external_vad_module;
    voice_activity_detection_attributes["enable_vad_module"] =
        g_enable_vad_module;
    root["voice_activity_detection_attributes"] =
        voice_activity_detection_attributes;

    if (g_mode == "kws_duplex") {
      Json::Value key_words_spotting_attributes;
      key_words_spotting_attributes["enable_external_kws_module"] = false;
      key_words_spotting_attributes["independent_kws_mode"] = false;
      root["key_words_spotting_attributes"] = key_words_spotting_attributes;
    }
  }

  /*
   * duplex和kws_duplex模式下，
   * 02C版本为纯云端，启用双向语音模式，即时刻发送音频。
   * 或
   * enable_vad_module和enable_external_vad_module均为false，启用双向
   * 语音模式，即时刻发送音频。
   */
  if ((g_mode == "duplex" || g_mode == "kws_duplex") &&
      (app_code == "02C" ||
       (!g_enable_vad_module && !g_enable_external_vad_module))) {
    tst->bidirectional_speech = true;
  }

  /* 万能参数接口演示 */
  Json::Value nls_rule, nls_params;
  Json::Value nls_rule_payload(Json::arrayValue);
  nls_rule_payload.append("Start_model");
  nls_rule["payload"] = nls_rule_payload;
  root["nls_rule"] = nls_rule;
  nls_params["Start_model"] = g_model;
  root["nls_params"] = nls_params;

  std::string result = writer.write(root);
  std::cout << "init params: " << result.c_str() << std::endl;
  return result;
}

void* pthreadFunc(void* arg) {
  ConvRetCode ret = kSuccess;
  struct timeval now;
  struct timespec outtime;

  ParamCallBack* tst = static_cast<ParamCallBack*>(arg);

  if (g_format == "raw-opus") {
    pthread_create(&(tst->thread_id), NULL, &raw_asr_thread, tst);
  } else {
    pthread_create(&(tst->thread_id), NULL, &asr_thread, tst);
  }
  pthread_detach((tst->thread_id));

  // 1. create conversation
  on_message_callback = onMessage;
  if (g_msg_flag == 1) {
    on_et_callback = onEtMessage;
  } else {
    on_et_callback = NULL;
  }
  tst->conversation = Conversation::CreateConversation(on_message_callback,
                                                       on_et_callback, tst);

  if (tst->conversation) {
    // 2. init and Conncet
    ret = tst->conversation->Connect(gen_init_params(tst).c_str());
    if (ret) {
      std::cout << "connect failed: " << ret << " with "
                << tst->conversation->GetVersion() << std::endl;
      tst->conversation->DestroyConversation();
      tst->conversation = NULL;
      return NULL;
    } else {
      std::cout << "connect success: " << tst->conversation->GetVersion()
                << std::endl;
    }

    while (tst->thread_id != 0) {
      usleep(100 * 1000);
    }

    // 7. disconnect
    ret = tst->conversation->Disconnect();
    if (ret == 0) {
      std::cout << "disconnect success" << std::endl;
    }
  }

  if (!tst->exit) {
    tst->exit = true;
    while (tst->thread_id != 0) {
      usleep(100 * 1000);
    }
  }

  tst->conversation->DestroyConversation();
  tst->conversation = NULL;

  std::cout << "pthreadFunc exit." << std::endl;
}

int conversationMultiInstances(int threads) {
  ParamCallBack cbParam[threads];
  for (int i = 0; i < threads; i++) {
    strncpy(cbParam[i].userInfo, "User.", 0);
  }

  std::vector<pthread_t> pthread_ids(threads);
  // 启动四个工作线程, 同时识别四个音频文件
  for (int j = 0; j < threads; j++) {
    pthread_create(&pthread_ids[j], NULL, &pthreadFunc, (void*)&(cbParam[j]));
    usleep(500 * 1000);
  }
  std::cout << "start pthread_join..." << std::endl;

  for (int j = 0; j < threads; j++) {
    pthread_join(pthread_ids[j], NULL);
  }

  std::cout << "conversationMultiInstances exit." << std::endl;
}

int main(int argc, char* argv[]) {
  if (parse_argv(argc, argv)) {
    std::cout
        << "params is not valid.\n"
        << "Usage:\n"
        << "  --threads <default is 1.>\n"
        << "  --url <The url of server, default is "
           "wss://dashscope.aliyuncs.com/api-ws/v1/inference>\n"
        << "  --apikey <The apikey associated with the account>\n"
        << "  --workspace <The directory where the local model file is "
           "located>\n"
        << "  --debug_path <>\n"
        << "  --file <WAV audio file that simulates human speech>\n"
        << "  --voice <the voice name, longxiaochun is default>\n"
        << "  --mode <Mode of operation: tap_to_talk / duplex / push_to_talk / "
           "kws_duplex>\n"
        << "  --model <The name of LLM, can not be filled>\n"
        << "  --app_id <The ID of application>\n"
        << "  --workspace_id <>\n"
        << "  --enable_external_aec_module <>\n"
        << "  --enable_external_vad_module <>\n"
        << "  --enable_vad_module <>\n"
        << "eg:\n"
        << "  ./convBaiLianApiDemo --url "
           "wss://dashscope.aliyuncs.com/api-ws/v1/inference "
           "--apikey xxxxxxx --workspace resources_kws_vad_android/ --file "
           "asr_example.wav --mode duplex --debug_path tmp\n"
        << std::endl;
    return -1;
  }

  signal(SIGINT, signal_handler_int);
  signal(SIGQUIT, signal_handler_quit);

  conversationMultiInstances(g_threads);

  g_exit = true;
  return 0;
}
