//
//  DashGummySpeechTranscriberViewController.mm
//  NUIdemo
//
//  Created by shichen.fsc on 2025/9/15.
//  Copyright © 2025 Alibaba idst. All rights reserved.
//

#import <Foundation/Foundation.h>
#define DEBUG_MODE
#import "nuisdk.framework/Headers/NeoNui.h"
#import "DashGummySpeechTranscriberViewController.h"
#import "AppDelegate.h"
#import "NuiSdkUtils.h"
#import "HWOptionButton.h"

#import "audio/AudioController.h"
#import <AudioToolbox/AudioToolbox.h>
#include <sys/time.h>
#include <time.h>

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static NSString *debug_path = @"";
static dispatch_queue_t st_work_queue;

// iOS SDK 详细说明：https://help.aliyun.com/zh/model-studio/ios-sdk-for-gummy
// Gummy实时语音识别翻译WebSocket API: https://help.aliyun.com/zh/model-studio/real-time-websocket-api
@interface DashGummySpeechTranscriberViewController ()<ConvVoiceRecorderDelegate, HWOptionButtonDelegate, NeoNuiSdkDelegate> {
    IBOutlet UIButton *StartButton;
    IBOutlet UIButton *StopButton;

    IBOutlet UILabel *TransLabel;
    IBOutlet UISwitch *switchTrans;
    IBOutlet UILabel *SaveWavLabel;
    IBOutlet UISwitch *switchSaveWav;

    IBOutlet UITextView *textViewDialog;
    IBOutlet UITextView *textViewTranslate;
    IBOutlet UITextView *textViewEvent;
}

@property(nonatomic, weak) HWOptionButton *sampleRateHW;
@property(nonatomic, weak) HWOptionButton *formatHW;
@property(nonatomic, weak) HWOptionButton *modelHW;

@property(nonatomic,strong) NeoNui* nui;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NSMutableData *recordedVoiceData;
@property(nonatomic,strong) NuiSdkUtils *utils;

- (IBAction)showStart;
- (IBAction)showStop;
@end

@implementation DashGummySpeechTranscriberViewController

#pragma mark - View Callback Action
- (void)viewDidLoad {
    [super viewDidLoad];
    TLog(@"DashGummySpeechTranscriberViewController did load");

    TLog(@"Get API Key: %@", self.apikey);
    TLog(@"Get URL: %@", self.url);

    [self InitView];

    _utils = [NuiSdkUtils alloc];
    debug_path = [_utils createDir];

    st_work_queue = dispatch_queue_create("NuiGummySTController", DISPATCH_QUEUE_CONCURRENT);

    [self initNui];
}

-(void)viewWillDisappear:(BOOL)animated {
    NSLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    if (_nui != nil) {
        [_nui nui_release];
        _nui.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - UI action
- (IBAction)StartButHandler:(UIButton *)sender{
    TLog(@"click START BUTTON, start recorder!");
    textViewEvent.text = @"";
    textViewDialog.text = @"";
    textViewTranslate.text = @"";
    [self showStart];
}

- (IBAction)StopButHandler:(id)sender {
    TLog(@"click STOP BUTTON, stop recorder!");
    [self appendEventResult:@"STOP BUTTON"];
    [self showStop];
}

-(IBAction)showStart {
    if (_audioController == nil) {
        // 注意：这里audioController模块仅用于录音示例，用户可根据业务场景自行实现这部分代码
        _audioController = [[AudioController alloc] init:only_recorder];
        _audioController.delegate = self;
    }

    if (_audioController != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [self->StartButton setEnabled:NO];
            [self->StartButton setAlpha:0.4];
            [self->StopButton setEnabled:YES];
            [self->StopButton setAlpha:1.0];
        });

        dispatch_async(st_work_queue, ^{
            if (_nui != nil) {
                // nui_set_params 需要在nui_initialize之后
                NSString * parameters = [self genParams];
                TLog(@"nui set params %@", parameters);
                [_nui nui_set_params:[parameters UTF8String]];

                //若需要修改apikey等参数, 可详见genDialogParams()
                NSString * tmp_params = [self genDialogParams];
                [_nui nui_dialog_start:MODE_P2T dialogParam:[tmp_params UTF8String]];
            } else {
                TLog(@"in StartButHandler no nui alloc");
            }
        });
    }
}

-(IBAction)showStop {
    self.recordedVoiceData = nil;
    
    if (_nui != nil) {
        [_nui nui_dialog_cancel:NO];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [self->StartButton setEnabled:YES];
            [self->StartButton setAlpha:1.0];
            [self->StopButton setEnabled:NO];
            [self->StopButton setAlpha:0.4];
        });
    } else {
        TLog(@"in StopButHandler no nui alloc");
    }
}

-(IBAction)switchSaveWavAction:(id)sender
{
    UISwitch *switchButton = (UISwitch*)sender;
    BOOL isButtonOn = [switchButton isOn];

    TLog(@"Set save wav to %@", isButtonOn ? @"YES" : @"NO");
    if (isButtonOn == YES) {
        NSString *showText = [NSString stringWithFormat:@"打开调试音频本地存储, 存储在: %@", debug_path];
        [self showToastWithMessage:showText showTime:1];
    } else {
        [self showToastWithMessage:@"关闭调试音频本地存储" showTime:1];
    }

    [self terminateNui];
    usleep(200*1000);
    [self initNui];
    [self showStop];
}

-(IBAction)switchTranslateAction:(id)sender
{
    UISwitch *switchButton = (UISwitch*)sender;
    BOOL isButtonOn = [switchButton isOn];
    
    TLog(@"Set translate mode to %@", isButtonOn ? @"YES" : @"NO");
    if (isButtonOn == YES) {
        TransLabel.text = @"打开翻译功能";
    } else {
        TransLabel.text = @"关闭翻译功能";
    }
}

- (void)terminateNui {
    NSLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    if (_nui != nil) {
        [_nui nui_release];
        _nui.delegate = nil;
    }
}

-(void)dealloc{
    NSLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    if (_nui != nil) {
        [_nui nui_release];
        _nui.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
}

#pragma mark - Voice Recorder Delegate
-(void) recorderDidStart{
    TLog(@"recorderDidStart");
}

-(void) recorderDidStop{
    [self.recordedVoiceData setLength:0];
    TLog(@"recorderDidStop");
}

-(void) voiceRecorded:(unsigned char*)buffer Length:(int)len {
    NSData *frame = [NSData dataWithBytes:buffer length:len];
    @synchronized(_recordedVoiceData){
        [_recordedVoiceData appendData:frame];
    }
}

-(void) voiceDidFail:(NSError*)error{
    TLog(@"recorder error ");
}

#pragma mark - Nui Listener
-(void)onNuiEventCallback:(NuiCallbackEvent)nuiEvent
                   dialog:(long)dialog
                kwsResult:(const char *)wuw
                asrResult:(const char *)asr_result
                 ifFinish:(BOOL)finish
                  retCode:(int)code {
    TLog(@"onNuiEventCallback event %d finish %d", nuiEvent, finish);
    if (nuiEvent == EVENT_TRANSCRIBER_STARTED) {
        // asr_result在此包含task_id，task_id有助于排查问题，请用户进行记录保存。
        NSString *startedInfo = [NSString stringWithFormat:@"EVENT_TRANSCRIBER_STARTED: %@",
                                 [NSString stringWithUTF8String:asr_result]];
        TLog(@"%@", startedInfo);
    } else if (nuiEvent == EVENT_TRANSCRIBER_COMPLETE) {
        const char* all_response = [_nui nui_get_all_response];
        NSString *allResponse = [NSString stringWithUTF8String:all_response];
        NSError *error;
        NSData *jsonData = [allResponse dataUsingEncoding:NSUTF8StringEncoding];
        NSDictionary *jsonObject = [NSJSONSerialization JSONObjectWithData:jsonData options:0 error:&error];
        if (error) {
            TLog(@"JSON 解析失败: %@", error.localizedDescription);
            return;
        }
        NSDictionary *payload = jsonObject[@"payload"];
        if (!payload) {
            NSLog(@"缺少 payload");
            return;
        }
        NSDictionary *output = payload[@"output"];
        if (!output) {
            NSLog(@"缺少 output");
            return;
        }
        
        NSDictionary *transcription = output[@"transcription"];
        if (transcription && [transcription isKindOfClass:[NSDictionary class]]) {
            NSString *asrText = transcription[@"text"];
            if (asrText && asrText.length > 0) {
                [self showDialogResult:asrText];
            }
        }
        NSArray *translations = output[@"translations"];
        NSMutableString *sb = [NSMutableString string];
        if (translations && [translations isKindOfClass:[NSArray class]]) {
            for (NSDictionary *item in translations) {
                if ([item isKindOfClass:[NSDictionary class]]) {
                    NSString *text = item[@"text"];
                    if (text && text.length > 0) {
                        [sb appendString:text];
                    }
                }
            }
        }
        NSString *transText = sb.length > 0 ? sb : @"";
        [self showTranslateResult:transText];
    } else if (nuiEvent == EVENT_ASR_PARTIAL_RESULT || nuiEvent == EVENT_SENTENCE_END) {
        const char* all_response = [_nui nui_get_all_response];
        NSString *allResponse = [NSString stringWithUTF8String:all_response];
        NSError *error;
        NSData *jsonData = [allResponse dataUsingEncoding:NSUTF8StringEncoding];
        NSDictionary *jsonObject = [NSJSONSerialization JSONObjectWithData:jsonData options:0 error:&error];
        if (error) {
            TLog(@"JSON 解析失败: %@", error.localizedDescription);
            return;
        }
        NSDictionary *payload = jsonObject[@"payload"];
        if (!payload) {
            NSLog(@"缺少 payload");
            return;
        }
        NSDictionary *output = payload[@"output"];
        if (!output) {
            NSLog(@"缺少 output");
            return;
        }
        
        NSDictionary *transcription = output[@"transcription"];
        if (transcription && [transcription isKindOfClass:[NSDictionary class]]) {
            NSString *asrText = transcription[@"text"];
            if (asrText && asrText.length > 0) {
                [self showDialogResult:asrText];
            }
        }
        NSArray *translations = output[@"translations"];
        NSMutableString *sb = [NSMutableString string];
        if (translations && [translations isKindOfClass:[NSArray class]]) {
            for (NSDictionary *item in translations) {
                if ([item isKindOfClass:[NSDictionary class]]) {
                    NSString *text = item[@"text"];
                    if (text && text.length > 0) {
                        [sb appendString:text];
                    }
                }
            }
        }
        NSString *transText = sb.length > 0 ? sb : @"";
        [self showTranslateResult:transText];
    } else if (nuiEvent == EVENT_VAD_START) {
        TLog(@"EVENT_VAD_START");
        [self appendEventResult:@"EVENT_VAD_START"];
    } else if (nuiEvent == EVENT_VAD_END) {
        TLog(@"EVENT_VAD_END");
        [self appendEventResult:@"EVENT_VAD_END"];
    } else if (nuiEvent == EVENT_ASR_ERROR) {
        // asr_result在EVENT_ASR_ERROR中为错误信息，搭配错误码code和其中的task_id更易排查问题，请用户进行记录保存。
        const char* all_response = [_nui nui_get_all_response];
        NSString *errorMessage = [NSString stringWithFormat:@"EVENT_ASR_ERROR error[%d], all mesg[%@]",
                                  (int)code, [NSString stringWithUTF8String:all_response]];
        TLog(@"%@", errorMessage);
        [self appendEventResult:errorMessage];

        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
            [_audioController startRecorder];
        }
    } else if (nuiEvent == EVENT_MIC_ERROR) {
        TLog(@"MIC ERROR");
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
            [_audioController startRecorder];
        }
    }
    
    return;
}

-(int)onNuiNeedAudioData:(char *)audioData length:(int)len {
    static int emptyCount = 0;
    @autoreleasepool {
        @synchronized(_recordedVoiceData){
            if (_recordedVoiceData.length > 0) {
                int recorder_len = 0;
                if (_recordedVoiceData.length > len)
                    recorder_len = len;
                else
                    recorder_len = _recordedVoiceData.length;
                NSData *tempData = [_recordedVoiceData subdataWithRange:NSMakeRange(0, recorder_len)];
                [tempData getBytes:audioData length:recorder_len];
                tempData = nil;
                NSInteger remainLength = _recordedVoiceData.length - recorder_len;
                NSRange range = NSMakeRange(recorder_len, remainLength);
                [_recordedVoiceData setData:[_recordedVoiceData subdataWithRange:range]];
                emptyCount = 0;
                return recorder_len;
            } else {
                if (emptyCount++ >= 50) {
                    TLog(@"_recordedVoiceData length = %lu! empty 50times.", (unsigned long)_recordedVoiceData.length);
                    emptyCount = 0;
                }
                return 0;
            }

        }
    }
    return 0;
}

-(void)onNuiAudioStateChanged:(NuiAudioState)state{
    TLog(@"onNuiAudioStateChanged state=%u", state);
    if (state == STATE_CLOSE) {
        [self appendEventResult:@"RECORDER STATE_CLOSE"];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
    } else if (state == STATE_PAUSE) {
        [self appendEventResult:@"RECORDER STATE_PAUSE"];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
    } else if (state == STATE_OPEN){
        [self appendEventResult:@"RECORDER STATE_OPEN"];
        self.recordedVoiceData = [NSMutableData data];
        if (_audioController != nil) {
            [_audioController startRecorder];
        }
    }
}

-(void)onNuiRmsChanged:(float)rms {
//    TLog(@"onNuiRmsChanged rms=%f", rms);
}

-(void)onNuiLogTrackCallback:(NuiSdkLogLevel)level
                  logMessage:(const char *)log {
    TLog(@"onNuiLogTrackCallback log level:%d, message -> %s", level, log);
}

#pragma mark - Private methods

-(void) showEventResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        self->textViewEvent.text = result;
    });
}

-(void) showDialogResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        self->textViewDialog.text = result;
    });
}

-(void) showTranslateResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        self->textViewTranslate.text = result;
    });
}

-(void) appendEventResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        NSString *currentText = self->textViewEvent.text;
        NSString *updatedText = [currentText stringByAppendingFormat:@"\n---\n%@", result];
        self->textViewEvent.text = updatedText;
    });
}

- (void)showToastWithMessage:(NSString *)message showTime:(int)seconds {
    if ([message length] != 0) {
        dispatch_async(dispatch_get_main_queue(), ^{
            UIAlertController *alertController = [UIAlertController alertControllerWithTitle:nil
                                                                                     message:message
                                                                              preferredStyle:UIAlertControllerStyleAlert];
            
            [self presentViewController:alertController animated:YES completion:^{
                dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(seconds * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
                    [alertController dismissViewControllerAnimated:YES completion:nil];
                });
            }];
        });
    }
}

-(NSString*) genInitParams {
    NSMutableDictionary *ticketJsonDict = [NSMutableDictionary dictionary];

    // 若此处设置长效apikey, 则不需要每次nui_dialog_start都设置
    // 注意！不推荐在这里设置apikey。推荐每次在nui_dialog_start时设置临时鉴权token。
//    [ticketJsonDict setObject:_apikey forKey:@"apikey"];
    
    if (_url == nil || _url.length == 0) {
        _url = @"wss://dashscope.aliyuncs.com/api-ws/v1/inference";
    }
    [ticketJsonDict setObject:_url forKey:@"url"];

    [ticketJsonDict setObject:@"empty_device_id" forKey:@"device_id"]; // V2.6.2版本开始纯云端功能可不设置, 推荐填入具有唯一性的id, 方便定位问题

    if (debug_path != nil && debug_path.length > 0) {
        //debug目录，当初始化SDK时的save_log参数取值为true时，该目录用于保存中间音频文件
        [ticketJsonDict setObject:debug_path forKey:@"debug_path"];
        //当初始化SDK时的save_log参数取值为true时，该参数生效。表示是否保存音频debug，该数据保存在debug目录中，需要确保debug_path有效可写
        [ticketJsonDict setObject:switchSaveWav.on ? @"true" : @"false" forKey:@"save_wav"];
        //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
        [ticketJsonDict setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];
    }

    //过滤SDK内部日志通过回调送回到用户层
    [ticketJsonDict setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_NONE] forKey:@"log_track_level"];

    //FullMix = 0   // 选用此模式开启本地功能并需要进行鉴权注册
    //FullCloud = 1 // 在线实时语音识别可以选这个
    //FullLocal = 2 // 选用此模式开启本地功能并需要进行鉴权注册
    //AsrMix = 3    // 选用此模式开启本地功能并需要进行鉴权注册
    //AsrCloud = 4  // 在线一句话识别可以选这个
    //AsrLocal = 5  // 选用此模式开启本地功能并需要进行鉴权注册
    [ticketJsonDict setObject:@"1" forKey:@"service_mode"]; // 必填

    NSData *data = [NSJSONSerialization dataWithJSONObject:ticketJsonDict options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genParams {
    NSMutableDictionary *nls_config = [NSMutableDictionary dictionary];

    //参数可根据实际业务进行配置
    //接口说明可见:
    
    // 设置待识别音频格式。
    // 注意！例如opus，表示将用户送入的pcm数据压缩成opus数据进行传输,而不是让用户送入opus数据。
    [nls_config setObject:_formatHW.title forKey:@"sr_format"];
    // 模型选择, 注意模型对应的采样率要求。
    [nls_config setObject:_modelHW.title forKey:@"model"];
    // 设置待识别音频采样率（单位Hz）。只支持16000Hz。
    int sr_int = [_sampleRateHW.title intValue];
    [nls_config setObject:@(sr_int) forKey:@"sample_rate"];

    // 设置热词ID，若未设置则不生效。v2及更高版本模型设置热词ID时使用该字段。
    // 在本次语音识别中，将应用与该热词ID对应的热词信息。
//    [nls_config setObject:@"1234567890" forKey:@"vocabulary_id"];

    // 设置是否启用识别功能。
    // 模型支持单独开启识别或翻译功能，也可同时启用两种功能，但至少需要开启其中一种能力。
    [nls_config setValue:@YES forKey:@"transcription_enabled"];
    // 设置是否启用翻译功能。要正常输出翻译结果，需配置translation_target_languages参数。
    // 模型支持单独开启识别或翻译功能，也可同时启用两种功能，但至少需要开启其中一种能力。
    [nls_config setValue:switchTrans.on ? @YES : @NO forKey:@"translation_enabled"];
    if (switchTrans.on) {
        // 设置源（待识别/翻译语言）语言代码。如果无法提前确定语种，可不设置，默认为auto。
        [nls_config setValue:@"auto" forKey:@"source_language"];
        
        // 设置翻译目标语言代码。目标语言的代码与source_language参数一致。
        // 目前暂不支持同时翻译为多种语言，请仅设置一个目标语言以完成翻译。
        NSArray *targetArray = @[@"en"];
        NSError *error;
        NSData *data = [NSJSONSerialization dataWithJSONObject:targetArray options:0 error:&error];
        NSString *jsonString = error ? nil : [[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding];
        [nls_config setValue:jsonString forKey:@"translation_target_languages"];
    }

    // 设置最大结束静音时长，单位为毫秒（ms），取值范围为200ms至6000ms。
    // 若语音结束后静音时长超过该预设值，系统将判定当前语句已结束。
//    [nls_config setObject:@(800) forKey:@"max_end_silence"];

    // DNS解析的超时时间设置(单位ms)，默认5000
//    [nls_config setObject:@(500) forKey:@"dns_timeout"];

    // 设置文档中不存在的参数, key为custom_params, value以json string的形式设置参数
    // 如下示例传入{parameters:{"custom_param_flag":true},"user111":"111"}表示在payload下添加参数
    // payload.user111 : 111
    // payload.parameters.custom_param_flag : true
//    NSMutableDictionary *custom_params = [NSMutableDictionary dictionary];
//    NSMutableDictionary *param = [NSMutableDictionary dictionary];
//    [custom_params setValue:@"111" forKey:@"user111"];
//    [param setValue:@(YES) forKey:@"custom_param_flag"];
//    [custom_params setValue:param forKey:@"parameters"];
//    [nls_config setObject:custom_params forKey:@"extend_config"];
    
    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];
    [dictM setObject:nls_config forKey:@"nls_config"];
    [dictM setValue:@(SERVICE_TYPE_SPEECH_TRANSCRIBER) forKey:@"service_type"]; // 必填

    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genDialogParams {
    NSMutableDictionary *dialog_params = [NSMutableDictionary dictionary];

    // 注意！！！不要在端侧使用长效API Key！！！
    // 注意！！！不要在端侧使用长效API Key！！！
    // 注意！！！不要在端侧使用长效API Key！！！
    // 将长效API Key硬编码在端侧代码中，会导致安全风险！！！
    // 请在自建服务端获得临时鉴权Token（有效期60s，最长可设置1800s），再下发到端侧进行使用。
    // 临时鉴权Token: https://help.aliyun.com/zh/model-studio/obtain-temporary-authentication-token
    //
    // 服务只需要在临时Token(API Key)快过期前刷新一次。各端侧在Token(API Key)快过期前从服务获得新的
    // 临时Token(API Key)。
    [dialog_params setValue:_apikey forKey:@"apikey"];
    
    NSData *data = [NSJSONSerialization dataWithJSONObject:dialog_params options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

- (void) initNui {
    if (_nui == NULL) {
        _nui = [NeoNui get_instance];
        _nui.delegate = self;
    }
    //请注意此处的参数配置，其中账号相关需要按照genInitParams的说明填入后才可访问服务
    NSString * initParam = [self genInitParams];

    NuiResultCode retCode = [_nui nui_initialize:[initParam UTF8String] logLevel:NUI_LOG_LEVEL_DEBUG saveLog:@(YES)];
    TLog(@"nui initialize with code:%d", retCode);
    if (retCode != 0) {
        NSString *errInfo = [_utils getGuideWithError:retCode withError:@"" withStatus:@"init"];
        NSString *errorShow = [NSString stringWithFormat:@"error code:%d\nerror mesg: %@",
                                 (int)retCode, errInfo];
        [self showToastWithMessage:errInfo showTime:5];
        return;
    }
}

#pragma mark - View
-(CGFloat)InitButtonView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    
    //Start Button
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 70;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    
    CGFloat start_x = border;
    CGFloat start_y = startY;
    
    StartButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StartButton.frame = CGRectMake(start_x, start_y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [StartButton setBackgroundImage:image forState:UIControlStateNormal];
    [StartButton setTitle:@"开始" forState:UIControlStateNormal];
    [StartButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StartButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StartButton addTarget:self action:@selector(StartButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StartButton];
    
    [StartButton setEnabled:YES];
    [StartButton setAlpha:1.0];
    
    //Stop Button
    CGFloat stop_x = global_size.width/SCREEN_WIDTH_BASE * (90 * 2) + start_x;
    CGFloat stop_y = start_y;
    
    StopButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StopButton.frame = CGRectMake(stop_x, stop_y, button_width, button_height);
    image = [UIImage imageNamed:@"button_start"];
    [StopButton setBackgroundImage:image forState:UIControlStateNormal];
    [StopButton setTitle:@"结束" forState:UIControlStateNormal];
    [StopButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StopButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StopButton addTarget:self action:@selector(StopButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StopButton];
    
    [StopButton setEnabled:NO];
    [StopButton setAlpha:0.4];
    
    return stop_y + button_height;
}

-(CGFloat)InitSwitchView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    CGFloat label_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat label_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat switch_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat switch_height = global_size.height/SCREEN_HEIGHT_BASE * 30;

    CGFloat interval_width = global_size.width/SCREEN_WIDTH_BASE * 10;
    CGFloat interval_height = global_size.height/SCREEN_HEIGHT_BASE * 10;
    
    CGFloat start_x = border;
    CGFloat start_y = startY;
    
    //translate label
    CGFloat translate_label_width = global_size.width/SCREEN_WIDTH_BASE * 100;
    CGFloat translate_label_x = start_x;
    CGFloat translate_label_y = start_y;

    CGRect translate_label_rect = CGRectMake(translate_label_x, translate_label_y, translate_label_width, label_height);
    if (!TransLabel) {
        TransLabel = [[UILabel alloc] initWithFrame:translate_label_rect];
    }
    TransLabel.text = @"关闭翻译功能";
    TransLabel.textColor = UIColor.blackColor;
    TransLabel.backgroundColor = UIColor.lightGrayColor;
    TransLabel.layer.cornerRadius = 5.0;
    TransLabel.layer.masksToBounds = true;
    [self.view addSubview:TransLabel];

    //translate switch
    CGFloat translate_switch_x = translate_label_x + translate_label_width + interval_width;
    CGFloat translate_switch_y = start_y;

    CGRect translate_switch_rect = CGRectMake(translate_switch_x, translate_switch_y, switch_width, switch_height);
    switchTrans = [[UISwitch alloc] initWithFrame:translate_switch_rect];
    [switchTrans setOn:NO animated:NO];
    [switchTrans addTarget:self action:@selector(switchTranslateAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:switchTrans];

    //save wav label
    CGFloat save_wav_label_x = global_size.width / 2 + border;
    CGFloat save_wav_label_y = start_y;

    CGRect save_wav_label_rect = CGRectMake(save_wav_label_x, save_wav_label_y, label_width, label_height);
    if (!SaveWavLabel) {
        SaveWavLabel = [[UILabel alloc] initWithFrame:save_wav_label_rect];
    }
    SaveWavLabel.text = @"音频保存";
    SaveWavLabel.textColor = UIColor.blackColor;
    SaveWavLabel.backgroundColor = UIColor.lightGrayColor;
    SaveWavLabel.layer.cornerRadius = 5.0;
    SaveWavLabel.layer.masksToBounds = true;
    [self.view addSubview:SaveWavLabel];

    //save wav switch
    CGFloat save_wav_switch_x = save_wav_label_x + label_width + interval_width;
    CGFloat save_wav_switch_y = save_wav_label_y;

    CGRect save_wav_switch_rect = CGRectMake(save_wav_switch_x, save_wav_switch_y, switch_width, switch_height);
    switchSaveWav = [[UISwitch alloc] initWithFrame:save_wav_switch_rect];
    [switchSaveWav addTarget:self action:@selector(switchSaveWavAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:switchSaveWav];
    
    return save_wav_switch_y + switch_height;
}

-(CGFloat)InitHWOptionsView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    CGFloat sampleRateHW_width = global_size.width/SCREEN_WIDTH_BASE * 128;
    CGFloat sampleRateHW_height = global_size.height/SCREEN_HEIGHT_BASE * 32;
    CGFloat sampleRateHW_x = border;
    CGFloat sampleRateHW_y = startY;
    HWOptionButton *sampleRateHW = [[HWOptionButton alloc] initWithFrame:CGRectMake(sampleRateHW_x, sampleRateHW_y, sampleRateHW_width, sampleRateHW_height)];
    sampleRateHW.backgroundColor = [UIColor lightGrayColor];
    sampleRateHW.array = @[@"16000"];
    sampleRateHW.selectIndex = 0;
    sampleRateHW.showPlaceholder = NO;
    sampleRateHW.dropdownTitle = @" - 采样率 - ";
    sampleRateHW.delegate = self;
    [self.view addSubview:sampleRateHW];
    self.sampleRateHW = sampleRateHW;

    CGFloat formatHW_width = global_size.width/SCREEN_WIDTH_BASE * 128;
    CGFloat formatHW_height = global_size.height/SCREEN_HEIGHT_BASE * 32;
    CGFloat formatHW_x = global_size.width / 2 + border;
    CGFloat formatHW_y = startY;
    HWOptionButton *formatHW = [[HWOptionButton alloc] initWithFrame:CGRectMake(formatHW_x, formatHW_y, formatHW_width, formatHW_height)];
    formatHW.backgroundColor = [UIColor lightGrayColor];
    formatHW.array = @[@"opus", @"pcm"];
    formatHW.selectIndex = 0;
    formatHW.showPlaceholder = NO;
    formatHW.dropdownTitle = @" - 音频编码格式 - ";
    formatHW.delegate = self;
    [self.view addSubview:formatHW];
    self.formatHW = formatHW;
    
    CGFloat modelHW_width = global_size.width/SCREEN_WIDTH_BASE * 220;
    CGFloat modelHW_height = global_size.height/SCREEN_HEIGHT_BASE * 32;
    CGFloat modelHW_x = border;
    CGFloat modelHW_y = startY + sampleRateHW_height + border;
    HWOptionButton *modelHW = [[HWOptionButton alloc] initWithFrame:CGRectMake(modelHW_x, modelHW_y, modelHW_width, modelHW_height)];
    modelHW.backgroundColor = [UIColor lightGrayColor];
    modelHW.array = @[@"gummy-realtime-v1"];
    modelHW.selectIndex = 0;
    modelHW.showPlaceholder = NO;
    modelHW.dropdownTitle = @" - 模型名称 - ";
    modelHW.delegate = self;
    [self.view addSubview:modelHW];
    self.modelHW = modelHW;
    
    return modelHW_y + modelHW_height;
}

-(CGFloat)InitTextView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    
    //asr text view
    CGFloat dialog_view_width = global_size.width - border * 2;
    CGFloat dialog_view_height = global_size.height/SCREEN_HEIGHT_BASE * 80;
    CGFloat dialog_x = border;
    CGFloat dialog_y = startY + border;
    
    CGRect dialog_rect = CGRectMake(dialog_x, dialog_y, dialog_view_width, dialog_view_height);
    if (!textViewDialog) {
        textViewDialog = [[UITextView alloc] initWithFrame:dialog_rect];
    }
    textViewDialog.layer.borderWidth = 0.6;
    textViewDialog.layer.borderColor = [UIColor blackColor].CGColor;
    textViewDialog.layer.cornerRadius = 10;
    [textViewDialog setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewDialog.scrollEnabled = YES;
    
    textViewDialog.text = @"识别内容";
    textViewDialog.textColor = [UIColor darkGrayColor];
    textViewDialog.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewDialog];
    
    textViewDialog.scrollEnabled = YES;
    textViewDialog.editable = NO;
    
    //translate text view
    CGFloat translate_view_width = global_size.width - border * 2;
    CGFloat translate_view_height = global_size.height/SCREEN_HEIGHT_BASE * 80;
    CGFloat translate_x = border;
    CGFloat translate_y = dialog_y + dialog_view_height + border;
    
    CGRect translate_rect = CGRectMake(translate_x, translate_y, translate_view_width, translate_view_height);
    if (!textViewTranslate) {
        textViewTranslate = [[UITextView alloc] initWithFrame:translate_rect];
    }
    textViewTranslate.layer.borderWidth = 0.6;
    textViewTranslate.layer.borderColor = [UIColor blackColor].CGColor;
    textViewTranslate.layer.cornerRadius = 10;
    [textViewTranslate setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewTranslate.scrollEnabled = YES;
    
    textViewTranslate.text = @"翻译内容";
    textViewTranslate.textColor = [UIColor darkGrayColor];
    textViewTranslate.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewTranslate];
    
    textViewTranslate.scrollEnabled = YES;
    textViewTranslate.editable = NO;
    
    //event text view
    CGFloat event_view_width = global_size.width - border * 2;
    CGFloat event_view_height = global_size.height/SCREEN_HEIGHT_BASE * 200;
    CGFloat event_x = border;
    CGFloat event_y = translate_y + translate_view_height + border;
    
    CGRect event_rect = CGRectMake(event_x, event_y, event_view_width, event_view_height);
    if (!textViewEvent) {
        textViewEvent = [[UITextView alloc] initWithFrame:event_rect];
    }
    textViewEvent.layer.borderWidth = 0.6;
    textViewEvent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewEvent.layer.cornerRadius = 10;
    [textViewEvent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewEvent.scrollEnabled = YES;
    
    textViewEvent.text = @"响应文本";
    textViewEvent.textColor = [UIColor darkGrayColor];
    textViewEvent.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewEvent];

    textViewEvent.scrollEnabled = YES;
    textViewEvent.editable = NO;
    
    return event_y + event_view_height;
}

-(void)InitView {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"实时转写";
    CGFloat border = global_size.width/SCREEN_WIDTH_BASE * 13;
    CGFloat division = global_size.height/SCREEN_HEIGHT_BASE * 10;
    CGFloat end_y = 0;

    CGFloat button_border = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat button_start_y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    end_y = [self InitButtonView:button_border startY:button_start_y];
    
    CGFloat switch_start_y = end_y + division;
    end_y = [self InitSwitchView:border startY:switch_start_y];
    
    CGFloat hwoptions_start_y = end_y + division;
    end_y = [self InitHWOptionsView:border startY:hwoptions_start_y];
    
    CGFloat view_start_y = end_y + division;
    end_y = [self InitTextView:border startY:view_start_y];
}


@end
