//
//  RealtimeMeetingViewController.m
//  NUIdemo
//
//  Created by shichen.fsc on 2024/6/7.
//  Copyright © 2024 Alibaba idst. All rights reserved.
//

#import <Foundation/Foundation.h>
#define DEBUG_MODE
#import "nuisdk.framework/Headers/NeoNui.h"
#import "RealtimeMeetingViewController.h"
#import "AppDelegate.h"
#import "NuiSdkUtils.h"

#import "audio/AudioController.h"
#include <sys/time.h>
#include <time.h>

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static BOOL save_wav = NO;
static BOOL save_log = NO;

@interface RealtimeMeetingViewController ()<ConvVoiceRecorderDelegate, NeoNuiSdkDelegate> {
    IBOutlet UIButton *StartButton;
    IBOutlet UIButton *StopButton;

    IBOutlet UITextView *textViewKws;
    IBOutlet UITextView *textViewAsr;
    IBOutlet UITextView *textViewDialog;

    IBOutlet UISwitch *switchLog;

    IBOutlet UITextField *textfield;
    IBOutlet UITextField *textVersion;
    
    IBOutlet UILabel *SaveWavLabel;
}
@property(nonatomic,strong) NeoNui* nui;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NSMutableData *recordedVoiceData;
@property(nonatomic,strong) NuiSdkUtils *utils;
@property(nonatomic) NSInteger callNum;

- (IBAction)showStart;
- (IBAction)showStop;
@end

@implementation RealtimeMeetingViewController

#pragma mark - View Callback Action
- (void)viewDidLoad {
    [super viewDidLoad];
    TLog(@"RealtimeMeetingViewController did load");
    
    [self InitView];

    _utils = [NuiSdkUtils alloc];
    _callNum = 0;

    [self initNui];
    NSString *version = [NSString stringWithUTF8String:[_nui nui_get_version]];
    self->textVersion.text = version;
}

-(void)viewWillDisappear:(BOOL)animated {
    NSLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    if (_nui != nil) {
        [_nui nui_release];
        _nui.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - UI action
- (IBAction)StartButHandler:(UIButton *)sender{
    TLog(@"click START BUTTON, start recorder!");
    [self showStart];
}

- (IBAction)StopButHandler:(id)sender {
    TLog(@"click STOP BUTTON, stop recorder!");
    [self showStop];
}

-(IBAction)showStart {
    if (_audioController == nil) {
        // 注意：这里audioController模块仅用于录音示例，用户可根据业务场景自行实现这部分代码
        _audioController = [[AudioController alloc] init:only_recorder];
        _audioController.delegate = self;
    }

    if (_audioController != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [self->StartButton setEnabled:NO];
            [self->StartButton setAlpha:0.4];
            [self->StopButton setEnabled:YES];
            [self->StopButton setAlpha:1.0];
            [self showKwsResult:@""];

            if (_nui != nil) {
                //若需要修改token等参数, 可详见genDialogParams()
                NSString * parameters = [self genDialogParams];
                [_nui nui_dialog_start:MODE_P2T dialogParam:[parameters UTF8String]];
            } else {
                TLog(@"in StartButHandler no nui alloc");
            }
        });
    }
}

-(IBAction)showStop {
    self.recordedVoiceData = nil;
    
    if (_nui != nil) {
        [_nui nui_dialog_cancel:NO];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [self->StartButton setEnabled:YES];
            [self->StartButton setAlpha:1.0];
            [self->StopButton setEnabled:NO];
            [self->StopButton setAlpha:0.4];
        });
    } else {
        TLog(@"in StopButHandler no nui alloc");
    }
}

-(IBAction)switchLogAction:(id)sender
{
    UISwitch *switchButton = (UISwitch*)sender;
    BOOL isButtonOn = [switchButton isOn];

    TLog(@"set save wav to %d from %d",
              isButtonOn, save_wav);
    save_wav = isButtonOn;
    save_log = isButtonOn;
    [self terminateNui];
    usleep(200*1000);
    [self initNui];
    [self showStop];
    [self showStart];
}


- (void)terminateNui {
    NSLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    [_nui nui_release];
}

-(void)dealloc{
    NSLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    if (_nui != nil) {
        [_nui nui_release];
        _nui.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
}

#pragma mark - Voice Recorder Delegate
-(void) recorderDidStart{
    TLog(@"recorderDidStart");
}

-(void) recorderDidStop{
    [self.recordedVoiceData setLength:0];
    TLog(@"recorderDidStop");
}

-(void) voiceRecorded:(unsigned char*)buffer Length:(int)len {
    NSData *frame = [NSData dataWithBytes:buffer length:len];
    @synchronized(_recordedVoiceData){
        [_recordedVoiceData appendData:frame];
    }
}

-(void) voiceDidFail:(NSError*)error{
    TLog(@"recorder error ");
}

#pragma mark - Nui Listener
-(void)onNuiEventCallback:(NuiCallbackEvent)nuiEvent
                   dialog:(long)dialog
                kwsResult:(const char *)wuw
                asrResult:(const char *)asr_result
                 ifFinish:(BOOL)finish
                  retCode:(int)code {
    if (asr_result != NULL) {
        TLog(@"onNuiEventCallback event %d finish %d, %s", nuiEvent, finish, asr_result);
    } else {
        TLog(@"onNuiEventCallback event %d finish %d", nuiEvent, finish);
    }
    if (nuiEvent == EVENT_SENTENCE_START) {
        const char* response = [_nui nui_get_all_response];
        NSString *result = [NSString stringWithUTF8String:response];
        [self showAsrResult:result];
    } else if (nuiEvent == EVENT_ASR_PARTIAL_RESULT || nuiEvent == EVENT_SENTENCE_END) {
        // asr_result在此包含task_id，task_id有助于排查问题，请用户进行记录保存。
        TLog(@"ASR RESULT %s finish %d", asr_result, finish);
        NSString *result = [NSString stringWithUTF8String:asr_result];
        [self showAsrResult:result];
    } else if (nuiEvent == EVENT_RESULT_TRANSLATED) {
        NSString *result = [NSString stringWithUTF8String:asr_result];
        [self showAsrResult:result];
    } else if (nuiEvent == EVENT_ASR_ERROR) {
        // asr_result在EVENT_ASR_ERROR中为错误信息，搭配错误码code和其中的task_id更易排查问题，请用户进行记录保存。
        TLog(@"EVENT_ASR_ERROR error[%d]", code);
    } else if (nuiEvent == EVENT_MIC_ERROR) {
        TLog(@"MIC ERROR");
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
            [_audioController startRecorder];
        }
    }
    
    //finish 为真（可能是发生错误，也可能是完成识别）表示一次任务生命周期结束，可以开始新的识别
    if (finish) {
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [self->StartButton setEnabled:YES];
            [self->StartButton setAlpha:1.0];
            [self->StopButton setEnabled:NO];
            [self->StopButton setAlpha:0.4];
        });
    }
    
    return;
}

-(int)onNuiNeedAudioData:(char *)audioData length:(int)len {
    static int emptyCount = 0;
    @autoreleasepool {
        @synchronized(_recordedVoiceData){
            if (_recordedVoiceData.length > 0) {
                int recorder_len = 0;
                if (_recordedVoiceData.length > len)
                    recorder_len = len;
                else
                    recorder_len = _recordedVoiceData.length;
                NSData *tempData = [_recordedVoiceData subdataWithRange:NSMakeRange(0, recorder_len)];
                [tempData getBytes:audioData length:recorder_len];
                tempData = nil;
                NSInteger remainLength = _recordedVoiceData.length - recorder_len;
                NSRange range = NSMakeRange(recorder_len, remainLength);
                [_recordedVoiceData setData:[_recordedVoiceData subdataWithRange:range]];
                emptyCount = 0;
                return recorder_len;
            } else {
                if (emptyCount++ >= 50) {
                    TLog(@"_recordedVoiceData length = %lu! empty 50times.", (unsigned long)_recordedVoiceData.length);
                    emptyCount = 0;
                }
                return 0;
            }

        }
    }
    return 0;
}

-(void)onNuiAudioStateChanged:(NuiAudioState)state{
    TLog(@"onNuiAudioStateChanged state=%u", state);
    if (state == STATE_CLOSE || state == STATE_PAUSE) {
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
    } else if (state == STATE_OPEN){
        self.recordedVoiceData = [NSMutableData data];
        if (_audioController != nil) {
            [_audioController startRecorder];
        }
    }
}

-(void)onNuiRmsChanged:(float)rms {
//    TLog(@"onNuiRmsChanged rms=%f", rms);
}

-(void)onNuiLogTrackCallback:(NuiSdkLogLevel)level
                  logMessage:(const char *)log {
    TLog(@"onNuiLogTrackCallback log level:%d, message -> %s", level, log);
}

#pragma mark - Private methods

-(void) showKwsResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        self->textViewKws.text = result;
    });
}

-(void) showAsrResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        self->textViewAsr.text = result;
    });
}

-(void) showDialogResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        self->textViewDialog.text = result;
    });
}

-(NSString*) genInitParams {
//    NSString *strResourcesBundle = [[NSBundle mainBundle] pathForResource:@"Resources" ofType:@"bundle"];
//    NSString *bundlePath = [[NSBundle bundleWithPath:strResourcesBundle] resourcePath]; // 注意: V2.6.2版本开始纯云端功能可不需要资源文件
    NSString *debug_path = [_utils createDir];

    // 1. 接口与实现：https://help.aliyun.com/zh/tingwu/interface-and-implementation?spm=a2c4g.11186623.0.0.2b045060jFsVFw
    //    按文档步骤，首先创建AccessKey和创建项目
    //    然后需要用户在自己的服务端调用CreateTask接口创建实时记录，获得MeetingJoinUrl
    //    此MeetingJoinUrl即为下方url
    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];

    //账号和项目创建
    //  ak_id ak_secret 如何获得,请查看https://help.aliyun.com/document_detail/72138.html
    [dictM setObject:@"default" forKey:@"app_key"]; // 必填，不改动
    [dictM setObject:@"default" forKey:@"token"]; // 必填，不改动

    // url中填入生成的MeetingJoinUrl。
    // 由于MeetingJoinUrl生成过程涉及ak/sk，移动端不可存储账号信息，故需要在服务端生成，并下发给移动端。
    // 详细请看： https://help.aliyun.com/zh/tingwu/api-tingwu-2023-09-30-createtask?spm=a2c4g.11186623.0.i32
    [dictM setObject:@"wss://tingwu-realtime-cn-hangzhou-pre.aliyuncs.com/api/ws/v1?XXXX" forKey:@"url"]; // 必填
    //工作目录路径，SDK从该路径读取配置文件
//    [ticketJsonDict setObject:bundlePath forKey:@"workspace"]; // V2.6.2版本开始纯云端功能可不设置workspace

    //当初始化SDK时的save_log参数取值为true时，该参数生效。表示是否保存音频debug，该数据保存在debug目录中，需要确保debug_path有效可写
    [dictM setObject:save_wav ? @"true" : @"false" forKey:@"save_wav"];
    //debug目录。当初始化SDK时的save_log参数取值为true时，该目录用于保存中间音频文件
    [dictM setObject:debug_path forKey:@"debug_path"];

    //过滤SDK内部日志通过回调送回到用户层
    [dictM setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_INFO] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [dictM setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    //FullCloud = 1 // 在线实时语音识别可以选这个
    [dictM setObject:@"1" forKey:@"service_mode"]; // 必填，不改动

    [dictM setObject:@"empty_device_id" forKey:@"device_id"]; // 必填, 推荐填入具有唯一性的id, 方便定位问题

    //默认true。
    // true: 将SentenceEnd的stash_result以EVENT_ASR_PARTIAL_RESULT给出。
    // false: SentenceEnd的stash_result不以EVENT_ASR_PARTIAL_RESULT事件给出。
    [dictM setObject:@"false" forKey:@"stash_partial_result"];

    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genParams {
    NSMutableDictionary *nls_config = [NSMutableDictionary dictionary];
    [nls_config setValue:@16000 forKey:@"sample_rate"];
    [nls_config setValue:@"pcm" forKey:@"sr_format"];
    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];
    [dictM setObject:nls_config forKey:@"nls_config"];
    [dictM setValue:@(SERVICE_TYPE_SPEECH_TRANSCRIBER) forKey:@"service_type"]; // 必填
    
    /*若文档中不包含某些参数，但是此功能支持这个参数，可以用如下万能接口设置参数*/
//    NSMutableDictionary *extend_config = [NSMutableDictionary dictionary];
//    [extend_config setValue:@YES forKey:@"custom_test"];
//    [dictM setObject:extend_config forKey:@"extend_config"];
    
    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genDialogParams {
    NSMutableDictionary *dialog_params = [NSMutableDictionary dictionary];    
    NSData *data = [NSJSONSerialization dataWithJSONObject:dialog_params options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

- (void) initNui {
    if (_nui == NULL) {
        _nui = [NeoNui get_instance];
        _nui.delegate = self;
    }
    //请注意此处的参数配置，其中账号相关需要按照genInitParams的说明填入后才可访问服务
    NSString * initParam = [self genInitParams];

    [_nui nui_initialize:[initParam UTF8String] logLevel:NUI_LOG_LEVEL_VERBOSE saveLog:save_log];
    NSString * parameters = [self genParams];
    [_nui nui_set_params:[parameters UTF8String]];
}

#pragma mark - View
-(void)InitView {
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"实时转写";
/*---------------------------------Button---------------------------------------------*/
    //Start Button
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 70;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 30;

    
    CGFloat x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    
    StartButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StartButton.frame = CGRectMake(x, y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [StartButton setBackgroundImage:image forState:UIControlStateNormal];
    [StartButton setTitle:@"开始" forState:UIControlStateNormal];
    [StartButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StartButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StartButton addTarget:self action:@selector(StartButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StartButton];
    
    //stop Button
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5+90*2);
    y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    
    StopButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StopButton.frame = CGRectMake(x, y, button_width, button_height);
    image = [UIImage imageNamed:@"button_start"];
    [StopButton setBackgroundImage:image forState:UIControlStateNormal];
    [StopButton setTitle:@"结束" forState:UIControlStateNormal];
    [StopButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StopButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StopButton addTarget:self action:@selector(StopButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StopButton];
    
    
/*---------------------------------UITextView---------------------------------------------*/
    //kws text view
    CGFloat kws_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat kws_view_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    y = global_size.height/SCREEN_HEIGHT_BASE * 110;
    
    CGRect textViewContent_rect = CGRectMake(x, y, kws_view_width, kws_view_height);
    if (!textViewKws) {
        textViewKws = [[UITextView alloc] initWithFrame:textViewContent_rect];
    }
    textViewKws.layer.borderWidth = 0.6;
    textViewKws.layer.borderColor = [UIColor blackColor].CGColor;
    textViewKws.layer.cornerRadius = 10;
    [textViewKws setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewKws.scrollEnabled = YES;
    
    textViewKws.textColor = [UIColor darkGrayColor];
    textViewKws.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewKws];
    
    //wakeup text view
    CGFloat asr_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat asr_view_height = global_size.height/SCREEN_HEIGHT_BASE * 60;
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    y = global_size.height/SCREEN_HEIGHT_BASE * 110 + kws_view_height + 10;
    
    CGRect asr_rect = CGRectMake(x, y, asr_view_width, asr_view_height);
    if (!textViewAsr) {
        textViewAsr = [[UITextView alloc] initWithFrame:asr_rect];
    }
    textViewAsr.layer.borderWidth = 0.6;
    textViewAsr.layer.borderColor = [UIColor blackColor].CGColor;
    textViewAsr.layer.cornerRadius = 10;
    [textViewAsr setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewAsr.scrollEnabled = YES;
    
    textViewAsr.textColor = [UIColor darkGrayColor];
    textViewAsr.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewAsr];
    
    //dialog text view
    CGFloat dialog_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat dialog_view_height = global_size.height/SCREEN_HEIGHT_BASE * 120;
    CGFloat dialog_x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    CGFloat dialog_y = global_size.height/SCREEN_HEIGHT_BASE * 110 + kws_view_height + 10 + asr_view_height + 10;
    
    CGRect dialog_rect = CGRectMake(dialog_x, dialog_y, dialog_view_width, dialog_view_height);
    if (!textViewDialog) {
        textViewDialog = [[UITextView alloc] initWithFrame:dialog_rect];
    }
    textViewDialog.layer.borderWidth = 0.6;
    textViewDialog.layer.borderColor = [UIColor blackColor].CGColor;
    textViewDialog.layer.cornerRadius = 10;
    [textViewDialog setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewDialog.scrollEnabled = YES;
    
    textViewDialog.textColor = [UIColor darkGrayColor];
    textViewDialog.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewDialog];
    
    /*---------------------------------Switch---------------------------------------------*/
    
    CGFloat switch_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat switch_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat label_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat label_height = global_size.height/SCREEN_HEIGHT_BASE * 30;

    //switch save wav
    CGFloat save_wav_label_x = dialog_x;
    CGFloat save_wav_label_y = dialog_y + dialog_view_height + label_height;

    CGRect save_wav_label_rect = CGRectMake(save_wav_label_x, save_wav_label_y, label_width, label_height);
    if (!SaveWavLabel) {
        SaveWavLabel = [[UILabel alloc] initWithFrame:save_wav_label_rect];
    }
    SaveWavLabel.text = @"音频保存";
    [self.view addSubview:SaveWavLabel];

    CGFloat save_wav_switch_x = save_wav_label_x + label_width;
    CGFloat save_wav_switch_y = save_wav_label_y;
    CGRect save_wav_switch_rect = CGRectMake(save_wav_switch_x, save_wav_switch_y, switch_width, switch_height);
    switchLog = [[UISwitch alloc] initWithFrame:save_wav_switch_rect];
    [switchLog addTarget:self action:@selector(switchLogAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:switchLog];
    
    /*---------------------------------------TextField----------------------------------*/
    
    CGFloat text_field_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat text_field_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    //text field
    CGFloat text_field_x = save_wav_label_x;
    CGFloat text_field_y = save_wav_switch_y + label_height + label_height;
    
    CGRect text_field_rect = CGRectMake(text_field_x, text_field_y, text_field_width, text_field_height);
    textfield = [[UITextField alloc] initWithFrame:text_field_rect];
    textfield.layer.borderWidth = 0.6;
    textfield.layer.borderColor = [UIColor blackColor].CGColor;
    textfield.layer.cornerRadius = 10;
    [textfield setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    
    textfield.textColor = [UIColor darkGrayColor];
    textfield.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textfield];
    
    //text sdk version
    CGFloat text_version_x = text_field_x;
    CGFloat text_version_y = text_field_y + text_field_height + 2;
    
    CGRect text_version_rect = CGRectMake(text_version_x, text_version_y, text_field_width, text_field_height);
    textVersion = [[UITextField alloc] initWithFrame:text_version_rect];

    textVersion.layer.borderWidth = 0.6;
    textVersion.layer.borderColor = [UIColor blackColor].CGColor;
    textVersion.layer.cornerRadius = 10;
    [textVersion setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    
    textVersion.textColor = [UIColor darkGrayColor];
    textVersion.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textVersion];
    
    
    [self->StartButton setEnabled:YES];
    [self->StartButton setAlpha:1.0];
    [self->StopButton setEnabled:NO];
    [self->StopButton setAlpha:0.4];
    
    textViewKws.scrollEnabled = YES;
    textViewAsr.scrollEnabled = YES;
    textViewDialog.scrollEnabled = YES;
    
    textViewKws.editable = NO;
    textViewAsr.editable = NO;
    textViewDialog.editable = NO;
    
    self->textVersion.minimumFontSize = 10;
    self->textfield.minimumFontSize = 10;
}


@end
