//
//  DashSambertTTSViewController.m
//  NUIdemo
//
//  Created by shichen.fsc on 2025/9/15.
//  Copyright © 2025 Alibaba idst. All rights reserved.
//

//#define DEBUG_MODE
//#define DEBUG_TTS_DATA_SAVE
#import "nuisdk.framework/Headers/NeoNuiTts.h"
#import "DashSambertTTSViewController.h"
#import "TTSViewController.h"
#import "HWOptionButton.h"
#import "NuiSdkUtils.h"

#import "audio/AudioController.h"
#import <AVFoundation/AVFoundation.h>

#ifdef DEBUG_TTS_DATA_SAVE
FILE * fp;
#endif

// 本样例展示在线语音合成使用方法
// iOS SDK 详细说明：https://help.aliyun.com/zh/model-studio/sambert-ios-sdk
// WebSocket API: https://help.aliyun.com/zh/model-studio/sambert-websocket-api
@interface DashSambertTTSViewController () <ConvVoiceRecorderDelegate, UITextFieldDelegate, HWOptionButtonDelegate, NeoNuiTtsDelegate> {
    IBOutlet UIButton *PlayButton;
    IBOutlet UIButton *PauseButton;
    IBOutlet UIButton *ResumeButton;
    IBOutlet UIButton *CancelButton;
    IBOutlet UIButton *QuitButton;
    
    IBOutlet UITextView *textViewContent;
    IBOutlet UITextView *textViewEvent;
}

@property(nonatomic,strong) NeoNuiTts* nui;
@property(nonatomic, weak) HWOptionButton *fontName;
@property(nonatomic, weak) HWOptionButton *formatType;
@property(nonatomic, weak) HWOptionButton *pitch;
@property(nonatomic, weak) HWOptionButton *rate;
@property(nonatomic, weak) HWOptionButton *volume;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NuiSdkUtils *utils;
@property(nonatomic) int ttsSampleRate;
@end

@implementation DashSambertTTSViewController

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static BOOL continuousPlaybackFlag = NO;
static BOOL SegmentFinishPlaying = NO;
static dispatch_queue_t tts_work_queue;

#pragma mark view controller methods

- (void)viewDidLoad {
    [super viewDidLoad];
    TLog(@"DashSambertTTSViewController did load");

    TLog(@"Get API Key: %@", self.apikey);
    TLog(@"Get URL: %@", self.url);

    [self InitView];
    
    _utils = [NuiSdkUtils alloc];

    tts_work_queue = dispatch_queue_create("NuiSambertTtsController", DISPATCH_QUEUE_CONCURRENT);

    [self NuiTtsInit];
}

-(void)dealloc {
    NSLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_tts_initialize无法覆盖已经设置的参数.
    if (_audioController != nil) {
        [_audioController cleanPlayerBuffer];
    }
    if (_nui != nil) {
        [_nui nui_tts_release];
        _nui.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
#ifdef DEBUG_TTS_DATA_SAVE
    if (fp) {
        fclose(fp);
        fp = nullptr;
    }
#endif
}

- (void)dismissKeyboard:(id)sender {
    [self.view endEditing:YES];
}

-(void)viewDidAppear:(BOOL)animated {
    NSLog(@"%s", __FUNCTION__);
    [super viewDidAppear:animated];
    [self InitView];
}

-(void)viewWillDisappear:(BOOL)animated {
    NSLog(@"%s", __FUNCTION__);

    if (_audioController != nil) {
        [_audioController cleanPlayerBuffer];
    }
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_tts_initialize无法覆盖已经设置的参数.
    if (_nui != nil) {
        [_nui nui_tts_release];
        _nui.delegate = nil;
    }
    if (_audioController != nil) {
        _audioController.delegate = nil;
    }
#ifdef DEBUG_TTS_DATA_SAVE
    if (fp) {
        fclose(fp);
        fp = nullptr;
    }
#endif
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - Audio Player Delegate
-(void)playerDidFinish {
    //播放被中止后回调。
    TLog(@"playerDidFinish");
    SegmentFinishPlaying = YES;
    if (continuousPlaybackFlag == NO) {
        TLog(@"update UI of PlayButton");
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    }
}
-(void)playerDrainDataFinish {
    //播放数据自然播放完成后回调。
    TLog(@"playerDrainDataFinish");
    SegmentFinishPlaying = YES;
    if (continuousPlaybackFlag == NO) {
        TLog(@"update UI of PlayButton");
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    }
}

#pragma mark -private methods

-(NSString *)genInitParams {
    NSString *debug_path = [_utils createDir];
    TLog(@"debug_path:%@", debug_path);

    NSMutableDictionary *ticketJsonDict = [NSMutableDictionary dictionary];

    // 注意！不推荐在这里设置长效apikey。推荐每次设置临时鉴权token。
    // 注意！不推荐在这里设置长效apikey。推荐每次设置临时鉴权token。
    // 注意！不推荐在这里设置长效apikey。推荐每次设置临时鉴权token。
//    [ticketJsonDict setObject:_apikey forKey:@"apikey"];

    if (_url == nil || _url.length == 0) {
        _url = @"wss://dashscope.aliyuncs.com/api-ws/v1/inference";
    }
    [ticketJsonDict setObject:_url forKey:@"url"];

    [ticketJsonDict setObject:@"empty_device_id" forKey:@"device_id"]; // 必填, 推荐填入具有唯一性的id, 方便定位问题

    //debug目录，当初始化SDK时的save_log参数取值为true时，该目录用于保存中间音频文件。
    [ticketJsonDict setObject:debug_path forKey:@"debug_path"];
    //过滤SDK内部日志通过回调送回到用户层
    [ticketJsonDict setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_NONE] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [ticketJsonDict setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    // 设置成在线语音合成模式, 这个设置很重要, 遗漏会导致无法运行
    [ticketJsonDict setObject:@"2" forKey:@"mode_type"]; // 必填

    NSData *data = [NSJSONSerialization dataWithJSONObject:ticketJsonDict options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

- (void)NuiTtsInit {
    if (_nui == NULL) {
        _nui = [NeoNuiTts get_instance];
        _nui.delegate = self;
    }
    //请注意此处的参数配置，其中账号相关需要按照genInitParams的说明填入后才可访问服务
    NSString * initParam = [self genInitParams];

    int retcode = [_nui nui_tts_initialize:[initParam UTF8String] logLevel:NUI_LOG_LEVEL_DEBUG saveLog:YES];
    if (retcode != 0) {
         TLog(@"init failed.retcode:%d", retcode);
         return;
     }
#ifdef DEBUG_TTS_DATA_SAVE
    NSString *sp = self.createDir;
    const char* savePath = [sp UTF8String];

    if (fp == nullptr) {
        NSString *debug_file = [NSString stringWithFormat:@"%@/tts_dump.pcm", sp];
        fp = fopen([debug_file UTF8String], "w");
    }
#endif
}

-(CGFloat)InitTextView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    // ---- textViewContent ---
    CGFloat textViewContent_width = global_size.width - border * 2;
    CGFloat textViewContent_height = global_size.height/SCREEN_HEIGHT_BASE * 200;
    CGFloat textViewContent_x = border;
    CGFloat textViewContent_y = startY;

    CGRect textViewContent_rect = CGRectMake(textViewContent_x, textViewContent_y, textViewContent_width, textViewContent_height);
    if (!textViewContent) {
        textViewContent = [[UITextView alloc] initWithFrame:textViewContent_rect];
    }
    textViewContent.layer.borderWidth = 0.6;
    textViewContent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewContent.layer.cornerRadius = 10;
    [textViewContent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewContent.scrollEnabled = YES;
    textViewContent.text = @"基于达摩院改良的自回归韵律模型，Sambert 融合了 SAMBERT+NSFGAN 深度神经网络算法与传统领域知识，提供高效的文字转语音服务。该技术具备推理速度快、合成效果卓越、读音精准、韵律自然、声音还原度高以及表现力强等优点。";
    textViewContent.textColor = [UIColor darkGrayColor];
    textViewContent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewContent];
    
    // ---- textViewEvent ---
    CGFloat textViewEvent_width = textViewContent_width;
    CGFloat textViewEvent_height = global_size.height/SCREEN_HEIGHT_BASE * 100;
    CGFloat textViewEvent_x = border;
    CGFloat textViewEvent_y = startY + textViewContent_height + border;

    CGRect textViewEvent_rect = CGRectMake(textViewEvent_x, textViewEvent_y, textViewEvent_width, textViewEvent_height);
    if (!textViewEvent) {
        textViewEvent = [[UITextView alloc] initWithFrame:textViewEvent_rect];
    }
    textViewEvent.layer.borderWidth = 0.6;
    textViewEvent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewEvent.layer.cornerRadius = 10;
    [textViewEvent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewEvent.scrollEnabled = YES;
    textViewEvent.text = @"事件展示";
    textViewEvent.textColor = [UIColor darkGrayColor];
    textViewEvent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewEvent];
    
    return textViewEvent_y + textViewEvent_height;
}

-(CGFloat)InitButtonView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    // ---- PlayButton ---
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 60;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat start_x = border;
    CGFloat start_y = startY;
    
    PlayButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PlayButton.frame = CGRectMake(start_x, start_y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
    [PlayButton setTitle:@"合成" forState:UIControlStateNormal];
    [PlayButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PlayButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PlayButton];

    // ---- PauseButton ---
    CGFloat PauseButton_x = start_x + button_width + border;
    CGFloat PauseButton_y = startY;

    PauseButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PauseButton.frame = CGRectMake(PauseButton_x, PauseButton_y, button_width, button_height);
    UIImage *PauseButton_image = [UIImage imageNamed:@"button_start"];
    [PauseButton setBackgroundImage:PauseButton_image forState:UIControlStateNormal];
    [PauseButton setTitle:@"暂停" forState:UIControlStateNormal];
    [PauseButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PauseButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PauseButton addTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PauseButton];

    // ---- ResumeButton ---
    CGFloat ResumeButton_x = PauseButton_x + button_width + border;
    CGFloat ResumeButton_y = startY;

    ResumeButton = [UIButton buttonWithType:UIButtonTypeCustom];
    ResumeButton.frame = CGRectMake(ResumeButton_x, ResumeButton_y, button_width, button_height);
    UIImage *ResumeButton_image = [UIImage imageNamed:@"button_start"];
    [ResumeButton setBackgroundImage:ResumeButton_image forState:UIControlStateNormal];
    [ResumeButton setTitle:@"恢复" forState:UIControlStateNormal];
    [ResumeButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    ResumeButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [ResumeButton addTarget:self action:@selector(resumeTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:ResumeButton];

    // ---- CancelButton ---
    CGFloat CancelButton_x = ResumeButton_x + button_width + border;
    CGFloat CancelButton_y = startY;

    CancelButton = [UIButton buttonWithType:UIButtonTypeCustom];
    CancelButton.frame = CGRectMake(CancelButton_x, CancelButton_y, button_width, button_height);
    UIImage *CancelButton_image = [UIImage imageNamed:@"button_start"];
    [CancelButton setBackgroundImage:CancelButton_image forState:UIControlStateNormal];
    [CancelButton setTitle:@"取消" forState:UIControlStateNormal];
    [CancelButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    CancelButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [CancelButton addTarget:self action:@selector(cancelTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:CancelButton];

    // ---- QuitButton ---
    CGFloat QuitButton_x = CancelButton_x + button_width + border;
    CGFloat QuitButton_y = startY;

    QuitButton = [UIButton buttonWithType:UIButtonTypeCustom];
    QuitButton.frame = CGRectMake(QuitButton_x, QuitButton_y, button_width, button_height);
    UIImage *QuitButton_image = [UIImage imageNamed:@"button_start"];
    [QuitButton setBackgroundImage:QuitButton_image forState:UIControlStateNormal];
    [QuitButton setTitle:@"取消" forState:UIControlStateNormal];
    [QuitButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    QuitButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [QuitButton addTarget:self action:@selector(quitTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:QuitButton];

    return QuitButton_y + button_height;
}

-(CGFloat)InitHWOptionsView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    CGFloat hw_height = global_size.height / SCREEN_HEIGHT_BASE * 40;

    // ---- fontName ---
    CGFloat fontName_width = global_size.width - border * 2;
    CGFloat fontName_x = border;
    CGFloat fontName_y = startY;

    HWOptionButton *fontNameBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(fontName_x, fontName_y, fontName_width, hw_height)];
    NSArray<NSString *> *voice_list = [_utils getVoiceList:@"SambertTts"];
//    NSLog(@"voice list: %@", voice_list);
    fontNameBtn.array = voice_list;
    fontNameBtn.selectIndex = 0;
    fontNameBtn.backgroundColor = [UIColor whiteColor];
    fontNameBtn.delegate = self;
    fontNameBtn.showPlaceholder = NO;
    fontNameBtn.showSearchBar = NO;
    fontNameBtn.dropdownTitle = @" - 说话人-请于官网查看更多说话人 - ";
    [self.view addSubview:fontNameBtn];
    self.fontName = fontNameBtn;
    
    // ---- formatType ---
    CGFloat formatType_width = global_size.width/2;
    CGFloat formatType_x = border;
    CGFloat formatType_y = startY + border + hw_height;

    HWOptionButton *formatTypeBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(formatType_x, formatType_y, formatType_width, hw_height)];
    formatTypeBtn.array = @[@"mp3", @"pcm", @"wav"];
    formatTypeBtn.selectIndex = 0;
    formatTypeBtn.backgroundColor = [UIColor whiteColor];
    formatTypeBtn.delegate = self;
    formatTypeBtn.showPlaceholder = NO;
    formatTypeBtn.showSearchBar = NO;
    formatTypeBtn.dropdownTitle = @" - 音频编码格式 - ";
    [self.view addSubview:formatTypeBtn];
    self.formatType = formatTypeBtn;

    // ---- pitch ---
    CGFloat pitch_width = global_size.width / 3 - border * 2;
    CGFloat pitch_x = border;
    CGFloat pitch_y = startY + hw_height;

    HWOptionButton *pitchBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(pitch_x, pitch_y, pitch_width, hw_height)];
    pitchBtn.backgroundColor = [UIColor whiteColor];
    pitchBtn.array = @[@"1.0", @"0.5", @"0.8", @"1.5", @"2.0"];
    pitchBtn.selectIndex = 0;
    pitchBtn.delegate = self;
    pitchBtn.showSearchBar = NO;
    pitchBtn.showPlaceholder = NO;
    pitchBtn.dropdownTitle = @" - 语调 - ";
    [self.view addSubview:pitchBtn];
    self.pitch = pitchBtn;
    
    // ---- rate ---
    CGFloat rate_width = global_size.width / 3 - border * 2;
    CGFloat rate_x = pitch_x + pitch_width + border;
    CGFloat rate_y = pitch_y;

    HWOptionButton *rateBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(rate_x, rate_y, rate_width, hw_height)];
    rateBtn.backgroundColor = [UIColor whiteColor];
    rateBtn.array = @[@"1.0", @"0.5", @"0.8", @"1.5", @"2.0"];
    rateBtn.selectIndex = 0;
    rateBtn.delegate = self;
    rateBtn.showSearchBar = NO;
    rateBtn.showPlaceholder = NO;
    rateBtn.dropdownTitle = @" - 语速 - ";
    [self.view addSubview:rateBtn];
    self.rate = rateBtn;
    
    // ---- volume ---
    CGFloat volume_width = global_size.width / 3 - border * 2;
    CGFloat volume_x = rate_x + rate_width + border;
    CGFloat volume_y = pitch_y;

    HWOptionButton *volumeBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(volume_x, volume_y, volume_width, hw_height)];
    volumeBtn.backgroundColor = [UIColor whiteColor];
    volumeBtn.array = @[@"50", @"10", @"30", @"70", @"80", @"100"];
    volumeBtn.selectIndex = 0;
    volumeBtn.delegate = self;
    volumeBtn.showSearchBar = NO;
    volumeBtn.showPlaceholder = NO;
    volumeBtn.dropdownTitle = @" - 音量 - ";
    [self.view addSubview:volumeBtn];
    self.volume = volumeBtn;
    
    return volume_y + hw_height;
}

-(void)InitView {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"Sambert语音合成";
    CGFloat border = global_size.width/SCREEN_WIDTH_BASE * 13;
    CGFloat division = global_size.height/SCREEN_HEIGHT_BASE * 10;
    CGFloat end_y = 0;

    CGFloat view_start_y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    end_y = [self InitTextView:border startY:view_start_y];

    CGFloat button_start_y = end_y + division;
    end_y = [self InitButtonView:border startY:button_start_y];

    CGFloat hwoptions_start_y = end_y + division;
    end_y = [self InitHWOptionsView:border startY:hwoptions_start_y];
}

-(void) UpdateTtsParams:(NSString *)content {
    NSArray<NSString *> *components = [_fontName.title componentsSeparatedByString:@";"];

    // Sambert语音合成发音人可以参考阿里云官网:
    // https://help.aliyun.com/zh/model-studio/sambert-websocket-api
    NSString *firstPart = components.firstObject;
    [self.nui nui_tts_set_param:"model" value:[firstPart UTF8String]];

    // 音频编码格式，支持pcm、wav和mp3格式。
    [self.nui nui_tts_set_param:"format" value:[_formatType.title UTF8String]];
    if ([_formatType.title isEqualToString:@"mp3"]) {
        // 若设置mp3格式, 可enable_audio_decoder打开内部解码器, 将mp3编码成pcm
        [self.nui nui_tts_set_param:"enable_audio_decoder" value:"1"];
    } else {
        [self.nui nui_tts_set_param:"enable_audio_decoder" value:"0"];
    }

    // 合成音频的采样率（单位：Hz）。
    // 建议使用模型默认采样率（参见模型列表），如果不匹配，服务会进行必要的升降采样处理。
    if (components.count > 0) {
        NSString *lastPart = components.lastObject;
        _ttsSampleRate = [lastPart intValue];
        [self.nui nui_tts_set_param:"sample_rate" value:[lastPart UTF8String]];
    }

    // 音量，取值范围：0～100。默认值：50。
    [self.nui nui_tts_set_param:"volume" value:[_volume.title UTF8String]];

    // 语速, 语速倍速区间为[0.5, 1.0, 2.0], 默认1.0
    // 0.5：表示默认语速的0.5倍速。
    // 1：表示默认语速。默认语速是指模型默认输出的合成语速，语速会依据每个发音人略有不同，约每秒钟4个字。
    // 2：表示默认语速的2倍速。
    [self.nui nui_tts_set_param:"rate" value:[_rate.title UTF8String]];

    // 合成音频的语调，取值范围：0.5~2。
    // 默认值：1.0。
    [self.nui nui_tts_set_param:"pitch" value:[_pitch.title UTF8String]];

    // 字级别音素边界功能开关。“1”表示打开，“0”表示关闭。默认关闭。
    [self.nui nui_tts_set_param:"word_timestamp_enabled" value:"1"];

    // 是否在开启字级别时间戳的基础上，显示音素级别时间戳。
    // 和word_timestamp_enabled搭配使用，“1”表示打开，“0”表示关闭。默认关闭。
//    [self.nui nui_tts_set_param:"phoneme_timestamp_enabled" value:"1"];

    // 设置文档中不存在的参数, key为custom_params, value以json string的形式设置参数
    // 如下示例传入{parameters:{"custom_param_flag":true},"user111":"111"}表示在payload下添加参数
    // payload.user111 : 111
    // payload.parameters.custom_param_flag : true
//    NSDictionary *parameters = @{@"custom_param_flag": @YES};
//    NSDictionary *custom_params = @{
//        @"user111": @"111",
//        @"parameters": parameters
//    };
//    NSError *error = nil;
//    NSData *jsonData = [NSJSONSerialization dataWithJSONObject:custom_params
//                                                       options:0
//                                                         error:&error];
//    NSString *customParamsString = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
//    [self.nui nui_tts_set_param:"custom_params" value:[customParamsString UTF8String]];

    // 打开音量回调onNuiTtsVolumeCallback
    // 注意！此音频是SDK刚收到合成数据的音量值，而非正在播放的音量值。
    // 正在播放音频的音量值可参考AudioController.m中的calculateVolumeFromPCMData
//    [self.nui nui_tts_set_param:"enable_callback_vol" value:"1"];
    
    // 设置dns超时时间, 当DNS解析超时时返回错误事件
//    [self.nui nui_tts_set_param:"dns_timeout" value:"500"];
    
    
    // 注意！！！不要在端侧使用长效API Key！！！
    // 注意！！！不要在端侧使用长效API Key！！！
    // 注意！！！不要在端侧使用长效API Key！！！
    // 将长效API Key硬编码在端侧代码中，会导致安全风险！！！
    // 请在自建服务端获得临时鉴权Token（有效期60s，最长可设置1800s），再下发到端侧进行使用。
    // 临时鉴权Token: https://help.aliyun.com/zh/model-studio/obtain-temporary-authentication-token
    //
    // 服务只需要在临时Token(API Key)快过期前刷新一次。各端侧在Token(API Key)快过期前从服务获得新的
    // 临时Token(API Key)。
    [self.nui nui_tts_set_param:"apikey" value:[_apikey UTF8String]];
}

#pragma mark - Button Action
- (IBAction)startTTS:(UIButton *)sender {
    if (!self.nui) {
        TLog(@"tts not init");
        return;
    }

    if (_nui != nil) {
        NSString *content = textViewContent.text;
        // 更新tts相关参数，尤其需要注意更新临时APIKey
        [self UpdateTtsParams:content];

        if (_audioController == nil) {
            // 注意：这里audioController模块仅用于播放示例，用户可根据业务场景自行实现这部分代码
            _audioController = [[AudioController alloc] init:only_player];
            _audioController.delegate = self;
            [_audioController setPlayerSampleRate:_ttsSampleRate];
        }

        dispatch_async(tts_work_queue, ^{
            int ret = [self.nui nui_tts_play:"1" taskId:"" text:[content UTF8String]];
            if (ret != SUCCESS) {
                const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
                TLog(@"tts pause fail(%d) with errmsg:%s ", ret, errmsg);
                self->textViewEvent.text = [NSString stringWithUTF8String:errmsg];
            }
        });
    }
}

- (IBAction)stopTTS:(UIButton *)sender {
    if (_nui != nil) {
        TLog(@"DashSambertTTSViewController stop tts");

        dispatch_async(tts_work_queue, ^{
            [self.nui nui_tts_cancel:NULL];
            if (_audioController != nil) {
                [_audioController stopPlayer];
            }
        });
    } else {
        TLog(@"in stopTTS, _nui == nil.");
    }
}

- (IBAction)pauseTTS:(UIButton *)sender {
    if (_nui != nil) {
        dispatch_async(tts_work_queue, ^{
            int ret = [self.nui nui_tts_pause];
            if (ret != SUCCESS) {
                const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
                TLog(@"tts pause fail(%d) with errmsg:%s ", ret, errmsg);
                self->textViewEvent.text = [NSString stringWithUTF8String:errmsg];
            }

            if (_audioController != nil) {
                [_audioController pausePlayer];
            }
        });
    } else {
        TLog(@"in pauseTTS, _nui == nil.");
    }
}

- (IBAction)resumeTTS:(UIButton *)sender {
    if (_nui != nil) {
        dispatch_async(tts_work_queue, ^{
            int ret = [self.nui nui_tts_resume];
            if (ret != SUCCESS) {
                const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
                TLog(@"tts resume fail(%d) with errmsg:%s ", ret, errmsg);
                self->textViewEvent.text = [NSString stringWithUTF8String:errmsg];
            }

            if (_audioController != nil) {
                [_audioController resumePlayer];
            }
        });
    } else {
        TLog(@"in resumeTTS, _nui == nil.");
    }
}

- (IBAction)quitTTS:(UIButton *)sender {
    self->textViewContent.text = @"";
}

#pragma mark - tts callback
- (void)onNuiTtsEventCallback:(NuiSdkTtsEvent)event taskId:(char*)taskid code:(int)code {
//    TLog(@"onNuiTtsEventCallback event[%d]", event);
    if (event == TTS_EVENT_START) {
        dispatch_async(dispatch_get_main_queue(), ^{
            self->textViewEvent.text = @"TTS_EVENT_START";
        });
        TLog(@"onNuiTtsEventCallback TTS_EVENT_START");
        if (_audioController != nil) {
            // 启动播放器
            [_audioController startPlayer];
        }
    } else if (event == TTS_EVENT_END || event == TTS_EVENT_CANCEL || event == TTS_EVENT_ERROR) {

        if (event == TTS_EVENT_END) {
            dispatch_async(dispatch_get_main_queue(), ^{
                self->textViewEvent.text = @"TTS_EVENT_END";
            });
            TLog(@"onNuiTtsEventCallback TTS_EVENT_END");
            // 注意这里的event事件是指语音合成完成，而非播放完成，播放完成需要由audioController对象来进行通知
            [_audioController drain];
        } else {
            TLog(@"onNuiTtsEventCallback (%d) TTS_EVENT_CANCEL or TTS_EVENT_ERROR", event);
            if (_audioController != nil) {
                // 取消播报、或者发生异常时终止播放
                [_audioController stopPlayer];
            }
        }
        if (event == TTS_EVENT_ERROR) {
            TLog(@"onNuiTtsEventCallback TTS_EVENT_ERROR with %d", code);
            const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
            TLog(@"tts get errmsg:%s", errmsg);
            dispatch_async(dispatch_get_main_queue(), ^{
                NSString *content = [NSString stringWithFormat:@"TTS_EVENT_ERROR %d: %s", code, errmsg];
                self->textViewEvent.text = content;
            });
        }
    }
}

- (void)onNuiTtsUserdataCallback:(char*)info infoLen:(int)info_len buffer:(char*)buffer len:(int)len taskId:(char*)task_id {
    if (info_len > 0) {
//        TLog(@"onNuiTtsUserdataCallback info text %s. index %d.", info, info_len);
    }
    if (len > 0 && _audioController != nil) {
//        TLog(@"onNuiTtsUserdataCallback get data %dbytes ...", len);
        [_audioController write:(char*)buffer Length:(unsigned int)len];
    }
}

-(void)onNuiTtsVolumeCallback:(int)volume taskId:(char*)task_id {
    ;
}

-(void)onNuiTtsLogTrackCallback:(NuiSdkLogLevel)level
                     logMessage:(const char *)log {
    TLog(@"onNuiTtsLogTrackCallback log level:%d, message -> %s", level, log);
}
@end
