/*
 * Decompiled with CFR 0.152.
 */
package com.aliyun.credentials.provider;

import com.aliyun.credentials.exception.CredentialException;
import com.aliyun.credentials.policy.NonBlocking;
import com.aliyun.credentials.policy.OneCallerBlocks;
import com.aliyun.credentials.policy.PrefetchStrategy;
import com.aliyun.credentials.provider.RefreshResult;
import com.aliyun.credentials.utils.ParameterHelper;
import com.aliyun.tea.logging.ClientLogger;
import com.aliyun.tea.utils.Validate;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public class RefreshCachedSupplier<T>
implements AutoCloseable {
    private static final ClientLogger logger = new ClientLogger(RefreshCachedSupplier.class);
    static final long STALE_TIME = 900000L;
    private static final long REFRESH_BLOCKING_MAX_WAIT = 5000L;
    private final Lock refreshLock = new ReentrantLock();
    final PrefetchStrategy prefetchStrategy;
    private final AtomicInteger consecutiveRefreshFailures = new AtomicInteger(0);
    final StaleValueBehavior staleValueBehavior;
    private static final Random JITTER = new Random();
    private volatile RefreshResult<T> cachedValue;
    private final Callable<RefreshResult<T>> refreshCallable;

    private RefreshCachedSupplier(Builder<T> builder) {
        this.staleValueBehavior = (StaleValueBehavior)((Object)Validate.notNull((Object)((Object)((Builder)builder).staleValueBehavior), (String)"StaleValueBehavior is null.", (Object[])new Object[0]));
        Validate.notNull((Object)((Builder)builder).jitterEnabled, (String)"JitterEnabled is null.", (Object[])new Object[0]);
        this.refreshCallable = (Callable)Validate.notNull((Object)((Builder)builder).refreshCallable, (String)"Refresh Callable is null.", (Object[])new Object[0]);
        this.prefetchStrategy = ((Builder)builder).asyncUpdateEnabled ? new NonBlocking() : new OneCallerBlocks();
    }

    public static <T> Builder<T> builder(Callable<RefreshResult<T>> refreshCallable) {
        return new Builder(refreshCallable);
    }

    public T get() {
        if (this.cacheIsStale()) {
            logger.verbose("Refreshing credentials synchronously");
            this.refreshCache();
        } else if (this.shouldInitiateCachePrefetch()) {
            logger.verbose("Prefetching credentials, using prefetch strategy: {}", new Object[]{this.prefetchStrategy.toString()});
            this.prefetchCache();
        } else {
            logger.verbose("get local credentials");
        }
        return this.cachedValue.value();
    }

    private void prefetchCache() {
        this.prefetchStrategy.prefetch(this::refreshCache);
    }

    private void refreshCache() {
        try {
            boolean lockAcquired = this.refreshLock.tryLock(5000L, TimeUnit.MILLISECONDS);
            try {
                if (this.cacheIsStale() || this.shouldInitiateCachePrefetch()) {
                    try {
                        this.cachedValue = this.handleFetchedSuccess(this.refreshCallable.call());
                    }
                    catch (Exception ex) {
                        this.cachedValue = this.handleFetchedFailure(ex);
                    }
                }
            }
            finally {
                if (lockAcquired) {
                    this.refreshLock.unlock();
                }
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Interrupted waiting to refresh the value.", ex);
        }
        catch (CredentialException ex) {
            throw ex;
        }
        catch (Exception e) {
            throw new CredentialException("Failed to refresh credentials.", e);
        }
    }

    @Override
    public void close() {
        this.prefetchStrategy.close();
    }

    private boolean cacheIsStale() {
        return this.cachedValue == null || new Date().getTime() >= this.cachedValue.staleTime();
    }

    private boolean shouldInitiateCachePrefetch() {
        return this.cachedValue == null || new Date().getTime() >= this.cachedValue.prefetchTime();
    }

    RefreshResult<T> handleFetchedSuccess(RefreshResult<T> value) {
        logger.verbose("Refresh credentials successfully, retrieved value is {}, cached value is {}", new Object[]{value, this.cachedValue});
        this.consecutiveRefreshFailures.set(0);
        long now = new Date().getTime();
        if (now < value.staleTime()) {
            logger.verbose("Retrieved value stale time is {}. Using staleTime of {}", new Object[]{ParameterHelper.getTimeString(value.staleTime()), ParameterHelper.getTimeString(value.staleTime())});
            return value;
        }
        if (now < value.staleTime() + 900000L) {
            logger.warning("Retrieved value stale time is in the past ({}). Using staleTime of {}", new Object[]{ParameterHelper.getTimeString(value.staleTime()), ParameterHelper.getTimeString(now)});
            return value.toBuilder().staleTime(now).build();
        }
        logger.warning("Retrieved value expiration time of the credential is in the past ({}). Trying use the cached value.", new Object[]{ParameterHelper.getTimeString(value.staleTime() + 900000L)});
        if (null == this.cachedValue) {
            throw new CredentialException("No cached value was found.");
        }
        if (now < this.cachedValue.staleTime()) {
            logger.warning("Cached value staleTime is {}. Using staleTime of {}", new Object[]{ParameterHelper.getTimeString(this.cachedValue.staleTime()), ParameterHelper.getTimeString(this.cachedValue.staleTime())});
            return this.cachedValue;
        }
        switch (this.staleValueBehavior) {
            case STRICT: {
                logger.warning("Cached value expiration is in the past (" + this.cachedValue.staleTime() + "). Using expiration of " + (now + 1000L));
                return this.cachedValue.toBuilder().staleTime(now + 1000L).build();
            }
            case ALLOW: {
                long waitUntilNextRefresh = 50000 + JITTER.nextInt(20001);
                long nextRefreshTime = now + waitUntilNextRefresh;
                logger.warning("Cached value expiration has been extended to " + nextRefreshTime + " because the downstream service returned a time in the past: " + value.staleTime());
                return this.cachedValue.toBuilder().staleTime(nextRefreshTime).build();
            }
        }
        throw new IllegalStateException("Unknown stale-value-behavior: " + (Object)((Object)this.staleValueBehavior));
    }

    RefreshResult<T> handleFetchedFailure(Exception exception) throws Exception {
        logger.warning("Refresh credentials failed, cached value is {}, error: {}", new Object[]{this.cachedValue, exception.getMessage()});
        RefreshResult<T> currentCachedValue = this.cachedValue;
        if (currentCachedValue == null) {
            throw (Exception)logger.logThrowableAsError((Throwable)exception);
        }
        long now = new Date().getTime();
        if (now < currentCachedValue.staleTime()) {
            return currentCachedValue;
        }
        int numFailures = this.consecutiveRefreshFailures.incrementAndGet();
        switch (this.staleValueBehavior) {
            case STRICT: {
                throw (Exception)logger.logThrowableAsError((Throwable)exception);
            }
            case ALLOW: {
                long newStaleTime = this.jitterTime(now, 1000L, this.maxStaleFailureJitter(numFailures));
                logger.warning("Cached value expiration has been extended to " + newStaleTime + " because calling the downstream service failed (consecutive failures: " + numFailures + ").");
                return currentCachedValue.toBuilder().staleTime(newStaleTime).build();
            }
        }
        throw new IllegalStateException("Unknown stale-value-behavior: " + (Object)((Object)this.staleValueBehavior));
    }

    long jitterTime(long time, long jitterStart, long jitterEnd) {
        long jitterRange = jitterEnd - jitterStart;
        long jitterAmount = Math.abs(JITTER.nextLong() % jitterRange);
        return time + jitterStart + jitterAmount;
    }

    long maxStaleFailureJitter(int numFailures) {
        long exponentialBackoffMillis = (1L << numFailures - 1) * 100L;
        return exponentialBackoffMillis > 10000L ? exponentialBackoffMillis : 10000L;
    }

    public static enum StaleValueBehavior {
        STRICT,
        ALLOW;

    }

    public static final class Builder<T> {
        private final Callable<RefreshResult<T>> refreshCallable;
        private boolean asyncUpdateEnabled;
        private Boolean jitterEnabled = true;
        private StaleValueBehavior staleValueBehavior = StaleValueBehavior.STRICT;

        private Builder(Callable<RefreshResult<T>> refreshCallable) {
            this.refreshCallable = refreshCallable;
        }

        public Builder<T> asyncUpdateEnabled(Boolean asyncUpdateEnabled) {
            this.asyncUpdateEnabled = asyncUpdateEnabled;
            return this;
        }

        public Builder<T> staleValueBehavior(StaleValueBehavior staleValueBehavior) {
            this.staleValueBehavior = staleValueBehavior;
            return this;
        }

        Builder<T> jitterEnabled(Boolean jitterEnabled) {
            this.jitterEnabled = jitterEnabled;
            return this;
        }

        public RefreshCachedSupplier<T> build() {
            return new RefreshCachedSupplier(this);
        }
    }
}

