/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.catalog;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nullable;
import org.apache.paimon.annotation.VisibleForTesting;
import org.apache.paimon.catalog.CachingCatalogLoader;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.CatalogLoader;
import org.apache.paimon.catalog.Database;
import org.apache.paimon.catalog.DelegateCatalog;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.catalog.PropertyChange;
import org.apache.paimon.fs.Path;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.MemorySize;
import org.apache.paimon.options.Options;
import org.apache.paimon.partition.Partition;
import org.apache.paimon.partition.PartitionStatistics;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Cache;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Ticker;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.system.SystemTableLoader;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.SegmentsCache;

public class CachingCatalog
extends DelegateCatalog {
    private final Options options;
    private final Duration expireAfterAccess;
    private final Duration expireAfterWrite;
    private final int snapshotMaxNumPerTable;
    private final long cachedPartitionMaxNum;
    protected Cache<String, Database> databaseCache;
    protected Cache<Identifier, Table> tableCache;
    @Nullable
    protected final SegmentsCache<Path> manifestCache;
    @Nullable
    protected Cache<Identifier, List<Partition>> partitionCache;

    public CachingCatalog(Catalog wrapped, Options options) {
        super(wrapped);
        this.options = options;
        MemorySize manifestMaxMemory = options.get(CatalogOptions.CACHE_MANIFEST_SMALL_FILE_MEMORY);
        long manifestCacheThreshold = options.get(CatalogOptions.CACHE_MANIFEST_SMALL_FILE_THRESHOLD).getBytes();
        Optional<MemorySize> maxMemory = options.getOptional(CatalogOptions.CACHE_MANIFEST_MAX_MEMORY);
        if (maxMemory.isPresent() && maxMemory.get().compareTo(manifestMaxMemory) > 0) {
            manifestMaxMemory = maxMemory.get();
            manifestCacheThreshold = Long.MAX_VALUE;
        }
        this.expireAfterAccess = options.get(CatalogOptions.CACHE_EXPIRE_AFTER_ACCESS);
        if (this.expireAfterAccess.isZero() || this.expireAfterAccess.isNegative()) {
            throw new IllegalArgumentException("When 'cache.expire-after-access' is set to negative or 0, the catalog cache should be disabled.");
        }
        this.expireAfterWrite = options.get(CatalogOptions.CACHE_EXPIRE_AFTER_WRITE);
        if (this.expireAfterWrite.isZero() || this.expireAfterWrite.isNegative()) {
            throw new IllegalArgumentException("When 'cache.expire-after-write' is set to negative or 0, the catalog cache should be disabled.");
        }
        this.snapshotMaxNumPerTable = options.get(CatalogOptions.CACHE_SNAPSHOT_MAX_NUM_PER_TABLE);
        this.manifestCache = SegmentsCache.create(manifestMaxMemory, manifestCacheThreshold);
        this.cachedPartitionMaxNum = options.get(CatalogOptions.CACHE_PARTITION_MAX_NUM);
        this.init(Ticker.systemTicker());
    }

    @VisibleForTesting
    void init(Ticker ticker) {
        this.databaseCache = Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(this.expireAfterAccess).expireAfterWrite(this.expireAfterWrite).ticker(ticker).build();
        this.tableCache = Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(this.expireAfterAccess).expireAfterWrite(this.expireAfterWrite).ticker(ticker).build();
        this.partitionCache = this.cachedPartitionMaxNum == 0L ? null : Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(this.expireAfterAccess).expireAfterWrite(this.expireAfterWrite).weigher((identifier, v) -> v.size()).maximumWeight(this.cachedPartitionMaxNum).ticker(ticker).build();
    }

    public static Catalog tryToCreate(Catalog catalog, Options options) {
        if (!options.get(CatalogOptions.CACHE_ENABLED).booleanValue()) {
            return catalog;
        }
        return new CachingCatalog(catalog, options);
    }

    @Override
    public CatalogLoader catalogLoader() {
        return new CachingCatalogLoader(this.wrapped.catalogLoader(), this.options);
    }

    @Override
    public Database getDatabase(String databaseName) throws Catalog.DatabaseNotExistException {
        Database database = this.databaseCache.getIfPresent(databaseName);
        if (database != null) {
            return database;
        }
        database = super.getDatabase(databaseName);
        this.databaseCache.put(databaseName, database);
        return database;
    }

    @Override
    public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade) throws Catalog.DatabaseNotExistException, Catalog.DatabaseNotEmptyException {
        super.dropDatabase(name, ignoreIfNotExists, cascade);
        this.databaseCache.invalidate(name);
        if (cascade) {
            ArrayList<Identifier> tables = new ArrayList<Identifier>();
            for (Identifier identifier : this.tableCache.asMap().keySet()) {
                if (!identifier.getDatabaseName().equals(name)) continue;
                tables.add(identifier);
            }
            tables.forEach(this.tableCache::invalidate);
        }
    }

    @Override
    public void alterDatabase(String name, List<PropertyChange> changes, boolean ignoreIfNotExists) throws Catalog.DatabaseNotExistException {
        super.alterDatabase(name, changes, ignoreIfNotExists);
        this.databaseCache.invalidate(name);
    }

    @Override
    public void dropTable(Identifier identifier, boolean ignoreIfNotExists) throws Catalog.TableNotExistException {
        super.dropTable(identifier, ignoreIfNotExists);
        this.invalidateTable(identifier);
        for (Identifier i : this.tableCache.asMap().keySet()) {
            if (!identifier.getTableName().equals(i.getTableName()) || !identifier.getDatabaseName().equals(i.getDatabaseName())) continue;
            this.tableCache.invalidate(i);
        }
    }

    @Override
    public void renameTable(Identifier fromTable, Identifier toTable, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.TableAlreadyExistException {
        super.renameTable(fromTable, toTable, ignoreIfNotExists);
        this.invalidateTable(fromTable);
    }

    @Override
    public void alterTable(Identifier identifier, List<SchemaChange> changes, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException {
        super.alterTable(identifier, changes, ignoreIfNotExists);
        this.invalidateTable(identifier);
    }

    @Override
    public Table getTable(Identifier identifier) throws Catalog.TableNotExistException {
        Table table = this.tableCache.getIfPresent(identifier);
        if (table != null) {
            return table;
        }
        if (identifier.isSystemTable()) {
            Identifier originIdentifier = new Identifier(identifier.getDatabaseName(), identifier.getTableName(), identifier.getBranchName(), null);
            Table originTable = this.getTable(originIdentifier);
            table = SystemTableLoader.load(Preconditions.checkNotNull(identifier.getSystemTableName()), (FileStoreTable)originTable);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        table = this.wrapped.getTable(identifier);
        this.putTableCache(identifier, table);
        return table;
    }

    private void putTableCache(Identifier identifier, Table table) {
        if (table instanceof FileStoreTable) {
            FileStoreTable storeTable = (FileStoreTable)table;
            storeTable.setSnapshotCache(Caffeine.newBuilder().softValues().expireAfterAccess(this.expireAfterAccess).expireAfterWrite(this.expireAfterWrite).maximumSize(this.snapshotMaxNumPerTable).executor(Runnable::run).build());
            storeTable.setStatsCache(Caffeine.newBuilder().softValues().expireAfterAccess(this.expireAfterAccess).expireAfterWrite(this.expireAfterWrite).maximumSize(5L).executor(Runnable::run).build());
            if (this.manifestCache != null) {
                storeTable.setManifestCache(this.manifestCache);
            }
        }
        this.tableCache.put(identifier, table);
    }

    @Override
    public List<Partition> listPartitions(Identifier identifier) throws Catalog.TableNotExistException {
        if (this.partitionCache == null) {
            return this.wrapped.listPartitions(identifier);
        }
        List<Partition> result = this.partitionCache.getIfPresent(identifier);
        if (result == null) {
            result = this.wrapped.listPartitions(identifier);
            this.partitionCache.put(identifier, result);
        }
        return result;
    }

    @Override
    public void dropPartitions(Identifier identifier, List<Map<String, String>> partitions) throws Catalog.TableNotExistException {
        this.wrapped.dropPartitions(identifier, partitions);
        if (this.partitionCache != null) {
            this.partitionCache.invalidate(identifier);
        }
    }

    @Override
    public void alterPartitions(Identifier identifier, List<PartitionStatistics> partitions) throws Catalog.TableNotExistException {
        this.wrapped.alterPartitions(identifier, partitions);
        if (this.partitionCache != null) {
            this.partitionCache.invalidate(identifier);
        }
    }

    @Override
    public void invalidateTable(Identifier identifier) {
        this.tableCache.invalidate(identifier);
        if (this.partitionCache != null) {
            this.partitionCache.invalidate(identifier);
        }
    }

    public void refreshPartitions(Identifier identifier) throws Catalog.TableNotExistException {
        if (this.partitionCache != null) {
            List<Partition> result = this.wrapped.listPartitions(identifier);
            this.partitionCache.put(identifier, result);
        }
    }

    public CacheSizes estimatedCacheSizes() {
        long databaseCacheSize = this.databaseCache.estimatedSize();
        long tableCacheSize = this.tableCache.estimatedSize();
        long manifestCacheSize = 0L;
        long manifestCacheBytes = 0L;
        if (this.manifestCache != null) {
            manifestCacheSize = this.manifestCache.estimatedSize();
            manifestCacheBytes = this.manifestCache.totalCacheBytes();
        }
        long partitionCacheSize = 0L;
        if (this.partitionCache != null) {
            for (Map.Entry entry : this.partitionCache.asMap().entrySet()) {
                partitionCacheSize += (long)((List)entry.getValue()).size();
            }
        }
        return new CacheSizes(databaseCacheSize, tableCacheSize, manifestCacheSize, manifestCacheBytes, partitionCacheSize);
    }

    public static class CacheSizes {
        private final long databaseCacheSize;
        private final long tableCacheSize;
        private final long manifestCacheSize;
        private final long manifestCacheBytes;
        private final long partitionCacheSize;

        public CacheSizes(long databaseCacheSize, long tableCacheSize, long manifestCacheSize, long manifestCacheBytes, long partitionCacheSize) {
            this.databaseCacheSize = databaseCacheSize;
            this.tableCacheSize = tableCacheSize;
            this.manifestCacheSize = manifestCacheSize;
            this.manifestCacheBytes = manifestCacheBytes;
            this.partitionCacheSize = partitionCacheSize;
        }

        public long databaseCacheSize() {
            return this.databaseCacheSize;
        }

        public long tableCacheSize() {
            return this.tableCacheSize;
        }

        public long manifestCacheSize() {
            return this.manifestCacheSize;
        }

        public long manifestCacheBytes() {
            return this.manifestCacheBytes;
        }

        public long partitionCacheSize() {
            return this.partitionCacheSize;
        }
    }
}

