//
//  StreamInputTtsViewController.m
//  NUIdemo
//
//  Created by lengjiayi on 2024/4/17.
//  Copyright © 2024 Alibaba idst. All rights reserved.
//

#import "nuisdk.framework/Headers/StreamInputTts.h"
#import "StreamInputTtsViewController.h"
#import "NuiSdkUtils.h"

#import "audio/AudioController.h"
#import <AudioToolbox/AudioToolbox.h>
#import <AVFoundation/AVFoundation.h>

static StreamInputTtsViewController *myself = nil;

@interface StreamInputTtsViewController ()<ConvVoiceRecorderDelegate, UITextFieldDelegate, StreamInputTtsDelegate> {
    UIButton *buttonStart;
    UIButton *buttonPlay;
    UIButton *buttonSend;
    UIButton *buttonStop;
    UIButton *buttonCancel;
    UIButton *buttonNew;
}

@property(nonatomic,strong) StreamInputTts* streamInputTtsSdk;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NuiSdkUtils *utils;
@property(nonatomic) int ttsSampleRate;
@end

@implementation StreamInputTtsViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    
    _streamInputTtsSdk = [StreamInputTts get_instance];
    _streamInputTtsSdk.delegate = self;
    myself = self;
    
    _utils = [NuiSdkUtils alloc];

    [self setupTextView];
    [self setupTextField];
    [self setupButtonGrid];
    [self setupNoteTextView];
}

-(void)dealloc {
    TLog(@"%s", __FUNCTION__);
    if (_audioController != nil) {
        [_audioController cleanPlayerBuffer];
    }
}

- (void)setupTextView {
    self.logTextView = [[UITextView alloc] initWithFrame:CGRectMake(20, 80, self.view.bounds.size.width - 40, 200)];
    self.logTextView.editable = NO; // 设置为只读以用作日志显示
    self.logTextView.text = [self.logTextView.text stringByAppendingString:@"输出日志"];
    self.logTextView.layer.borderColor = [[UIColor blackColor] CGColor];
    self.logTextView.layer.borderWidth = 1.0;
    [self.view addSubview:self.logTextView];
    self.logTextView.delegate = self;
}

- (void)setupTextField {
    self.inputTextField = [[UITextView alloc] initWithFrame:CGRectMake(20, CGRectGetMaxY(self.logTextView.frame) + 20, self.view.bounds.size.width - 40, 200)];
    self.inputTextField.text = @"唧唧复唧唧，木兰当户织。\n不闻机杼声，惟闻女叹息。\n问女何所思，问女何所忆。\n女亦无所思，女亦无所忆。\n昨夜见军帖，可汗大点兵。\n军书十二卷，卷卷有爷名。\n阿爷无大儿，木兰无长兄。\n愿为市鞍马，从此替爷征。";
    self.inputTextField.borderStyle = UITextBorderStyleRoundedRect;
    self.inputTextField.layer.borderColor = [[UIColor blackColor] CGColor];
    self.inputTextField.layer.borderWidth = 1.0;
    [self.view addSubview:self.inputTextField];
    self.inputTextField.delegate = self;
}

- (void)setupButtonGrid {
    const int columns = 4; // 按键数量
    CGFloat buttonWidth = 100.0;
    CGFloat buttonHeight = 50.0;
    CGFloat padding = 10.0;
    CGFloat startY = CGRectGetMaxY(self.inputTextField.frame) + 20;
    CGFloat startX = (self.view.bounds.size.width - (buttonWidth + padding) * columns + padding) / 2.0;

    buttonNew = [UIButton buttonWithType:UIButtonTypeSystem];
    [buttonNew setTitle:[NSString stringWithFormat:@"新实例"] forState:UIControlStateNormal];
    [buttonNew setFrame:CGRectMake(startX + (buttonWidth + padding) * 1, startY + (buttonHeight + padding) * 0, buttonWidth, buttonHeight)];
    [buttonNew addTarget:self action:@selector(newInstancePressed:) forControlEvents:UIControlEventTouchUpInside];
    buttonNew.tag = 0;
    [self.view addSubview:buttonNew];
    
    
    buttonStart = [UIButton buttonWithType:UIButtonTypeSystem];
    [buttonStart setTitle:[NSString stringWithFormat:@"开始"] forState:UIControlStateNormal];
    [buttonStart setFrame:CGRectMake(startX + (buttonWidth + padding) * 0, startY + (buttonHeight + padding) * 1, buttonWidth, buttonHeight)];
    [buttonStart addTarget:self action:@selector(startPressed:) forControlEvents:UIControlEventTouchUpInside];
    buttonStart.tag = 1;
    [self.view addSubview:buttonStart];
    
    buttonPlay = [UIButton buttonWithType:UIButtonTypeSystem];
    [buttonPlay setTitle:[NSString stringWithFormat:@"单句合成"] forState:UIControlStateNormal];
    [buttonPlay setFrame:CGRectMake(startX + (buttonWidth + padding) * 1, startY + (buttonHeight + padding) * 1, buttonWidth, buttonHeight)];
    [buttonPlay addTarget:self action:@selector(playPressed:) forControlEvents:UIControlEventTouchUpInside];
    buttonPlay.tag = 2;
    [self.view addSubview:buttonPlay];
    
    buttonSend = [UIButton buttonWithType:UIButtonTypeSystem];
    [buttonSend setTitle:[NSString stringWithFormat:@"发送"] forState:UIControlStateNormal];
    [buttonSend setFrame:CGRectMake(startX + (buttonWidth + padding) * 2, startY + (buttonHeight + padding) * 1, buttonWidth, buttonHeight)];
    [buttonSend addTarget:self action:@selector(sendPressed:) forControlEvents:UIControlEventTouchUpInside];
    buttonSend.tag = 3;
    [self.view addSubview:buttonSend];
    
    buttonStop = [UIButton buttonWithType:UIButtonTypeSystem];
    [buttonStop setTitle:[NSString stringWithFormat:@"停止"] forState:UIControlStateNormal];
    [buttonStop setFrame:CGRectMake(startX + (buttonWidth + padding) * 3, startY + (buttonHeight + padding) * 1, buttonWidth, buttonHeight)];
    [buttonStop addTarget:self action:@selector(stopPressed:) forControlEvents:UIControlEventTouchUpInside];
    buttonStop.tag = 4;
    [self.view addSubview:buttonStop];
    
    buttonCancel = [UIButton buttonWithType:UIButtonTypeSystem];
    [buttonCancel setTitle:[NSString stringWithFormat:@"取消"] forState:UIControlStateNormal];
    [buttonCancel setFrame:CGRectMake(startX + (buttonWidth + padding) * 4, startY + (buttonHeight + padding) * 1, buttonWidth, buttonHeight)];
    [buttonCancel addTarget:self action:@selector(cancelPressed:) forControlEvents:UIControlEventTouchUpInside];
    buttonCancel.tag = 5;
    [self.view addSubview:buttonCancel];
}

- (void)setupNoteTextView {
    CGFloat buttonHeight = 50.0;
    CGFloat padding = 10.0;
    CGFloat startY = CGRectGetMaxY(self.inputTextField.frame) + 20 + (buttonHeight + padding) * 2;
    
    self.noteTextField = [[UITextView alloc] initWithFrame:CGRectMake(20, startY, self.view.bounds.size.width - 40, 130)];
    self.noteTextField.editable = NO; // 设置为只读以用作日志显示
    self.noteTextField.text = [self.noteTextField.text stringByAppendingString:@"点击<开始>\n连续点击<发送>逐行发送合成文本\n点击<停止>表示合成完成\n注意！<停止>前超过10s不发送任何文本会收到超时错误信息\n\n 144500错误码表示超时退出或者已经停止工作时仍然发送合成问题，即工作状态无效"];
    self.noteTextField.layer.borderColor = [[UIColor grayColor] CGColor];
    self.noteTextField.layer.borderWidth = 1.0;
    [self.view addSubview:self.noteTextField];
    self.noteTextField.delegate = self;
}

- (void)newInstancePressed:(UIButton *)sender {
    if (_audioController != nil) {
        [_audioController stopPlayer];
    }

    self.logTextView.text = [self.logTextView.text stringByAppendingString:@"\n按下新实例按钮，播放器停止"];
    TLog(@"\n按下新实例按钮，播放器停止");
    self.inputTextField.text = @"唧唧复唧唧，木兰当户织。\n不闻机杼声，惟闻女叹息。\n问女何所思，问女何所忆。\n女亦无所思，女亦无所忆。\n昨夜见军帖，可汗大点兵。\n军书十二卷，卷卷有爷名。\n阿爷无大儿，木兰无长兄。\n愿为市鞍马，从此替爷征。";
    _streamInputTtsSdk = [StreamInputTts get_instance];
    _streamInputTtsSdk.delegate = self;
    NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
}

#pragma mark - Audio Player Delegate
-(void)playerDidFinish {
    //播放被中止后回调。
    TLog(@"playerDidFinish");
}
-(void)playerDrainDataFinish {
    //播放数据自然播放完成后回调。
    TLog(@"playerDrainDataFinish");
}
#pragma mark -private methods

- (void)startPressed:(UIButton *)sender {
    if (_audioController == nil) {
        // 注意：这里audioController模块仅用于播放示例，用户可根据业务场景自行实现这部分代码
        _audioController = [[AudioController alloc] init:only_player];
        _audioController.delegate = self;
        _ttsSampleRate = 16000;  // 返回音频的采样率
        [_audioController setPlayerSampleRate:_ttsSampleRate];
    }

    if (_audioController == nil) {
        return;
    }

    self.logTextView.text = [self.logTextView.text stringByAppendingString:@"\n按下开始按钮"];
    TLog(@"\n按下开始按钮");
    
    NSMutableDictionary *ticketJsonDict = [NSMutableDictionary dictionary];
    [ticketJsonDict setObject:@"wss://nls-gateway-cn-beijing.aliyuncs.com/ws/v1" forKey:@"url"]; // 必填
    [ticketJsonDict setObject:@"10000" forKey:@"complete_waiting_ms"];

    //debug目录，当初始化SDK时的saveLog参数取值为YES时，该目录用于保存日志等调试信息
    NSString *debug_path = [_utils createDir];
    [ticketJsonDict setObject:debug_path forKey:@"debug_path"];
    //过滤SDK内部日志通过回调送回到用户层
    [ticketJsonDict setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_INFO] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [ticketJsonDict setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    //获取账号访问凭证：
    [_utils getTicket:ticketJsonDict Type:get_token_from_server_for_online_features];
    if ([ticketJsonDict objectForKey:@"token"] != nil) {
        NSString *tokenValue = [ticketJsonDict objectForKey:@"token"];
        if ([tokenValue length] == 0) {
            TLog(@"The 'token' key exists but the value is empty.");
        }
    } else {
        TLog(@"The 'token' key does not exist.");
    }
    
    NSError *error;
    NSData *ticketJsonData = [NSJSONSerialization dataWithJSONObject:ticketJsonDict options:0 error:&error];
    NSString *ticket = [[NSString alloc] initWithData:ticketJsonData encoding:NSUTF8StringEncoding];
    self.logTextView.text = [self.logTextView.text stringByAppendingString:ticket];
    TLog(ticket);

    // 接口说明：https://help.aliyun.com/zh/isi/developer-reference/interface-description
    NSDictionary *paramsJsonDict = @{
        @"voice": @"zhixiaoxia",
        @"format": @"pcm",
        @"sample_rate": @(16000),
        @"volume": @(50),
        @"speech_rate": @(0),
        @"pitch_rate": @(0),
        @"enable_subtitle": @(YES)
    };
    NSData *paramsJsonData = [NSJSONSerialization dataWithJSONObject:paramsJsonDict options:0 error:&error];
    NSString *parameters = [[NSString alloc] initWithData:paramsJsonData encoding:NSUTF8StringEncoding];
    self.logTextView.text = [self.logTextView.text stringByAppendingString:parameters];
    TLog(parameters);
    
    int ret = [_streamInputTtsSdk startStreamInputTts:[ticket UTF8String] parameters:[parameters UTF8String] sessionId:nil logLevel:NUI_LOG_LEVEL_VERBOSE saveLog:YES];
    NSString *retLog = [NSString stringWithFormat:@"\n开始 返回值：%d", ret];
    TLog(retLog);
    
    self.logTextView.text = [self.logTextView.text stringByAppendingString:retLog];
    NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
}

- (void)playPressed:(UIButton *)sender {
    if (_audioController == nil) {
        // 注意：这里audioController模块仅用于播放示例，用户可根据业务场景自行实现这部分代码
        _audioController = [[AudioController alloc] init:only_player];
        _audioController.delegate = self;
        _ttsSampleRate = 16000;  // 返回音频的采样率
        [_audioController setPlayerSampleRate:_ttsSampleRate];
    }

    if (_audioController == nil) {
        return;
    }

    self.logTextView.text = [self.logTextView.text stringByAppendingString:@"\n按下开始按钮"];
    TLog(@"\n按下开始按钮");
    
    NSMutableDictionary *ticketJsonDict = [NSMutableDictionary dictionary];
    [ticketJsonDict setObject:@"wss://nls-gateway-cn-beijing.aliyuncs.com/ws/v1" forKey:@"url"]; // 必填
    [ticketJsonDict setObject:@"10000" forKey:@"complete_waiting_ms"];

    //debug目录，当初始化SDK时的saveLog参数取值为YES时，该目录用于保存日志等调试信息
    NSString *debug_path = [_utils createDir];
    [ticketJsonDict setObject:debug_path forKey:@"debug_path"];
    //过滤SDK内部日志通过回调送回到用户层
    [ticketJsonDict setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_INFO] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [ticketJsonDict setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    //获取账号访问凭证：
    [_utils getTicket:ticketJsonDict Type:get_token_from_server_for_online_features];
    if ([ticketJsonDict objectForKey:@"token"] != nil) {
        NSString *tokenValue = [ticketJsonDict objectForKey:@"token"];
        if ([tokenValue length] == 0) {
            TLog(@"The 'token' key exists but the value is empty.");
        }
    } else {
        TLog(@"The 'token' key does not exist.");
    }
    
    NSError *error;
    NSData *ticketJsonData = [NSJSONSerialization dataWithJSONObject:ticketJsonDict options:0 error:&error];
    NSString *ticket = [[NSString alloc] initWithData:ticketJsonData encoding:NSUTF8StringEncoding];
    self.logTextView.text = [self.logTextView.text stringByAppendingString:ticket];
    TLog(ticket);

    // 接口说明：https://help.aliyun.com/zh/isi/developer-reference/interface-description
    NSDictionary *paramsJsonDict = @{
        @"voice": @"longhua_v2",
        @"format": @"mp3",
        @"enable_audio_decoder": @(YES), // 若设置mp3格式, 可enable_audio_decoder打开内部解码器, 将mp3编码成pcm
        @"sample_rate": @(16000),
        @"volume": @(50),
        @"speech_rate": @(0),
        @"pitch_rate": @(0),
        @"enable_subtitle": @(YES)
    };
    NSData *paramsJsonData = [NSJSONSerialization dataWithJSONObject:paramsJsonDict options:0 error:&error];
    NSString *parameters = [[NSString alloc] initWithData:paramsJsonData encoding:NSUTF8StringEncoding];
    self.logTextView.text = [self.logTextView.text stringByAppendingString:parameters];
    TLog(parameters);
    
    NSString *inputContents = _inputTextField.text;
    int ret = [_streamInputTtsSdk asyncPlayStreamInputTts:[ticket UTF8String] parameters:[parameters UTF8String] text:[inputContents UTF8String] sessionId:nil logLevel:NUI_LOG_LEVEL_VERBOSE saveLog:YES];
    NSString *retLog = [NSString stringWithFormat:@"\n开始 返回值：%d", ret];
    TLog(retLog);
    
    self.logTextView.text = [self.logTextView.text stringByAppendingString:retLog];
    NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
}

- (void)sendPressed:(UIButton *)sender {
    self.logTextView.text = [self.logTextView.text stringByAppendingString:@"\n按下发送按钮"];
    NSString *inputContents = _inputTextField.text;
    NSArray<NSString *> *lines = [inputContents componentsSeparatedByCharactersInSet:[NSCharacterSet newlineCharacterSet]];
    int line_num = [lines count];
    TLog(@"\n按下发送按钮，发送一行文本（%d）", line_num);
    if (line_num > 0) {
        NSString * oneLine = [lines firstObject];
        NSString *retLog;
        int ret = [_streamInputTtsSdk sendStreamInputTts:[oneLine UTF8String]];
        retLog = [NSString stringWithFormat:@"\n发送：%@\n 发送返回值：%d", oneLine, ret];
        self.logTextView.text = [self.logTextView.text stringByAppendingString:retLog];
        NSString *updatedText = @"";
        if (line_num > 1) {
            updatedText = [[lines subarrayWithRange:NSMakeRange(1, [lines count] - 1)] componentsJoinedByString:@"\n"];
        }
        dispatch_async(dispatch_get_main_queue(), ^{
            // 更新textField中的文本
            if (updatedText.length == 0) {
                self.inputTextField.text = @"唧唧复唧唧，木兰当户织。\n不闻机杼声，惟闻女叹息。\n问女何所思，问女何所忆。\n女亦无所思，女亦无所忆。\n昨夜见军帖，可汗大点兵。\n军书十二卷，卷卷有爷名。\n阿爷无大儿，木兰无长兄。\n愿为市鞍马，从此替爷征。";
            } else {
                _inputTextField.text = updatedText;
            }
        });
    }
    NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
}


// stop为阻塞接口，因此在新线程中使用。但是asycStop不阻塞。
- (void)stopPressed:(UIButton *)sender {
    self.logTextView.text = [self.logTextView.text stringByAppendingString:@"\n按下停止按钮"];
    NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
    TLog(@"\n按下停止按钮");
//    int ret = [_streamInputTtsSdk stopStreamInputTts]; // 阻塞
    int ret = [_streamInputTtsSdk asyncStopStreamInputTts]; // 非阻塞
    NSString *retLog = [NSString stringWithFormat:@"\n停止 返回值：%d", ret];
    TLog(retLog);
    self.logTextView.text = [self.logTextView.text stringByAppendingString:retLog];
    bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
}

// 立即取消当前合成
- (void)cancelPressed:(UIButton *)sender {
    self.logTextView.text = [self.logTextView.text stringByAppendingString:@"\n按下停止按钮"];
    NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
    TLog(@"\n按下取消按钮");
    int ret = [_streamInputTtsSdk cancelStreamInputTts];
    NSString *retLog = [NSString stringWithFormat:@"\n取消 返回值：%d", ret];
    TLog(retLog);
    if (_audioController != nil) {
        [_audioController stopPlayer];
    }
    self.logTextView.text = [self.logTextView.text stringByAppendingString:retLog];
    bottom = NSMakeRange(self.logTextView.text.length -1, 1);
    [self.logTextView scrollRangeToVisible:bottom];
}

#pragma stream input tts callback

- (void)onStreamInputTtsEventCallback:(StreamInputTtsCallbackEvent)event taskId:(char*)taskid sessionId:(char*)sessionId ret_code:(int)ret_code error_msg:(char*)error_msg timestamp:(char*)timestamp all_response:(char*)all_response {
    NSString *log = [NSString stringWithFormat:@"\n事件回调（%d）：%s", event, all_response];
    TLog(@"%@", log);
    
    dispatch_async(dispatch_get_main_queue(), ^{
        self.logTextView.text = [self.logTextView.text stringByAppendingString:log];
        NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
        [self.logTextView scrollRangeToVisible:bottom];
    });
    
    if (event == TTS_EVENT_SYNTHESIS_STARTED) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SYNTHESIS_STARTED");
        if (_audioController != nil) {
            // 合成启动，启动播放器
            [_audioController startPlayer];
        }
    } else if (event == TTS_EVENT_SENTENCE_BEGIN) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_BEGIN");
    } else if (event == TTS_EVENT_SENTENCE_SYNTHESIS) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_SYNTHESIS");
    } else if (event == TTS_EVENT_SENTENCE_END) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_END");
    } else if (event == TTS_EVENT_SYNTHESIS_COMPLETE) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SYNTHESIS_COMPLETE");
        if (_audioController != nil) {
            // 注意这里的event事件是指语音合成完成，而非播放完成，播放完成需要由voicePlayer对象来进行通知
            [_audioController drain];
            [_audioController stopPlayer];
        }
    } else if (event == TTS_EVENT_TASK_FAILED) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_TASK_FAILED:%s", error_msg);
        if (_audioController != nil) {
            // 注意这里的event事件是指语音合成完成，而非播放完成，播放完成需要由voicePlayer对象来进行通知
            [_audioController drain];
        }
    }
}

- (void)onStreamInputTtsDataCallback:(char*)buffer len:(int)len {
    NSString *log = [NSString stringWithFormat:@"\n音频回调 %d bytes", len];
    TLog(log);
    dispatch_async(dispatch_get_main_queue(), ^{
        self.logTextView.text = [self.logTextView.text stringByAppendingString:log];
        NSRange bottom = NSMakeRange(self.logTextView.text.length -1, 1);
        [self.logTextView scrollRangeToVisible:bottom];
    });
    if (buffer != NULL && len > 0 && _audioController != nil) {
        [_audioController write:(char*)buffer Length:(unsigned int)len];
    }
}

-(void)onStreamInputTtsLogTrackCallback:(NuiSdkLogLevel)level
                             logMessage:(const char *)log {
    TLog(@"onStreamInputTtsLogTrackCallback log level:%d, message -> %s", level, log);
}

- (BOOL)textView:(UITextView *)textView shouldChangeTextInRange:(NSRange)range replacementText:(NSString *)text {
    if ([text isEqualToString:@"\n"]) { // 检测到“完成”按钮按下
        [textView resignFirstResponder]; // 让UITextView放弃first responder状态来隐藏键盘
        return NO; // 返回NO以防止输入换行符
    }
    return YES;
}


@end
