//
//  SpeechRecognizerViewController.mm
//  NUIdemo
//
//  Created by zhouguangdong on 2020/6/11.
//  Copyright © 2020 Alibaba idst. All rights reserved.
//

#import <Foundation/Foundation.h>
//控制demo中的log打印，不定义则Tlog不打印
#define DEBUG_MODE

#import "nuisdk.framework/Headers/NeoNui.h"
#import "SpeechRecognizerViewController.h"
#import "AppDelegate.h"
#import "NuiSdkUtils.h"

#import "audio/AudioController.h"
#import <AudioToolbox/AudioToolbox.h>
#include <sys/time.h>
#include <time.h>

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667


static BOOL save_wav = NO;
static BOOL save_log = NO;
static BOOL is_stopping = NO;
static NuiVadMode vad_mode = MODE_VAD;
static NSString *debug_path = @"";

static SpeechRecognizerViewController *myself = nil;
static dispatch_queue_t sr_work_queue;

@interface SpeechRecognizerViewController ()<ConvVoiceRecorderDelegate, NeoNuiSdkDelegate> {
    IBOutlet UIButton *StartButton;
    IBOutlet UIButton *StopButton;

    IBOutlet UITextView *textViewEvent;
    IBOutlet UITextView *textViewDialog;

    IBOutlet UILabel *VadModeLabel;
    IBOutlet UISwitch *switchVadMode;

    IBOutlet UILabel *SaveLogLabel;
    IBOutlet UISwitch *switchSaveLog;

    IBOutlet UILabel *SaveWavLabel;
    IBOutlet UISwitch *switchSaveWav;

    IBOutlet UITextField *textVersion;
}
@property(nonatomic,strong) NeoNui* nui;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NSMutableData *recordedVoiceData;
@property(nonatomic,strong) NuiSdkUtils *utils;

- (IBAction)showStart;
- (IBAction)showStop;
@end

@implementation SpeechRecognizerViewController


#pragma mark - View Callback Action
- (void)viewDidLoad {
    [super viewDidLoad];
    TLog(@"DialogViewController did load");
    myself = self;
    
    [self InitView];

    _utils = [NuiSdkUtils alloc];
    debug_path = [_utils createDir];

    sr_work_queue = dispatch_queue_create("NuiSRController", DISPATCH_QUEUE_CONCURRENT);

    [self initNui];
    NSString *version = [NSString stringWithUTF8String:[_nui nui_get_version]];
    NSString *show_version = [NSString stringWithFormat:@"SDK版本号: %@", version];
    myself->textVersion.text = show_version;
}

-(void)viewWillDisappear:(BOOL)animated {
    TLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    [_nui nui_release];
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - UI action
- (IBAction)StartButHandler:(UIButton *)sender{
    TLog(@"click START BUTTON, start recorder!");
    is_stopping = NO;
    textViewEvent.text = @"";
    textViewDialog.text = @"";
    [myself showStart];
}

- (IBAction)StopButHandler:(id)sender {
    TLog(@"click STOP BUTTON, stop recorder!");
    [myself appendEventResult:@"STOP BUTTON"];
    [myself showStop];
}

-(IBAction)showStart {
    if (_audioController == nil) {
        // 注意：这里audioController模块仅用于录音示例，用户可根据业务场景自行实现这部分代码
        _audioController = [[AudioController alloc] init:only_recorder];
        _audioController.delegate = self;
    }

    if (_audioController != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [myself->StartButton setEnabled:NO];
            [myself->StartButton setAlpha:0.4];
            [myself->StopButton setEnabled:YES];
            [myself->StopButton setAlpha:1.0];
        });

        dispatch_async(sr_work_queue, ^{
            if (_nui != nil) {
                //若需要修改token等参数, 可详见genDialogParams()
                NSString * parameters = [self genDialogParams];
                //若要使用VAD模式，则需要设置nls_config参数启动在线VAD模式(见genParams())
                // 因为VAD断句用到的是云端VAD，所以nui_dialog_start入参为MODE_P2T
                [_nui nui_dialog_start:MODE_P2T dialogParam:[parameters UTF8String]];
            } else {
                TLog(@"in StartButHandler no nui alloc");
            }
        });
    }
}

-(IBAction)showStop {
    self.recordedVoiceData = nil;
    is_stopping = YES;
    
    if (_nui != nil) {
        [_nui nui_dialog_cancel:NO];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [myself->StartButton setEnabled:YES];
            [myself->StartButton setAlpha:1.0];
            [myself->StopButton setEnabled:NO];
            [myself->StopButton setAlpha:0.4];
        });
    } else {
        TLog(@"in StopButHandler no nui alloc");
    }
}

-(IBAction)switchSaveWavAction:(id)sender
{
    UISwitch *switchButton = (UISwitch*)sender;
    BOOL isButtonOn = [switchButton isOn];

    TLog(@"Set save wav from %@ to %@",
         save_wav ? @"YES" : @"NO", isButtonOn ? @"YES" : @"NO");
    save_wav = isButtonOn;
    if (save_wav == YES) {
        NSString *showText = [NSString stringWithFormat:@"打开调试音频本地存储, 存储在: %@", debug_path];
        [self showToastWithMessage:showText showTime:1];
    } else {
        [self showToastWithMessage:@"关闭调试音频本地存储" showTime:1];
    }

    [myself terminateNui];
    usleep(200*1000);
    [myself initNui];
    [myself showStop];
}

-(IBAction)switchSaveLogAction:(id)sender
{
    UISwitch *switchButton = (UISwitch*)sender;
    BOOL isButtonOn = [switchButton isOn];

    TLog(@"Set save log from %@ to %@",
         save_wav ? @"YES" : @"NO", isButtonOn ? @"YES" : @"NO");
    save_log = isButtonOn;
    if (save_log == YES) {
        NSString *showText = [NSString stringWithFormat:@"打开调试日志本地存储, 存储在: %@", debug_path];
        [self showToastWithMessage:showText showTime:1];
    } else {
        [self showToastWithMessage:@"关闭调试日志本地存储" showTime:1];
    }

    [myself terminateNui];
    usleep(200*1000);
    [myself initNui];
    [myself showStop];
}

-(IBAction)switchVadModeAction:(id)sender
{
    UISwitch *switchButton = (UISwitch*)sender;
    BOOL isButtonOn = [switchButton isOn];
    
    TLog(@"Set VAD mode from %@ to %@",
         vad_mode == MODE_VAD ? @"VAD" : @"P2T",
         isButtonOn ? @"VAD" : @"P2T");
    if (isButtonOn == YES) {
        vad_mode = MODE_VAD;
        VadModeLabel.text = @"打开VAD，自动结束识别";
    } else {
        vad_mode = MODE_P2T;
        VadModeLabel.text = @"关闭VAD，点击<结束停止识别>";
    }
    NSString *showText = [NSString stringWithFormat:@"切换到 %@ 模式", vad_mode == MODE_VAD ? @"VAD" : @"P2T"];
    [self showToastWithMessage:showText showTime:1];
    [myself terminateNui];
    usleep(200*1000);
    [myself initNui];
    [myself showStop];
}

- (void)terminateNui {
    TLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    [_nui nui_release];
}

-(void)dealloc{
    TLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    [_nui nui_release];
}

#pragma mark - Voice Recorder Delegate
-(void) recorderDidStart{
    TLog(@"recorderDidStart");
    [myself appendEventResult:@"RECORDER STARTED"];
}

-(void) recorderDidStop{
    [self.recordedVoiceData setLength:0];
    TLog(@"recorderDidStop");
    [myself appendEventResult:@"RECORDER STOPPED"];
}

-(void) voiceRecorded:(unsigned char*)buffer Length:(int)len{
    NSData *frame = [NSData dataWithBytes:buffer length:len];
    @synchronized(_recordedVoiceData){
        [_recordedVoiceData appendData:frame];
    }
}

-(void) voiceDidFail:(NSError*)error{
    TLog(@"recorder error ");
    [myself appendEventResult:@"RECORDER ERROR!"];
}

#pragma mark - Nui Listener
-(void)onNuiEventCallback:(NuiCallbackEvent)nuiEvent
                   dialog:(long)dialog
                kwsResult:(const char *)wuw
                asrResult:(const char *)asr_result
                 ifFinish:(bool)finish
                  retCode:(int)code {
    TLog(@"onNuiEventCallback event %d finish %d", nuiEvent, finish);
    if (nuiEvent == EVENT_ASR_STARTED) {
        // asr_result在此包含task_id，task_id有助于排查问题，请用户进行记录保存。
        NSString *startedInfo = [NSString stringWithFormat:@"EVENT_ASR_STARTED: %@",
                                 [NSString stringWithUTF8String:asr_result]];
        TLog(@"%@", startedInfo);
    } else if (nuiEvent == EVENT_ASR_PARTIAL_RESULT) {
        // asr_result在此包含task_id，task_id有助于排查问题，请用户进行记录保存。
        NSString *asrPartialResult = [NSString stringWithFormat:@"EVENT_ASR_PARTIAL_RESULT: %@",
                                      [NSString stringWithUTF8String:asr_result]];
        TLog(@"%@", asrPartialResult);
        if (is_stopping == YES) {
            [myself appendEventResult:asrPartialResult];
        }
        [myself showDialogResult:[myself getAsrFromResult:asr_result]];
    } else if (nuiEvent == EVENT_ASR_RESULT) {
        // asr_result在此包含task_id，task_id有助于排查问题，请用户进行记录保存。
        NSString *asrFinalResult = [NSString stringWithFormat:@"EVENT_ASR_RESULT: %@, finish: %@",
                                    [NSString stringWithUTF8String:asr_result],
                                    finish ? @"YES" : @"NO"];
        TLog(@"%@", asrFinalResult);
        [myself appendEventResult:asrFinalResult];
        [myself showDialogResult:[myself getAsrFromResult:asr_result]];
        is_stopping = NO;
    } else if (nuiEvent == EVENT_VAD_START) {
        TLog(@"EVENT_VAD_START");
        [myself appendEventResult:@"EVENT_VAD_START"];
    } else if (nuiEvent == EVENT_VAD_END) {
        TLog(@"EVENT_VAD_END");
        [myself appendEventResult:@"EVENT_VAD_END"];
    } else if (nuiEvent == EVENT_ASR_ERROR) {
        // asr_result在EVENT_ASR_ERROR中为错误信息，搭配错误码code和其中的task_id更易排查问题，请用户进行记录保存。
        const char* all_response = [_nui nui_get_all_response];
        NSString *errorMessage = [NSString stringWithFormat:@"EVENT_ASR_ERROR error[%d], all mesg[%@]",
                                  (int)code, [NSString stringWithUTF8String:all_response]];
        TLog(@"%@", errorMessage);
        [myself appendEventResult:errorMessage];

        NSString *result = [NSString stringWithUTF8String:asr_result];
        NSString *errInfo = [_utils getGuideWithError:code withError:result withStatus:@"run"];
        [myself showToastWithMessage:errInfo showTime:5];

        NSString *errorDetail = [NSString stringWithFormat:@"error code:%d\nerror mesg: %@\n\n%@",
                                 (int)code, [myself getErrMesgFromResponse:all_response], errInfo];
        [myself showDialogResult:errorDetail];

        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
            [_audioController startRecorder];
        }
        is_stopping = NO;
    
    } else if (nuiEvent == EVENT_MIC_ERROR) {
        TLog(@"MIC ERROR");
        [myself appendEventResult:@"EVENT_MIC_ERROR"];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
            [_audioController startRecorder];
        }
        is_stopping = NO;
    }
    
    //finish 为真（可能是发生错误，也可能是完成识别）表示一次任务生命周期结束，可以开始新的识别
    if (finish) {
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [myself->StartButton setEnabled:YES];
            [myself->StartButton setAlpha:1.0];
            [myself->StopButton setEnabled:NO];
            [myself->StopButton setAlpha:0.4];
        });
    }
    
    return;
}

-(int)onNuiNeedAudioData:(char *)audioData length:(int)len {
//    TLog(@"onNuiNeedAudioData");
    static int emptyCount = 0;
    @autoreleasepool {
        @synchronized(_recordedVoiceData){
            if (_recordedVoiceData.length > 0) {
                int recorder_len = 0;
                if (_recordedVoiceData.length > len)
                    recorder_len = len;
                else
                    recorder_len = _recordedVoiceData.length;
                NSData *tempData = [_recordedVoiceData subdataWithRange:NSMakeRange(0, recorder_len)];
                [tempData getBytes:audioData length:recorder_len];
                tempData = nil;
                NSInteger remainLength = _recordedVoiceData.length - recorder_len;
                NSRange range = NSMakeRange(recorder_len, remainLength);
                [_recordedVoiceData setData:[_recordedVoiceData subdataWithRange:range]];
                emptyCount = 0;
                return recorder_len;
            } else {
                if (emptyCount++ >= 50) {
                    TLog(@"_recordedVoiceData length = %lu! empty 50times.", (unsigned long)_recordedVoiceData.length);
                    emptyCount = 0;
                }
                return 0;
            }

        }
    }
    return 0;
}
-(void)onNuiAudioStateChanged:(NuiAudioState)state{
    TLog(@"onNuiAudioStateChanged state=%u", state);
    if (state == STATE_CLOSE) {
        [myself appendEventResult:@"RECORDER STATE_CLOSE"];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
    } else if (state == STATE_PAUSE) {
        [myself appendEventResult:@"RECORDER STATE_PAUSE"];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
    } else if (state == STATE_OPEN){
        [myself appendEventResult:@"RECORDER STATE_OPEN"];
        self.recordedVoiceData = [NSMutableData data];
        if (_audioController != nil) {
            [_audioController startRecorder];
        }
    }
}

-(void)onNuiRmsChanged:(float)rms {
//    TLog(@"onNuiRmsChanged rms=%f", rms);
}

-(void)onNuiLogTrackCallback:(NuiSdkLogLevel)level
                  logMessage:(const char *)log {
    TLog(@"onNuiLogTrackCallback log level:%d, message -> %s", level, log);
}

#pragma mark - Private methods

-(void) showEventResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        myself->textViewEvent.text = result;
    });
}

-(void) showDialogResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        myself->textViewDialog.text = result;
    });
}

-(void) appendEventResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        NSString *currentText = myself->textViewEvent.text;
        NSString *updatedText = [currentText stringByAppendingFormat:@"\n---\n%@", result];
        myself->textViewEvent.text = updatedText;
    });
}

- (void)showToastWithMessage:(NSString *)message showTime:(int)seconds {
    if ([message length] != 0) {
        dispatch_async(dispatch_get_main_queue(), ^{
            UIAlertController *alertController = [UIAlertController alertControllerWithTitle:nil
                                                                                     message:message
                                                                              preferredStyle:UIAlertControllerStyleAlert];
            
            [self presentViewController:alertController animated:YES completion:^{
                dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(seconds * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
                    [alertController dismissViewControllerAnimated:YES completion:nil];
                });
            }];
        });
    }
}

-(NSString*) getErrMesgFromResponse:(const char*)response {
    if (response == NULL || strlen(response) == 0) {
        return @"";
    } else {
        // 将 const char* 转换为 NSString
        NSString *jsonStr = [NSString stringWithUTF8String:response];
        // 将 JSON 字符串转换为 NSData
        NSData *data = [jsonStr dataUsingEncoding:NSUTF8StringEncoding];
        // 使用 NSJSONSerialization 解析 JSON
        NSError *error;
        NSDictionary *jsonDict = [NSJSONSerialization JSONObjectWithData:data options:0 error:&error];
        
        if (!error && jsonDict) {
            // 从字典中获取 status_text
            NSDictionary *header = jsonDict[@"header"];
            NSString *statusText = header[@"status_text"];
            return statusText;
        } else {
            return @"";
        }
    }
}

-(NSString*) getAsrFromResult:(const char*)response {
    if (response == NULL || strlen(response) == 0) {
        return @"";
    } else {
        // 将 const char* 转换为 NSString
        NSString *jsonStr = [NSString stringWithUTF8String:response];
        // 将 JSON 字符串转换为 NSData
        NSData *data = [jsonStr dataUsingEncoding:NSUTF8StringEncoding];
        // 使用 NSJSONSerialization 解析 JSON
        NSError *error;
        NSDictionary *jsonDict = [NSJSONSerialization JSONObjectWithData:data options:0 error:&error];
        
        if (!error && jsonDict) {
            // 从字典中获取 status_text
            NSDictionary *payload = jsonDict[@"payload"];
            NSString *result = payload[@"result"];
            return result;
        } else {
            return @"";
        }
    }
}

-(NSString*) genInitParams {
//    NSString *strResourcesBundle = [[NSBundle mainBundle] pathForResource:@"Resources" ofType:@"bundle"];
//    NSString *bundlePath = [[NSBundle bundleWithPath:strResourcesBundle] resourcePath]; // 注意: V2.6.2版本开始纯云端功能可不需要资源文件

    NSMutableDictionary *ticketJsonDict = [NSMutableDictionary dictionary];
    //获取账号访问凭证：
    [_utils getTicket:ticketJsonDict Type:get_token_from_server_for_online_features];
    if ([ticketJsonDict objectForKey:@"token"] != nil) {
        NSString *tokenValue = [ticketJsonDict objectForKey:@"token"];
        if ([tokenValue length] == 0) {
            TLog(@"The 'token' key exists but the value is empty.");
        }
    } else {
        TLog(@"The 'token' key does not exist.");
    }

    [ticketJsonDict setObject:@"wss://nls-gateway.cn-shanghai.aliyuncs.com:443/ws/v1" forKey:@"url"]; // 默认

    //工作目录路径，SDK从该路径读取配置文件
//    [ticketJsonDict setObject:bundlePath forKey:@"workspace"]; // V2.6.2版本开始纯云端功能可不设置workspace

    //当初始化SDK时的save_log参数取值为true时，该参数生效。表示是否保存音频debug，该数据保存在debug目录中，需要确保debug_path有效可写
    [ticketJsonDict setObject:save_wav ? @"true" : @"false" forKey:@"save_wav"];
    //debug目录，当初始化SDK时的save_log参数取值为true时，该目录用于保存中间音频文件
    [ticketJsonDict setObject:debug_path forKey:@"debug_path"];

    //过滤SDK内部日志通过回调送回到用户层
    [ticketJsonDict setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_INFO] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [ticketJsonDict setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    //FullMix = 0   // 选用此模式开启本地功能并需要进行鉴权注册
    //FullCloud = 1 // 在线实时语音识别可以选这个
    //FullLocal = 2 // 选用此模式开启本地功能并需要进行鉴权注册
    //AsrMix = 3    // 选用此模式开启本地功能并需要进行鉴权注册
    //AsrCloud = 4  // 在线一句话识别可以选这个
    //AsrLocal = 5  // 选用此模式开启本地功能并需要进行鉴权注册
    [ticketJsonDict setObject:@"4" forKey:@"service_mode"]; // 必填

    [ticketJsonDict setObject:@"empty_device_id" forKey:@"device_id"]; // V2.6.2版本开始纯云端功能可不设置, 推荐填入具有唯一性的id, 方便定位问题

    NSData *data = [NSJSONSerialization dataWithJSONObject:ticketJsonDict options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genParams {
    NSMutableDictionary *nls_config = [NSMutableDictionary dictionary];
    [nls_config setValue:@YES forKey:@"enable_intermediate_result"]; // 是否返回中间识别结果，默认值：False
    [nls_config setValue:@YES forKey:@"enable_punctuation_prediction"]; // 是否在后处理中添加标点，默认值：False

    //参数可根据实际业务进行配置
    //接口说明可见: https://help.aliyun.com/document_detail/173298.html
    //查看 2.开始识别

    //由于对外的SDK(01B版本)不带有本地VAD模块(仅带有唤醒功能(029版本)的SDK具有VAD模块)，
    //若要使用VAD模式，则需要设置nls_config参数启动在线VAD模式(见genParams())
    if (vad_mode == MODE_VAD) {
        [nls_config setValue:@YES forKey:@"enable_voice_detection"];
        [nls_config setValue:@10000 forKey:@"max_start_silence"];
        [nls_config setValue:@800 forKey:@"max_end_silence"];
    }

//    [nls_config setValue:@"<更新token>" forKey:@"token"];
//    [nls_config setValue:@YES forKey:@"enable_punctuation_prediction"];
//    [nls_config setValue:@YES forKey:@"enable_inverse_text_normalization"];
//    [nls_config setValue:@16000 forKey:@"sample_rate"];
//    [nls_config setValue:@"opus" forKey:@"sr_format"];

    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];
    [dictM setObject:nls_config forKey:@"nls_config"];
    [dictM setValue:@(SERVICE_TYPE_ASR) forKey:@"service_type"]; // 必填

//    如果有HttpDns则可进行设置
//    [dictM setObject:[_utils getDirectIp] forKey:@"direct_ip"];
    
    /*若文档中不包含某些参数，但是此功能支持这个参数，可以用如下万能接口设置参数*/
//    NSMutableDictionary *extend_config = [NSMutableDictionary dictionary];
//    [extend_config setValue:@YES forKey:@"custom_test"];
//    [dictM setObject:extend_config forKey:@"extend_config"];
    
    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genDialogParams {
    NSMutableDictionary *dialog_params = [NSMutableDictionary dictionary];

    // 运行过程中可以在nui_dialog_start时更新临时参数，尤其是更新过期token
    // 注意: 若下一轮对话不再设置参数，则继续使用初始化时传入的参数
    long distance_expire_time_5m = 300;
    [_utils refreshTokenIfNeed:dialog_params distanceExpireTime:distance_expire_time_5m];

    // 注意: 若需要更换appkey和token，可以直接传入参数
//    [dialog_params setValue:@"" forKey:@"app_key"];
//    [dialog_params setValue:@"" forKey:@"token"];
    
    NSData *data = [NSJSONSerialization dataWithJSONObject:dialog_params options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

- (void) initNui {
    if (_nui == NULL) {
        _nui = [NeoNui get_instance];
        _nui.delegate = self;
    }
    //请注意此处的参数配置，其中账号相关需要按照genInitParams的说明填入后才可访问服务
    NSString * initParam = [self genInitParams];
    
    NuiResultCode retCode = [_nui nui_initialize:[initParam UTF8String] logLevel:NUI_LOG_LEVEL_VERBOSE saveLog:save_log];
    TLog(@"nui initialize with code:%d", retCode);
    if (retCode != 0) {
        NSString *errInfo = [_utils getGuideWithError:retCode withError:@"" withStatus:@"init"];
        NSString *errorShow = [NSString stringWithFormat:@"error code:%d\nerror mesg: %@",
                                 (int)retCode, errInfo];
        [myself showToastWithMessage:errInfo showTime:5];
        return;
    }

    NSString * parameters = [self genParams];
    [_nui nui_set_params:[parameters UTF8String]];
}


#pragma mark - View
-(void)InitView {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"一句话识别";
    /*---------------------------------Button---------------------------------------------*/
    CGFloat interval_width = global_size.width/SCREEN_WIDTH_BASE * 10;
    CGFloat interval_height = global_size.height/SCREEN_HEIGHT_BASE * 10;

    //Start Button
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 70;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 30;

    
    CGFloat x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    
    StartButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StartButton.frame = CGRectMake(x, y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [StartButton setBackgroundImage:image forState:UIControlStateNormal];
    [StartButton setTitle:@"开始" forState:UIControlStateNormal];
    [StartButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StartButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StartButton addTarget:self action:@selector(StartButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StartButton];
    
    //Stop Button
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5+90*2);
    y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    
    StopButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StopButton.frame = CGRectMake(x, y, button_width, button_height);
    image = [UIImage imageNamed:@"button_start"];
    [StopButton setBackgroundImage:image forState:UIControlStateNormal];
    [StopButton setTitle:@"结束" forState:UIControlStateNormal];
    [StopButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StopButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StopButton addTarget:self action:@selector(StopButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StopButton];
    
    
    /*---------------------------------UITextView---------------------------------------------*/
    //event text view
    CGFloat event_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat event_view_height = global_size.height/SCREEN_HEIGHT_BASE * 200;
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    y = global_size.height/SCREEN_HEIGHT_BASE * 110;
    
    CGRect event_rect = CGRectMake(x, y, event_view_width, event_view_height);
    if (!textViewEvent) {
        textViewEvent = [[UITextView alloc] initWithFrame:event_rect];
    }
    textViewEvent.layer.borderWidth = 0.6;
    textViewEvent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewEvent.layer.cornerRadius = 10;
    [textViewEvent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewEvent.scrollEnabled = YES;
    
    textViewEvent.text = @"事件状态栏";
    textViewEvent.textColor = [UIColor darkGrayColor];
    textViewEvent.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewEvent];

    //dialog text view
    CGFloat dialog_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat dialog_view_height = global_size.height/SCREEN_HEIGHT_BASE * 80;
    CGFloat dialog_x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    CGFloat dialog_y = global_size.height/SCREEN_HEIGHT_BASE * 110 + event_view_height + interval_height;
    
    CGRect dialog_rect = CGRectMake(dialog_x, dialog_y, dialog_view_width, dialog_view_height);
    if (!textViewDialog) {
        textViewDialog = [[UITextView alloc] initWithFrame:dialog_rect];
    }
    textViewDialog.layer.borderWidth = 0.6;
    textViewDialog.layer.borderColor = [UIColor blackColor].CGColor;
    textViewDialog.layer.cornerRadius = 10;
    [textViewDialog setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewDialog.scrollEnabled = YES;
    
    textViewDialog.text = @"识别结果栏";
    textViewDialog.textColor = [UIColor darkGrayColor];
    textViewDialog.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewDialog];
    
    /*---------------------------------Switch---------------------------------------------*/
    CGFloat label_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat label_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat switch_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat switch_height = global_size.height/SCREEN_HEIGHT_BASE * 30;

    //VAD mode label
    CGFloat vad_mode_label_width = global_size.width/SCREEN_WIDTH_BASE * 240;
    CGFloat vad_mode_label_x = dialog_x;
    CGFloat vad_mode_label_y = dialog_y + dialog_view_height + label_height;

    CGRect vad_mode_label_rect = CGRectMake(vad_mode_label_x, vad_mode_label_y, vad_mode_label_width, label_height);
    if (!VadModeLabel) {
        VadModeLabel = [[UILabel alloc] initWithFrame:vad_mode_label_rect];
    }
    VadModeLabel.text = @"打开VAD，自动结束识别";
    VadModeLabel.textColor = UIColor.blackColor;
    VadModeLabel.backgroundColor = UIColor.darkGrayColor;
    VadModeLabel.layer.cornerRadius = 5.0;
    VadModeLabel.layer.masksToBounds = true;
    [self.view addSubview:VadModeLabel];

    //VAD mode switch
    CGFloat vad_mode_switch_x = vad_mode_label_x + vad_mode_label_width + interval_width;
    CGFloat vad_mode_switch_y = vad_mode_label_y;

    CGRect vad_mode_switch_rect = CGRectMake(vad_mode_switch_x, vad_mode_switch_y, switch_width, switch_height);
    switchVadMode = [[UISwitch alloc] initWithFrame:vad_mode_switch_rect];
    [switchVadMode setOn:YES animated:YES];
    vad_mode = MODE_VAD;
    [switchVadMode addTarget:self action:@selector(switchVadModeAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:switchVadMode];

    //save wav label
    CGFloat save_wav_label_x = dialog_x;
    CGFloat save_wav_label_y = vad_mode_label_y + label_height + interval_height * 3;

    CGRect save_wav_label_rect = CGRectMake(save_wav_label_x, save_wav_label_y, label_width, label_height);
    if (!SaveWavLabel) {
        SaveWavLabel = [[UILabel alloc] initWithFrame:save_wav_label_rect];
    }
    SaveWavLabel.text = @"音频保存";
    SaveWavLabel.textColor = UIColor.blackColor;
    SaveWavLabel.backgroundColor = UIColor.lightGrayColor;
    SaveWavLabel.layer.cornerRadius = 5.0;
    SaveWavLabel.layer.masksToBounds = true;
    [self.view addSubview:SaveWavLabel];

    //save wav switch
    CGFloat save_wav_switch_x = save_wav_label_x + label_width + interval_width;
    CGFloat save_wav_switch_y = save_wav_label_y;

    CGRect save_wav_switch_rect = CGRectMake(save_wav_switch_x, save_wav_switch_y, switch_width, switch_height);
    switchSaveWav = [[UISwitch alloc] initWithFrame:save_wav_switch_rect];
    [switchSaveWav addTarget:self action:@selector(switchSaveWavAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:switchSaveWav];
  
    //save log label
    CGFloat save_log_label_x = dialog_x;
    CGFloat save_log_label_y = save_wav_label_y + label_height + interval_height;

    CGRect save_log_label_rect = CGRectMake(save_log_label_x, save_log_label_y, label_width, label_height);
    if (!SaveLogLabel) {
        SaveLogLabel = [[UILabel alloc] initWithFrame:save_log_label_rect];
    }
    SaveLogLabel.text = @"日志保存";
    SaveLogLabel.textColor = UIColor.blackColor;
    SaveLogLabel.backgroundColor = UIColor.lightGrayColor;
    SaveLogLabel.layer.cornerRadius = 5.0;
    SaveLogLabel.layer.masksToBounds = true;
    [self.view addSubview:SaveLogLabel];

    //save log switch
    CGFloat save_log_switch_x = save_log_label_x + label_width + interval_width;
    CGFloat save_log_switch_y = save_log_label_y;

    CGRect save_log_switch_rect = CGRectMake(save_log_switch_x, save_log_switch_y, switch_width, switch_height);
    switchSaveLog = [[UISwitch alloc] initWithFrame:save_log_switch_rect];
    [switchSaveLog addTarget:self action:@selector(switchSaveLogAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:switchSaveLog];

    /*---------------------------------------TextField----------------------------------*/
    
    CGFloat text_field_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat text_field_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat text_field_x = save_wav_label_x;
    
    //text sdk version
    CGFloat text_version_x = text_field_x;
    CGFloat text_version_y = global_size.height - text_field_height * 2;
    
    CGRect text_version_rect = CGRectMake(text_version_x, text_version_y, text_field_width, text_field_height);
    textVersion = [[UITextField alloc] initWithFrame:text_version_rect];

    textVersion.layer.borderWidth = 0.6;
    textVersion.layer.borderColor = [UIColor blackColor].CGColor;
    textVersion.layer.cornerRadius = 10;
    [textVersion setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    
    textVersion.textColor = [UIColor darkGrayColor];
    textVersion.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textVersion];
    
    
    [myself->StartButton setEnabled:YES];
    [myself->StartButton setAlpha:1.0];
    [myself->StopButton setEnabled:NO];
    [myself->StopButton setAlpha:0.4];
    
    textViewEvent.scrollEnabled = YES;
    textViewEvent.editable = NO;

    textViewDialog.scrollEnabled = YES;
    textViewDialog.editable = NO;
    
    myself->textVersion.minimumFontSize = 10;
}


@end
