//
//  OnlyWakeupViewController.mm
//  NUIdemo
//
//  Created by shichen.fsc on 2024/6/7.
//  Copyright © 2024 Alibaba idst. All rights reserved.
//

#import <Foundation/Foundation.h>
#define DEBUG_MODE
#import "nuisdk.framework/Headers/NeoNui.h"
#import "OnlyWakeupViewController.h"
#import "AppDelegate.h"
#import "NuiSdkUtils.h"

#import "audio/AudioController.h"
#import <AudioToolbox/AudioToolbox.h>
#include <sys/time.h>
#include <time.h>

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static OnlyWakeupViewController *myself = nil;

static BOOL save_wav = NO;
static BOOL save_log = NO;
static int wakeup_count = 0;
static dispatch_queue_t kws_work_queue;

@interface OnlyWakeupViewController ()<ConvVoiceRecorderDelegate, NeoNuiSdkDelegate> {
    IBOutlet UIButton *StartButton;
    IBOutlet UIButton *StopButton;

    IBOutlet UITextView *textViewKws;
    IBOutlet UITextView *textViewAsr;
    IBOutlet UITextView *textViewDialog;

    IBOutlet UISwitch *switchLog;

    IBOutlet UITextField *textfield;
    IBOutlet UITextField *textVersion;
    
    IBOutlet UILabel *SaveWavLabel;
}
@property(nonatomic,strong) NeoNui* nui;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NSMutableData *recordedVoiceData;
@property(nonatomic,strong) NuiSdkUtils *utils;
@property(nonatomic) NSInteger callNum;

- (IBAction)showStart;
- (IBAction)showStop;
@end

@implementation OnlyWakeupViewController

#pragma mark - View Callback Action
- (void)viewDidLoad {
    [super viewDidLoad];
    TLog(@"OnlyWakeupViewController did load");
    myself = self;
    
    [self InitView];

    _utils = [NuiSdkUtils alloc];
    _callNum = 0;

    kws_work_queue = dispatch_queue_create("NuiKwsController", DISPATCH_QUEUE_CONCURRENT);

    [self initNui];
    NSString *version = [NSString stringWithUTF8String:[_nui nui_get_version]];
    myself->textVersion.text = version;
}

-(void)viewWillDisappear:(BOOL)animated {
    TLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    [_nui nui_release];
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - UI action
- (IBAction)StartButHandler:(UIButton *)sender{
    TLog(@"click START BUTTON, start recorder!");
    [myself showStart];
}

- (IBAction)StopButHandler:(id)sender {
    TLog(@"click STOP BUTTON, stop recorder!");
    [myself showStop];
}

-(IBAction)showStart {
    if (_audioController == nil) {
        // 注意：这里audioController模块仅用于录音示例，用户可根据业务场景自行实现这部分代码
        _audioController = [[AudioController alloc] init:only_recorder];
        _audioController.delegate = self;
    }

    if (_audioController != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [myself->StartButton setEnabled:NO];
            [myself->StartButton setAlpha:0.4];
            [myself->StopButton setEnabled:YES];
            [myself->StopButton setAlpha:1.0];
            [myself showKwsResult:@""];
        });

        dispatch_async(kws_work_queue, ^{
            if (_nui != nil) {
                //若需要修改token等参数, 可详见genDialogParams()
                NSString * parameters = [self genDialogParams];
                [_nui nui_dialog_start:MODE_ONLY_KWS dialogParam:[parameters UTF8String]];
            } else {
                TLog(@"in StartButHandler no nui alloc");
            }
        });
    }
}

-(IBAction)showStop {
    self.recordedVoiceData = nil;
    
    if (_nui != nil) {
        [_nui nui_dialog_cancel:NO];
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
        wakeup_count = 0;
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [myself->StartButton setEnabled:YES];
            [myself->StartButton setAlpha:1.0];
            [myself->StopButton setEnabled:NO];
            [myself->StopButton setAlpha:0.4];
        });
    } else {
        TLog(@"in StopButHandler no nui alloc");
    }
}

-(IBAction)switchLogAction:(id)sender
{
    UISwitch *switchButton = (UISwitch*)sender;
    BOOL isButtonOn = [switchButton isOn];

    TLog(@"set save wav to %d from %d",
              isButtonOn, save_wav);
    save_wav = isButtonOn;
    save_log = isButtonOn;
    [myself terminateNui];
    usleep(200*1000);
    [myself initNui];
    [myself showStop];
    [myself showStart];
}


- (void)terminateNui {
    TLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    [_nui nui_release];
}

-(void)dealloc{
    TLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_initialize无法覆盖已经设置的参数.
    [_nui nui_release];
}

#pragma mark - Voice Recorder Delegate
-(void) recorderDidStart{
    TLog(@"recorderDidStart");
}

-(void) recorderDidStop{
    [self.recordedVoiceData setLength:0];
    TLog(@"recorderDidStop");
}

-(void) voiceRecorded:(unsigned char*)buffer Length:(int)len {
    NSData *frame = [NSData dataWithBytes:buffer length:len];
    @synchronized(_recordedVoiceData){
        [_recordedVoiceData appendData:frame];
    }
}

-(void) voiceDidFail:(NSError*)error{
    TLog(@"recorder error ");
}

#pragma mark - Nui Listener
-(void)onNuiEventCallback:(NuiCallbackEvent)nuiEvent
                   dialog:(long)dialog
                kwsResult:(const char *)wuw
                asrResult:(const char *)asr_result
                 ifFinish:(BOOL)finish
                  retCode:(int)code {
    TLog(@"onNuiEventCallback event %d finish %d", nuiEvent, finish);
    if (nuiEvent == EVENT_WUW_TRUSTED) {
        wakeup_count++;
        NSString *result = [NSString stringWithUTF8String:wuw];
        TLog(@"EVENT_WUW %s", wuw);
        [myself showKwsResult:result];
        NSString *combinedString = [NSString stringWithFormat:@"%@: %d", @"wakeup", wakeup_count];
        [myself showAsrResult:combinedString];
    } else if (nuiEvent == EVENT_WUW_END) {
    } else if (nuiEvent == EVENT_ASR_ERROR) {
        TLog(@"EVENT_ASR_ERROR error[%d]", code);
    } else if (nuiEvent == EVENT_MIC_ERROR) {
        TLog(@"MIC ERROR");
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
            [_audioController startRecorder];
        }
    }
    
    if (finish) {
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            [myself->StartButton setEnabled:YES];
            [myself->StartButton setAlpha:1.0];
            [myself->StopButton setEnabled:NO];
            [myself->StopButton setAlpha:0.4];
        });
    }
    
    return;
}

-(int)onNuiNeedAudioData:(char *)audioData length:(int)len {
    static int emptyCount = 0;
    @autoreleasepool {
        @synchronized(_recordedVoiceData){
            if (_recordedVoiceData.length > 0) {
                int recorder_len = 0;
                if (_recordedVoiceData.length > len)
                    recorder_len = len;
                else
                    recorder_len = _recordedVoiceData.length;
                NSData *tempData = [_recordedVoiceData subdataWithRange:NSMakeRange(0, recorder_len)];
                [tempData getBytes:audioData length:recorder_len];
                tempData = nil;
                NSInteger remainLength = _recordedVoiceData.length - recorder_len;
                NSRange range = NSMakeRange(recorder_len, remainLength);
                [_recordedVoiceData setData:[_recordedVoiceData subdataWithRange:range]];
                emptyCount = 0;
                return recorder_len;
            } else {
                if (emptyCount++ >= 50) {
                    TLog(@"_recordedVoiceData length = %lu! empty 50times.", (unsigned long)_recordedVoiceData.length);
                    emptyCount = 0;
                }
                return 0;
            }

        }
    }
    return 0;
}

-(void)onNuiAudioStateChanged:(NuiAudioState)state{
    TLog(@"onNuiAudioStateChanged state=%u", state);
    if (state == STATE_CLOSE || state == STATE_PAUSE) {
        if (_audioController != nil) {
            [_audioController stopRecorder:NO];
        }
    } else if (state == STATE_OPEN){
        self.recordedVoiceData = [NSMutableData data];
        if (_audioController != nil) {
            [_audioController startRecorder];
        }
    }
}
-(void)onNuiRmsChanged:(float)rms {
//    TLog(@"onNuiRmsChanged rms=%f", rms);
}

-(void)onNuiLogTrackCallback:(NuiSdkLogLevel)level
                  logMessage:(const char *)log {
    TLog(@"onNuiLogTrackCallback log level:%d, message -> %s", level, log);
}

#pragma mark - Private methods

-(void) showKwsResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        myself->textViewKws.text = result;
    });
}

-(void) showAsrResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        myself->textViewAsr.text = result;
    });
}

-(void) showDialogResult:(NSString *) result {
    dispatch_async(dispatch_get_main_queue(), ^{
        myself->textViewDialog.text = result;
    });
}

-(NSString*) genInitParams {
    NSString *strResourcesBundle = [[NSBundle mainBundle] pathForResource:@"Resources" ofType:@"bundle"];
    NSString *bundlePath = [[NSBundle bundleWithPath:strResourcesBundle] resourcePath];
    NSString *debug_path = [_utils createDir];
    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];

    //账号和项目创建
    // 需要特别注意：ak_id/ak_secret/app_key/sdk_code/device_id等参数必须传入SDK
    // 离线语音合成sdk_code取值：精品版为software_nls_tts_offline， 标准版为software_nls_tts_offline_standard
    // 离线语音合成账户的sdk_code也可用于唤醒
    // 鉴权信息获取参：https://help.aliyun.com/document_detail/69835.htm?spm=a2c4g.11186623.2.28.10b33069T2ydLk#topic-1917889
    [_utils getTicket:dictM Type:get_sts_access_from_server_for_offline_features];

    //请参照阿里云官网获取鉴权信息获取配额
    //  https://help.aliyun.com/document_detail/251488.html?spm=a2c4g.11186623.6.638.1f0d530eut95Jn
    //  如果配额已耗尽，请联系客户扩大配额
    //  如果合成失败，通常是由于鉴权失败，可以参照阿里云官网Q&A部分
    //  https://help.aliyun.com/document_detail/204191.html?spm=a2c4g.11186623.6.657.3cde7340qMll1h ，根据错误日志判别导致鉴权失败的原因


    [dictM setObject:@"wss://nls-gateway.cn-shanghai.aliyuncs.com:443/ws/v1" forKey:@"url"];

    [dictM setObject:bundlePath forKey:@"workspace"];
    [dictM setObject:debug_path forKey:@"debug_path"];
    [dictM setObject:save_wav ? @"true" : @"false" forKey:@"save_wav"];

    //过滤SDK内部日志通过回调送回到用户层
    [dictM setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_INFO] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [dictM setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    // 特别说明: 鉴权所用的id是由以下device_id，与手机内部的一些唯一码进行组合加密生成的。
    //   更换手机或者更换device_id都会导致重新鉴权计费。
    //   此外, 以下device_id请设置有意义且具有唯一性的id, 比如用户账号(手机号、IMEI等),
    //   传入相同或随机变换的device_id会导致鉴权失败或重复收费。
    //   NSString *id_string = [[[ASIdentifierManager sharedManager] advertisingIdentifier] UUIDString]; 并不能保证生成不变的device_id，请不要使用
    [dictM setObject:@"empty_device_id" forKey:@"device_id"]; // 必填

    //如果使用外部唤醒资源，在此设置模型文件路径参数
    // 举例1：模型文件pack_kws.bin可以放在thirdparty
//    NSString *file = [[NSBundle mainBundle] pathForResource:@"thirdparty/pack_kws" ofType:@"bin"];
//    [dictM setObject:file forKey:@"upgrade_file"];

    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genParams {
    // 纯唤醒模式无需传入参数
    NSMutableDictionary *nls_config = [NSMutableDictionary dictionary];
    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];
    [dictM setObject:nls_config forKey:@"nls_config"];
    [dictM setValue:@(SERVICE_TYPE_ASR) forKey:@"service_type"];
    
//    可以通过以下方式设置自定义唤醒词进行体验，如果需要更好的唤醒效果请进行唤醒词定制
//    注意：动态唤醒词只有在设置了唤醒模型的前提下才可以使用
    NSMutableDictionary *dictWuw = [NSMutableDictionary dictionary];
    [dictWuw setObject:@"小白小白" forKey:@"name"];
    [dictWuw setObject:@"main" forKey:@"type"];
    NSMutableArray *wuwArray = [[NSMutableArray alloc] init];
    [wuwArray addObject:dictWuw];
    [dictM setObject:wuwArray forKey:@"wuw"];
    
    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString*) genDialogParams {
    // 纯唤醒模式无需传入参数
    NSMutableDictionary *dialog_params = [NSMutableDictionary dictionary];
    NSData *data = [NSJSONSerialization dataWithJSONObject:dialog_params options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

- (void) initNui {
    if (_nui == NULL) {
        _nui = [NeoNui get_instance];
        _nui.delegate = self;
    }
    //请注意此处的参数配置，其中账号相关需要按照genInitParams的说明填入后才可访问服务
    NSString * initParam = [self genInitParams];

    [_nui nui_initialize:[initParam UTF8String] logLevel:NUI_LOG_LEVEL_VERBOSE saveLog:save_log];
    NSString * parameters = [self genParams];
    [_nui nui_set_params:[parameters UTF8String]];
}


#pragma mark - View
-(void)InitView {
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"唤醒识别";
/*---------------------------------Button---------------------------------------------*/
    //Start Button
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 70;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 30;

    
    CGFloat x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    
    StartButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StartButton.frame = CGRectMake(x, y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [StartButton setBackgroundImage:image forState:UIControlStateNormal];
    [StartButton setTitle:@"开始" forState:UIControlStateNormal];
    [StartButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StartButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StartButton addTarget:self action:@selector(StartButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StartButton];
    
    //stop Button
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5+90*2);
    y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    
    StopButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StopButton.frame = CGRectMake(x, y, button_width, button_height);
    image = [UIImage imageNamed:@"button_start"];
    [StopButton setBackgroundImage:image forState:UIControlStateNormal];
    [StopButton setTitle:@"结束" forState:UIControlStateNormal];
    [StopButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StopButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StopButton addTarget:self action:@selector(StopButHandler:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StopButton];
    
    
/*---------------------------------UITextView---------------------------------------------*/
    //kws text view
    CGFloat kws_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat kws_view_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    y = global_size.height/SCREEN_HEIGHT_BASE * 110;
    
    CGRect textViewContent_rect = CGRectMake(x, y, kws_view_width, kws_view_height);
    if (!textViewKws) {
        textViewKws = [[UITextView alloc] initWithFrame:textViewContent_rect];
    }
    textViewKws.layer.borderWidth = 0.6;
    textViewKws.layer.borderColor = [UIColor blackColor].CGColor;
    textViewKws.layer.cornerRadius = 10;
    [textViewKws setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewKws.scrollEnabled = YES;
    
    textViewKws.textColor = [UIColor darkGrayColor];
    textViewKws.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewKws];
    
    //wakeup text view
    CGFloat asr_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat asr_view_height = global_size.height/SCREEN_HEIGHT_BASE * 60;
    x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    y = global_size.height/SCREEN_HEIGHT_BASE * 110 + kws_view_height + 10;
    
    CGRect asr_rect = CGRectMake(x, y, asr_view_width, asr_view_height);
    if (!textViewAsr) {
        textViewAsr = [[UITextView alloc] initWithFrame:asr_rect];
    }
    textViewAsr.layer.borderWidth = 0.6;
    textViewAsr.layer.borderColor = [UIColor blackColor].CGColor;
    textViewAsr.layer.cornerRadius = 10;
    [textViewAsr setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewAsr.scrollEnabled = YES;
    
    textViewAsr.textColor = [UIColor darkGrayColor];
    textViewAsr.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewAsr];
    
    //dialog text view
    CGFloat dialog_view_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat dialog_view_height = global_size.height/SCREEN_HEIGHT_BASE * 120;
    CGFloat dialog_x = global_size.width/SCREEN_WIDTH_BASE * (27.5);
    CGFloat dialog_y = global_size.height/SCREEN_HEIGHT_BASE * 110 + kws_view_height + 10 + asr_view_height + 10;
    
    CGRect dialog_rect = CGRectMake(dialog_x, dialog_y, dialog_view_width, dialog_view_height);
    if (!textViewDialog) {
        textViewDialog = [[UITextView alloc] initWithFrame:dialog_rect];
    }
    textViewDialog.layer.borderWidth = 0.6;
    textViewDialog.layer.borderColor = [UIColor blackColor].CGColor;
    textViewDialog.layer.cornerRadius = 10;
    [textViewDialog setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewDialog.scrollEnabled = YES;
    
    textViewDialog.textColor = [UIColor darkGrayColor];
    textViewDialog.font = [UIFont systemFontOfSize:10];
    [self.view addSubview:textViewDialog];
    
    /*---------------------------------Switch---------------------------------------------*/
    
    CGFloat switch_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat switch_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat label_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat label_height = global_size.height/SCREEN_HEIGHT_BASE * 30;

    //switch save wav
    CGFloat save_wav_label_x = dialog_x;
    CGFloat save_wav_label_y = dialog_y + dialog_view_height + label_height;

    CGRect save_wav_label_rect = CGRectMake(save_wav_label_x, save_wav_label_y, label_width, label_height);
    if (!SaveWavLabel) {
        SaveWavLabel = [[UILabel alloc] initWithFrame:save_wav_label_rect];
    }
    SaveWavLabel.text = @"音频保存";
    [self.view addSubview:SaveWavLabel];

    CGFloat save_wav_switch_x = save_wav_label_x + label_width;
    CGFloat save_wav_switch_y = save_wav_label_y;
    CGRect save_wav_switch_rect = CGRectMake(save_wav_switch_x, save_wav_switch_y, switch_width, switch_height);
    switchLog = [[UISwitch alloc] initWithFrame:save_wav_switch_rect];
    [switchLog addTarget:self action:@selector(switchLogAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:switchLog];
    
    /*---------------------------------------TextField----------------------------------*/
    
    CGFloat text_field_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat text_field_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    //text field
    CGFloat text_field_x = save_wav_label_x;
    CGFloat text_field_y = save_wav_switch_y + label_height + label_height;
    
    CGRect text_field_rect = CGRectMake(text_field_x, text_field_y, text_field_width, text_field_height);
    textfield = [[UITextField alloc] initWithFrame:text_field_rect];
    textfield.layer.borderWidth = 0.6;
    textfield.layer.borderColor = [UIColor blackColor].CGColor;
    textfield.layer.cornerRadius = 10;
    [textfield setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    
    textfield.textColor = [UIColor darkGrayColor];
    textfield.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textfield];
    
    //text sdk version
    CGFloat text_version_x = text_field_x;
    CGFloat text_version_y = text_field_y + text_field_height + 2;
    
    CGRect text_version_rect = CGRectMake(text_version_x, text_version_y, text_field_width, text_field_height);
    textVersion = [[UITextField alloc] initWithFrame:text_version_rect];

    textVersion.layer.borderWidth = 0.6;
    textVersion.layer.borderColor = [UIColor blackColor].CGColor;
    textVersion.layer.cornerRadius = 10;
    [textVersion setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    
    textVersion.textColor = [UIColor darkGrayColor];
    textVersion.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textVersion];
    
    
    [myself->StartButton setEnabled:YES];
    [myself->StartButton setAlpha:1.0];
    [myself->StopButton setEnabled:NO];
    [myself->StopButton setAlpha:0.4];
    
    textViewKws.scrollEnabled = YES;
    textViewAsr.scrollEnabled = YES;
    textViewDialog.scrollEnabled = YES;
    
    textViewKws.editable = NO;
    textViewAsr.editable = NO;
    textViewDialog.editable = NO;
    
    myself->textVersion.minimumFontSize = 10;
    myself->textfield.minimumFontSize = 10;
}


@end
