//
//  DashCosyVoiceStreamInputTtsViewController.m
//  NUIdemo
//
//  Created by shichen.fsc on 2025/9/15.
//  Copyright © 2025 Alibaba idst. All rights reserved.
//

#import "nuisdk.framework/Headers/StreamInputTts.h"
#import "DashCosyVoiceStreamInputTtsViewController.h"
#import "HWOptionButton.h"
#import "NuiSdkUtils.h"

#import "audio/AudioController.h"
#import <AVFoundation/AVFoundation.h>

static DashCosyVoiceStreamInputTTSViewController *myself = nil;

// 本样例展示在线语音合成CosyVoice（流式输入流式输出）使用方法
// iOS SDK 详细说明：
// WebSocket API: https://help.aliyun.com/zh/model-studio/cosyvoice-websocket-api
@interface DashCosyVoiceStreamInputTTSViewController () <ConvVoiceRecorderDelegate, UITextFieldDelegate, HWOptionButtonDelegate, StreamInputTtsDelegate> {
    IBOutlet UIButton *StartButton;
    IBOutlet UIButton *PlayButton;
    IBOutlet UIButton *StopButton;
    IBOutlet UIButton *CancelButton;
    
    IBOutlet UITextView *textViewContent;
    IBOutlet UITextView *textViewEvent;
}

@property(nonatomic,strong) StreamInputTts* streamInputTtsSdk;
@property(nonatomic, weak) HWOptionButton *modelName;
@property(nonatomic, weak) HWOptionButton *fontName;
@property(nonatomic, weak) HWOptionButton *formatType;
@property(nonatomic, weak) HWOptionButton *pitch;
@property(nonatomic, weak) HWOptionButton *rate;
@property(nonatomic, weak) HWOptionButton *volume;
@property(nonatomic,strong) AudioController *audioController;
@property(nonatomic,strong) NuiSdkUtils *utils;
@property(nonatomic) int ttsSampleRate;
@end

@implementation DashCosyVoiceStreamInputTTSViewController

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static BOOL continuousPlaybackFlag = NO;
static BOOL SegmentFinishPlaying = NO;
static dispatch_queue_t tts_work_queue;

#pragma mark view controller methods

- (void)viewDidLoad {
    [super viewDidLoad];
    TLog(@"DashCosyVoiceStreamInputTTSViewController did load");
    
    _streamInputTtsSdk = [StreamInputTts get_instance];
    _streamInputTtsSdk.delegate = self;
    myself = self;

    TLog(@"Get API Key: %@", self.apikey);
    TLog(@"Get URL: %@", self.url);

    [self InitView];
    
    _utils = [NuiSdkUtils alloc];

    tts_work_queue = dispatch_queue_create("NuiCosyVoiceTtsController", DISPATCH_QUEUE_CONCURRENT);
}

-(void)dealloc {
    TLog(@"%s", __FUNCTION__);
    // 若_nui未进行释放, 下次进入此view使用的_nui处于已初始化,
    // 则再调用nui_tts_initialize无法覆盖已经设置的参数.
    if (_audioController != nil) {
        [_audioController cleanPlayerBuffer];
    }
}

- (void)dismissKeyboard:(id)sender {
    [self.view endEditing:YES];
}

-(void)viewDidAppear:(BOOL)animated {
    TLog(@"DashSambertTTSViewController-viewDidAppear");
    [super viewDidAppear:animated];
    [self InitView];
}

-(void)viewWillDisappear:(BOOL)animated {
    TLog(@"DashSambertTTSViewController-viewWillDisappear");

    if (_audioController != nil) {
        [_audioController cleanPlayerBuffer];
    }
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - Audio Player Delegate
-(void)playerDidFinish {
    //播放被中止后回调。
    TLog(@"playerDidFinish");
    SegmentFinishPlaying = YES;
    if (continuousPlaybackFlag == NO) {
        TLog(@"update UI of PlayButton");
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    }
}
-(void)playerDrainDataFinish {
    //播放数据自然播放完成后回调。
    TLog(@"playerDrainDataFinish");
    SegmentFinishPlaying = YES;
    if (continuousPlaybackFlag == NO) {
        TLog(@"update UI of PlayButton");
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    }
}

#pragma mark -private methods

-(NSString *)genTickets {
    NSString *debug_path = [_utils createDir];
    TLog(@"debug_path:%@", debug_path);

    NSMutableDictionary *ticketJsonDict = [NSMutableDictionary dictionary];

    // 注意！！！不要在端侧使用长效API Key！！！
    // 注意！！！不要在端侧使用长效API Key！！！
    // 注意！！！不要在端侧使用长效API Key！！！
    // 将长效API Key硬编码在端侧代码中，会导致安全风险！！！
    // 请在自建服务端获得临时鉴权Token（有效期60s，最长可设置1800s），再下发到端侧进行使用。
    // 临时鉴权Token: https://help.aliyun.com/zh/model-studio/obtain-temporary-authentication-token
    //
    // 服务只需要在临时Token(API Key)快过期前刷新一次。各端侧在Token(API Key)快过期前从服务获得新的
    // 临时Token(API Key)。
    [ticketJsonDict setObject:_apikey forKey:@"apikey"];

    if (_url == nil || _url.length == 0) {
        _url = @"wss://dashscope.aliyuncs.com/api-ws/v1/inference";
    }
    [ticketJsonDict setObject:_url forKey:@"url"];

    [ticketJsonDict setObject:@"empty_device_id" forKey:@"device_id"]; // 必填, 推荐填入具有唯一性的id, 方便定位问题

    //debug目录，当初始化SDK时的save_log参数取值为true时，该目录用于保存中间音频文件。
    [ticketJsonDict setObject:debug_path forKey:@"debug_path"];
    //过滤SDK内部日志通过回调送回到用户层
    [ticketJsonDict setObject:[NSString stringWithFormat:@"%d", NUI_LOG_LEVEL_NONE] forKey:@"log_track_level"];
    //设置本地存储日志文件的最大字节数, 最大将会在本地存储2个设置字节大小的日志文件
    [ticketJsonDict setObject:@(50 * 1024 * 1024) forKey:@"max_log_file_size"];

    NSData *data = [NSJSONSerialization dataWithJSONObject:ticketJsonDict options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

-(NSString *)genParameters {
    NSMutableString *jsonString = [NSMutableString string];
    
    @try {
        NSMutableDictionary *object = [[NSMutableDictionary alloc] init];
        
        NSArray<NSString *> *font_components = [_fontName.title componentsSeparatedByString:@"-"];
        NSString *firstPart = font_components.firstObject;
        [object setObject:firstPart forKey:@"voice"];
        [object setObject:_modelName.title forKey:@"model"];
        /**
         * 音频编码格式。
         * 所有模型均支持的编码格式：pcm、wav和mp3（默认）
         * 除cosyvoice-v1外，其他模型支持的编码格式：opus
         * 音频格式为opus时，支持通过bit_rate参数调整码率。
         * */
        [object setObject:_formatType.title forKey:@"format"];
        /**
         * 音频编码格式。
         * 所有模型均支持的编码格式：pcm、wav和mp3（默认）
         * 除cosyvoice-v1外，其他模型支持的编码格式：opus
         * 音频格式为opus时，支持通过bit_rate参数调整码率。
         * */
        BOOL enableDecoder = [_formatType.title isEqualToString:@"mp3"] || [_formatType.title isEqualToString:@"opus"];
        [object setObject:@(enableDecoder) forKey:@"enable_audio_decoder"];
        /**
         * 音频采样率，支持下述采样率（单位：Hz）：
         * 8000, 16000, 22050（默认）, 24000, 44100, 48000。
         * */
        [object setObject:@(24000) forKey:@"sample_rate"];
        /**
         * 音量，取值范围：0～100。默认值：50。
         * */
        NSInteger volume = [_volume.title integerValue];
        [object setObject:@(volume) forKey:@"volume"];
        /**
         * 合成音频的语速，取值范围：0.5~2。
         * 0.5：表示默认语速的0.5倍速。
         * 1：表示默认语速。默认语速是指模型默认输出的合成语速，语速会依据每一个音色略有不同，约每秒钟4个字。
         * 2：表示默认语速的2倍速。
         * 默认值：1.0。
         * */
        float rate = [_rate.title floatValue];
        [object setObject:@(rate) forKey:@"rate"];
        /**
         * 合成音频的语调，取值范围：0.5~2。
         * 默认值：1.0。
         * */
        float pitch = [_pitch.title floatValue];
        [object setObject:@(pitch) forKey:@"pitch"];

        /**
         * bit_rate:
         * 指定音频的码率，取值范围：6~510kbps。
         * 码率越大，音质越好，音频文件体积越大。
         * 仅在音频格式（format）为opus时可用。
         * cosyvoice-v1模型不支持该参数。
         * */
//        [object setObject:@(16) forKey:@"bit_rate"];

        // 特定模型：cosyvoice-v3 / cosyvoice-v3-plus
        NSString *model = _modelName.title;
        if ([model isEqualToString:@"cosyvoice-v3"] || [model isEqualToString:@"cosyvoice-v3-plus"]) {
            /**
             * seed:
             * 生成时使用的随机数种子，使合成的效果产生变化。默认值0。取值范围：0~65535。
             * 仅cosyvoice-v3、cosyvoice-v3-plus支持该功能。
             * */
//            [object setObject:@(0) forKey:@"seed"];

            /**
             * style:
             * 调整风格。默认值0。取值应为大于等于0的整数。
             * 仅cosyvoice-v3、cosyvoice-v3-plus支持该功能。
             * */
//            [object setObject:@(0) forKey:@"style"];
            
            /**
             * instruction:
             * 设置提示词。
             * 仅cosyvoice-v3、cosyvoice-v3-plus支持该功能。
             * 目前仅支持设置情感。
             * 格式：“你说话的情感是<情感值>。”。
             * 支持的情感值：Neutral、Fearful、Angry、Sad、Surprised、Happy、Disgusted。
             * */
//            NSString *instruction = @"Neutral";
//            if (instruction.length > 0) {
//                NSString *instructionPrompt = [NSString stringWithFormat:@"你说话的情感是%@。", instruction]; // 注意句号
//                [object setObject:instructionPrompt forKey:@"instruction"];
//            }
            
            /**
             * language_hints:
             * 设置合成语种。
             * 仅cosyvoice-v3、cosyvoice-v3-plus支持该功能。
             * 当前只支持同时配置一个语种。
             * 取值范围：
             * "zh"：中文
             * "en"：英文
             * */
        }

        NSError *jsonError;
        NSData *finalData = [NSJSONSerialization dataWithJSONObject:object options:NSJSONWritingPrettyPrinted error:&jsonError];
        if (finalData) {
            jsonString = [NSMutableString stringWithString:[[NSString alloc] initWithData:finalData encoding:NSUTF8StringEncoding]];
        } else {
            NSLog(@"生成 JSON 失败: %@", jsonError.localizedDescription);
        }

    } @catch (NSException *exception) {
        NSLog(@"异常: %@", exception);
    }

    NSLog(@"user parameters: %@", jsonString);

    return [jsonString copy];
}

-(CGFloat)InitTextView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    // ---- textViewContent ---
    CGFloat textViewContent_width = global_size.width - border * 2;
    CGFloat textViewContent_height = global_size.height/SCREEN_HEIGHT_BASE * 200;
    CGFloat textViewContent_x = border;
    CGFloat textViewContent_y = startY;

    CGRect textViewContent_rect = CGRectMake(textViewContent_x, textViewContent_y, textViewContent_width, textViewContent_height);
    if (!textViewContent) {
        textViewContent = [[UITextView alloc] initWithFrame:textViewContent_rect];
    }
    textViewContent.layer.borderWidth = 0.6;
    textViewContent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewContent.layer.cornerRadius = 10;
    [textViewContent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewContent.scrollEnabled = YES;
    textViewContent.text = @"唧唧复唧唧，木兰当户织。不闻机杼声，唯闻女叹息。\n问女何所思，问女何所忆。女亦无所思，女亦无所忆。\n昨夜见军帖，可汗大点兵，军书十二卷，卷卷有爷名。\n阿爷无大儿，木兰无长兄，愿为市鞍马，从此替爷征。\n东市买骏马，西市买鞍鞯，南市买辔头，北市买长鞭。旦辞爷娘去，暮宿黄河边，不闻爷娘唤女声，但闻黄河流水鸣溅溅。\n旦辞黄河去，暮至黑山头，不闻爷娘唤女声，但闻燕山胡骑鸣啾啾。万里赴戎机，关山度若飞。朔气传金柝，寒光照铁衣。将军百战死，壮士十年归。\n归来见天子，天子坐明堂。策勋十二转，赏赐百千强。可汗问所欲，木兰不用尚书郎，愿驰千里足，送儿还故乡。爷娘闻女来，出郭相扶将；阿姊闻妹来，当户理红妆；小弟闻姊来，磨刀霍霍向猪羊。\n开我东阁门，坐我西阁床。脱我战时袍，著我旧时裳。当窗理云鬓，对镜帖花黄。出门看火伴，火伴皆惊忙：同行十二年，不知木兰是女郎。\n雄兔脚扑朔，雌兔眼迷离；双兔傍地走，安能辨我是雄雌？";
    textViewContent.textColor = [UIColor darkGrayColor];
    textViewContent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewContent];
    
    // ---- textViewEvent ---
    CGFloat textViewEvent_width = textViewContent_width;
    CGFloat textViewEvent_height = global_size.height/SCREEN_HEIGHT_BASE * 100;
    CGFloat textViewEvent_x = border;
    CGFloat textViewEvent_y = startY + textViewContent_height + border;

    CGRect textViewEvent_rect = CGRectMake(textViewEvent_x, textViewEvent_y, textViewEvent_width, textViewEvent_height);
    if (!textViewEvent) {
        textViewEvent = [[UITextView alloc] initWithFrame:textViewEvent_rect];
    }
    textViewEvent.layer.borderWidth = 0.6;
    textViewEvent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewEvent.layer.cornerRadius = 10;
    [textViewEvent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewEvent.scrollEnabled = YES;
    textViewEvent.text = @"事件展示";
    textViewEvent.textColor = [UIColor darkGrayColor];
    textViewEvent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewEvent];
    
    return textViewEvent_y + textViewEvent_height;
}

-(CGFloat)InitButtonView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;

    // ---- StartButton ---
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 60;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 30;
    CGFloat new_border = (global_size.width - button_width * 4) / 5;
    CGFloat start_x = new_border;
    CGFloat start_y = startY;
    
    StartButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StartButton.frame = CGRectMake(start_x, start_y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [StartButton setBackgroundImage:image forState:UIControlStateNormal];
    [StartButton setTitle:@"合成" forState:UIControlStateNormal];
    [StartButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StartButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StartButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StartButton];
    
    // ---- PlayButton ---
    CGFloat PlayButton_x = start_x + button_width + new_border;
    CGFloat PlayButton_y = startY;
    
    PlayButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PlayButton.frame = CGRectMake(PlayButton_x, PlayButton_y, button_width, button_height);
    UIImage *Play_image = [UIImage imageNamed:@"button_start"];
    [PlayButton setBackgroundImage:Play_image forState:UIControlStateNormal];
    [PlayButton setTitle:@"长文本" forState:UIControlStateNormal];
    [PlayButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PlayButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PlayButton addTarget:self action:@selector(playTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PlayButton];

    // ---- StopButton ---
    CGFloat StopButton_x = PlayButton_x + button_width + new_border;
    CGFloat StopButton_y = startY;

    StopButton = [UIButton buttonWithType:UIButtonTypeCustom];
    StopButton.frame = CGRectMake(StopButton_x, StopButton_y, button_width, button_height);
    UIImage *StopButton_image = [UIImage imageNamed:@"button_start"];
    [StopButton setBackgroundImage:StopButton_image forState:UIControlStateNormal];
    [StopButton setTitle:@"停止" forState:UIControlStateNormal];
    [StopButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    StopButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [StopButton addTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:StopButton];

    // ---- CancelButton ---
    CGFloat CancelButton_x = StopButton_x + button_width + new_border;
    CGFloat CancelButton_y = startY;

    CancelButton = [UIButton buttonWithType:UIButtonTypeCustom];
    CancelButton.frame = CGRectMake(CancelButton_x, CancelButton_y, button_width, button_height);
    UIImage *CancelButton_image = [UIImage imageNamed:@"button_start"];
    [CancelButton setBackgroundImage:CancelButton_image forState:UIControlStateNormal];
    [CancelButton setTitle:@"取消" forState:UIControlStateNormal];
    [CancelButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    CancelButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [CancelButton addTarget:self action:@selector(cancelTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:CancelButton];

    return CancelButton_y + button_height;
}

-(CGFloat)InitHWOptionsView:(CGFloat)border startY:(CGFloat)startY {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    CGFloat hw_height = global_size.height / SCREEN_HEIGHT_BASE * 40;

    // ---- modelName ---
    CGFloat modelName_width = global_size.width - border * 2;
    CGFloat modelName_x = border;
    CGFloat modelName_y = startY;

    HWOptionButton *modelNameBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(modelName_x, modelName_y, modelName_width, hw_height)];
    modelNameBtn.array = @[@"cosyvoice-v2", @"cosyvoice-v3", @"cosyvoice-v3-plus", @"cosyvoice-v1"];
    modelNameBtn.selectIndex = 0;
    modelNameBtn.backgroundColor = [UIColor whiteColor];
    modelNameBtn.delegate = self;
    modelNameBtn.showPlaceholder = NO;
    modelNameBtn.showSearchBar = NO;
    modelNameBtn.dropdownTitle = @" - 模型-请于官网查看更多模型 - ";
    [self.view addSubview:modelNameBtn];
    self.modelName = modelNameBtn;

    // ---- fontName ---
    CGFloat fontName_width = global_size.width - border * 2;
    CGFloat fontName_x = border;
    CGFloat fontName_y = modelName_y + hw_height;

    HWOptionButton *fontNameBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(fontName_x, fontName_y, fontName_width, hw_height)];
    NSArray<NSString *> *voice_list = [_utils getVoiceList:_modelName.title];
//    NSLog(@"voice list: %@", voice_list);
    fontNameBtn.array = voice_list;
    fontNameBtn.selectIndex = 0;
    fontNameBtn.backgroundColor = [UIColor whiteColor];
    fontNameBtn.delegate = self;
    fontNameBtn.showPlaceholder = NO;
    fontNameBtn.showSearchBar = NO;
    fontNameBtn.dropdownTitle = @" - 说话人-请于官网查看更多说话人 - ";
    [self.view addSubview:fontNameBtn];
    self.fontName = fontNameBtn;

    // ---- formatType ---
    CGFloat formatType_width = global_size.width/2;
    CGFloat formatType_x = border;
    CGFloat formatType_y = fontName_y + hw_height;

    HWOptionButton *formatTypeBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(formatType_x, formatType_y, formatType_width, hw_height)];
    formatTypeBtn.array = @[@"opus", @"mp3", @"pcm", @"wav"];
    formatTypeBtn.selectIndex = 0;
    formatTypeBtn.backgroundColor = [UIColor whiteColor];
    formatTypeBtn.delegate = self;
    formatTypeBtn.showPlaceholder = NO;
    formatTypeBtn.showSearchBar = NO;
    formatTypeBtn.dropdownTitle = @" - 音频编码格式 - ";
    [self.view addSubview:formatTypeBtn];
    self.formatType = formatTypeBtn;

    // ---- pitch ---
    CGFloat pitch_width = global_size.width / 3 - border * 2;
    CGFloat pitch_x = border;
    CGFloat pitch_y = formatType_y + hw_height;

    HWOptionButton *pitchBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(pitch_x, pitch_y, pitch_width, hw_height)];
    pitchBtn.backgroundColor = [UIColor whiteColor];
    pitchBtn.array = @[@"1.0", @"0.5", @"0.8", @"1.5", @"2.0"];
    pitchBtn.selectIndex = 0;
    pitchBtn.delegate = self;
    pitchBtn.showSearchBar = NO;
    pitchBtn.showPlaceholder = NO;
    pitchBtn.dropdownTitle = @" - 语调 - ";
    [self.view addSubview:pitchBtn];
    self.pitch = pitchBtn;
    
    // ---- rate ---
    CGFloat rate_width = global_size.width / 3 - border * 2;
    CGFloat rate_x = pitch_x + pitch_width + border;
    CGFloat rate_y = pitch_y;

    HWOptionButton *rateBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(rate_x, rate_y, rate_width, hw_height)];
    rateBtn.backgroundColor = [UIColor whiteColor];
    rateBtn.array = @[@"1.0", @"0.5", @"0.8", @"1.5", @"2.0"];
    rateBtn.selectIndex = 0;
    rateBtn.delegate = self;
    rateBtn.showSearchBar = NO;
    rateBtn.showPlaceholder = NO;
    rateBtn.dropdownTitle = @" - 语速 - ";
    [self.view addSubview:rateBtn];
    self.rate = rateBtn;
    
    // ---- volume ---
    CGFloat volume_width = global_size.width / 3 - border * 2;
    CGFloat volume_x = rate_x + rate_width + border;
    CGFloat volume_y = pitch_y;

    HWOptionButton *volumeBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(volume_x, volume_y, volume_width, hw_height)];
    volumeBtn.backgroundColor = [UIColor whiteColor];
    volumeBtn.array = @[@"50", @"10", @"30", @"70", @"80", @"100"];
    volumeBtn.selectIndex = 0;
    volumeBtn.delegate = self;
    volumeBtn.showSearchBar = NO;
    volumeBtn.showPlaceholder = NO;
    volumeBtn.dropdownTitle = @" - 音量 - ";
    [self.view addSubview:volumeBtn];
    self.volume = volumeBtn;
    
    return volume_y + hw_height;
}

-(void)InitView {
    CGSize global_size = [UIScreen mainScreen].bounds.size;
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"Sambert语音合成";
    CGFloat border = global_size.width/SCREEN_WIDTH_BASE * 13;
    CGFloat division = global_size.height/SCREEN_HEIGHT_BASE * 10;
    CGFloat end_y = 0;

    CGFloat button_start_y = global_size.height/SCREEN_HEIGHT_BASE * 70;
    end_y = [self InitButtonView:border startY:button_start_y];

    CGFloat hwoptions_start_y = end_y + division;
    end_y = [self InitHWOptionsView:border startY:hwoptions_start_y];
    
    CGFloat view_start_y = end_y + division;
    end_y = [self InitTextView:border startY:view_start_y];
}

#pragma mark - Button Action
- (IBAction)startTTS:(UIButton *)sender {
    if (_streamInputTtsSdk != nil) {
        if (_audioController == nil) {
            // 注意：这里audioController模块仅用于播放示例，用户可根据业务场景自行实现这部分代码
            _audioController = [[AudioController alloc] init:only_player];
            _audioController.delegate = self;
            _ttsSampleRate = 24000;
            [_audioController setPlayerSampleRate:_ttsSampleRate];
        }

        NSString *tickets = [self genTickets];
        NSString *parameters = [self genParameters];

        dispatch_async(tts_work_queue, ^{
            int ret = [_streamInputTtsSdk startStreamInputTts:[tickets UTF8String] parameters:[parameters UTF8String] sessionId:nil logLevel:NUI_LOG_LEVEL_DEBUG saveLog:YES];
            if (ret != SUCCESS) {
                TLog(@"stream input tts start fail(%d)", ret);
            }
        });
    }
}

- (IBAction)playTTS:(UIButton *)sender {
    if (_streamInputTtsSdk != nil) {
        if (_audioController == nil) {
            // 注意：这里audioController模块仅用于播放示例，用户可根据业务场景自行实现这部分代码
            _audioController = [[AudioController alloc] init:only_player];
            _audioController.delegate = self;
            _ttsSampleRate = 24000;
            [_audioController setPlayerSampleRate:_ttsSampleRate];
        }

        NSString *tickets = [self genTickets];
        NSString *parameters = [self genParameters];

        dispatch_async(tts_work_queue, ^{
            int ret = [_streamInputTtsSdk asyncPlayStreamInputTts:[tickets UTF8String] parameters:[parameters UTF8String] text:[myself->textViewContent.text UTF8String] sessionId:nil logLevel:NUI_LOG_LEVEL_DEBUG saveLog:YES];
            if (ret != SUCCESS) {
                TLog(@"stream input tts play fail(%d)", ret);
            }
        });
    }
}

- (IBAction)stopTTS:(UIButton *)sender {
    if (_streamInputTtsSdk != nil) {
        TLog(@"DashCosyVoiceStreamInputTTSViewController stop tts");

        dispatch_async(tts_work_queue, ^{
            [_streamInputTtsSdk asyncStopStreamInputTts];
        });
    } else {
        TLog(@"in stopTTS, _streamInputTtsSdk == nil.");
    }
}

- (IBAction)cancelTTS:(UIButton *)sender {
    if (_streamInputTtsSdk != nil) {
        TLog(@"DashCosyVoiceStreamInputTTSViewController cancel tts");

        dispatch_async(tts_work_queue, ^{
            [_streamInputTtsSdk cancelStreamInputTts];
            if (_audioController != nil) {
                [_audioController stopPlayer];
            }
        });
    } else {
        TLog(@"in cancelTTS, _streamInputTtsSdk == nil.");
    }
}

#pragma stream input tts callback
- (void)onStreamInputTtsEventCallback:(StreamInputTtsCallbackEvent)event taskId:(char*)taskid sessionId:(char*)sessionId ret_code:(int)ret_code error_msg:(char*)error_msg timestamp:(char*)timestamp all_response:(char*)all_response {
//    NSString *log = [NSString stringWithFormat:@"\n事件回调（%d）：%s", event, all_response];
//    TLog(@"%@", log);

    if (event == TTS_EVENT_SYNTHESIS_STARTED) {
        dispatch_async(dispatch_get_main_queue(), ^{
            myself->textViewEvent.text = @"TTS_EVENT_SYNTHESIS_STARTED";
        });
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SYNTHESIS_STARTED");
        if (_audioController != nil) {
            // 启动播放器
            [_audioController startPlayer];
        }
    } else if (event == TTS_EVENT_SENTENCE_SYNTHESIS) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_SYNTHESIS");
    } else if (event == TTS_EVENT_SENTENCE_END) {
        TLog(@"onStreamInputTtsEventCallback TTS_EVENT_SENTENCE_END");
    } else if (event == TTS_EVENT_SYNTHESIS_COMPLETE || event == TTS_EVENT_TASK_FAILED) {
        // 注意这里的event事件是指语音合成完成，而非播放完成，播放完成需要由audioController对象来进行通知
        [_audioController drain];

        if (event == TTS_EVENT_TASK_FAILED) {
            NSString *errorCode = @"";
            NSString *allResponse = [NSString stringWithUTF8String:all_response];
            if (allResponse) {
                NSError *jsonError;
                NSData *jsonData = [allResponse dataUsingEncoding:NSUTF8StringEncoding];
                NSDictionary *jsonObject = [NSJSONSerialization JSONObjectWithData:jsonData options:0 error:&jsonError];
                if (!jsonError) {
                    NSDictionary *header = jsonObject[@"header"];
                    if ([header isKindOfClass:[NSDictionary class]] && header[@"error_code"]) {
                        errorCode = header[@"error_code"];
                    }
                }
            }
            
            dispatch_async(dispatch_get_main_queue(), ^{
                NSString *content = [NSString stringWithFormat:@"TTS_EVENT_TASK_FAILED error_message(%s) error_code(%@)",
                                      error_msg,
                                       errorCode];
                myself->textViewEvent.text = content;
            });
            TLog(@"onStreamInputTtsEventCallback TTS_EVENT_END");

        } else {
            dispatch_async(dispatch_get_main_queue(), ^{
                myself->textViewEvent.text = @"TTS_EVENT_SYNTHESIS_COMPLETE";
            });
            TLog(@"onStreamInputTtsEventCallback (%d) TTS_EVENT_TASK_FAILED", event);
            if (_audioController != nil) {
                // 取消播报、或者发生异常时终止播放
                [_audioController stopPlayer];
            }
        }
    }
}

- (void)onStreamInputTtsDataCallback:(char*)buffer len:(int)len {
//    NSString *log = [NSString stringWithFormat:@"\n音频回调 %d bytes", len];
//    TLog(log);
    if (buffer != NULL && len > 0 && _audioController != nil) {
        [_audioController write:(char*)buffer Length:(unsigned int)len];
    }
}

-(void)onStreamInputTtsLogTrackCallback:(NuiSdkLogLevel)level
                             logMessage:(const char *)log {
    TLog(@"onStreamInputTtsLogTrackCallback log level:%d, message -> %s", level, log);
}

#pragma mark - HWOptionButtonDelegate
- (void)didSelectOptionInHWOptionButton:(HWOptionButton *)optionButton {
    if (optionButton == self.modelName) {
        NSArray<NSString *> *voice_list = [_utils getVoiceList:_modelName.title];
    //    NSLog(@"voice list: %@", voice_list);
        self.fontName.array = voice_list;
    }
}

@end

