/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.aliyun.lindorm.ldspark.examples

import org.apache.spark.SparkConf
import org.apache.spark.sql.SparkSession

/**
 * This example shows how to manipulate data in "Lindorm Wide Column" engine:
 * 1. Reading data, and construct as Spark DataFrame;
 * 2. Define computing logic by SQL;
 * 3. Save result in another table;
 *
 * Before running below example, user should prepare an input table with some data, and an output table:
 * input_table -- student (id int, age int, name string, score int)
 * output_table -- student_res (id int, age int, name string, score int)
 */
object ReadWriteWideColumnTables {

  def main(args:Array[String]): Unit ={

    // Please fill in below information by refering to Lindorm document.
    val conf = new SparkConf()
    conf.set("spark.sql.catalog.lindorm_table", "com.alibaba.lindorm.ldspark.datasources.v2.LindormCatalog")
    conf.set("spark.sql.catalog.lindorm_table.url", "") // e.g. ld-XXXXXXXXXXXX.aliyuncs.com:30020
    conf.set("spark.sql.catalog.lindorm_table.username", "root") // e.g. root
    conf.set("spark.sql.catalog.lindorm_table.password", "root") // e.g. root

    val spark = SparkSession.builder()
      .config(conf)
      // Do not set master parameter please, Lindorm Spark will set the master address.
      .appName("LindormSparkSQLExamples")
      .getOrCreate()

    // 1. Construct DataFrame by sql.
    val studentDF = spark.sql("select * from lindorm_table.example_db.student;")
    studentDF.show()
    // Output:
    // +---+---+----+-----+
    // | id|age|name|score|
    // +---+---+----+-----+
    // |  3| 12|john|   96|
    // |  2| 13| tom|   98|
    // |  1| 12|jack|   95|
    // +---+---+----+-----+

    /// 2. Compute by DataFrame API.
    studentDF.where("age = 12").show()
    // Output:
    // +---+---+----+-----+
    // | id|age|name|score|
    // +---+---+----+-----+
    // |  3| 12|john|   96|
    // |  1| 12|jack|   95|
    // +---+---+----+-----+

    // 3. Compute by SQL.
    spark.sql("select * from lindorm_table.example_db.student where score = 95;").show()
    // Output:
    // +---+---+----+-----+
    // | id|age|name|score|
    // +---+---+----+-----+
    // |  1| 12|jack|   95|
    // +---+---+----+-----+

    // 4. Save result by "INSERT INTO" clause.
    spark.sql("insert into student_res select * from lindorm_table.example_db.student where score = 95;").show()
    // sleep 5min. You can check the job status at Alibaba Cloud console
    Thread.sleep(5 * 60 * 1000)
    spark.close()
  }

}
