from flask import Flask, render_template, jsonify
from alibabacloud_tea_openapi.models import Config
from alibabacloud_sts20150401.client import Client as Sts20150401Client
from alibabacloud_sts20150401 import models as sts_20150401_models
from alibabacloud_credentials.client import Client as CredentialClient
import os
import json
import base64
import hmac
import datetime
import time
import hashlib

app = Flask(__name__)

# Configure environment variables OSS_ACCESS_KEY_ID, OSS_ACCESS_KEY_SECRET, and OSS_STS_ROLE_ARN.
access_key_id = os.environ.get('OSS_ACCESS_KEY_ID')
access_key_secret = os.environ.get('OSS_ACCESS_KEY_SECRET')
role_arn_for_oss_upload = os.environ.get('OSS_STS_ROLE_ARN')  # RAM Role ARN

# Replace with actual bucket name and region-id
bucket = 'web-direct-upload'
region_id = 'cn-hangzhou'

host = f'http://{bucket}.oss-cn-hangzhou.aliyuncs.com'

expire_time = 3600  # Specify expiration time in seconds
upload_dir = 'dir'  # Specify the file prefix for uploading to OSS.
role_session_name = 'role_session_name'  # Custom session name

def hmacsha256(key, data):
    """
    Function to calculate HMAC-SHA256 hash value

    :param key: Key used for hashing, byte type
    :param data: Data to be hashed, string type
    :return: Calculated HMAC-SHA256 hash value, byte type
    """
    try:
        mac = hmac.new(key, data.encode(), hashlib.sha256)
        hmacBytes = mac.digest()
        return hmacBytes
    except Exception as e:
        raise RuntimeError(f"Failed to calculate HMAC-SHA256 due to {e}")

@app.route("/")
def hello_world():
    return render_template('index.html')

@app.route('/get_post_signature_for_oss_upload', methods=['GET'])
def generate_upload_params():
    # Initialize configuration, directly pass credentials
    config = Config(
        region_id=region_id,
        access_key_id=access_key_id,
        access_key_secret=access_key_secret
    )

    # Create STS client and get temporary credentials
    sts_client = Sts20150401Client(config=config)
    assume_role_request = sts_20150401_models.AssumeRoleRequest(
        role_arn=role_arn_for_oss_upload,
        role_session_name=role_session_name
    )
    response = sts_client.assume_role(assume_role_request)
    token_data = response.body.credentials.to_map()

    # Use temporary credentials returned by STS
    sts_access_key_id = token_data['AccessKeyId']
    sts_access_key_secret = token_data['AccessKeySecret']
    security_token = token_data['SecurityToken']

    now = int(time.time())
    # Convert timestamp to datetime object
    dt_obj = datetime.datetime.utcfromtimestamp(now)
    # Add 3 hours to the current time, set as the request's expiration time
    dt_obj_plus_3h = dt_obj + datetime.timedelta(hours=3)

    # Request time
    dt_obj_1 = dt_obj.strftime('%Y%m%dT%H%M%S') + 'Z'
    # Request date
    dt_obj_2 = dt_obj.strftime('%Y%m%d')
    # Request expiration time
    expiration_time = dt_obj_plus_3h.strftime('%Y-%m-%dT%H:%M:%S.000Z')

    # Build Policy and generate signature
    policy = {
        "expiration": expiration_time,
        "conditions": [
            ["eq", "$success_action_status", "200"],
            {"x-oss-signature-version": "OSS4-HMAC-SHA256"},
            {"x-oss-credential": f"{sts_access_key_id}/{dt_obj_2}/cn-hangzhou/oss/aliyun_v4_request"},
            {"x-oss-security-token": security_token},
            {"x-oss-date": dt_obj_1},
        ]
    }
    print(dt_obj_1)
    policy_str = json.dumps(policy).strip()

    # Step 2: Construct the string to sign (StringToSign)
    stringToSign = base64.b64encode(policy_str.encode()).decode()

    # Step 3: Calculate SigningKey
    dateKey = hmacsha256(("aliyun_v4" + sts_access_key_secret).encode(), dt_obj_2)
    dateRegionKey = hmacsha256(dateKey, "cn-hangzhou")
    dateRegionServiceKey = hmacsha256(dateRegionKey, "oss")
    signingKey = hmacsha256(dateRegionServiceKey, "aliyun_v4_request")

    # Step 4: Calculate Signature
    result = hmacsha256(signingKey, stringToSign)
    signature = result.hex()

    # Organize response data
    response_data = {
        'policy': stringToSign,  # Form field
        'x_oss_signature_version': "OSS4-HMAC-SHA256",  # Specifies the version and algorithm of the signature, fixed value is OSS4-HMAC-SHA256
        'x_oss_credential': f"{sts_access_key_id}/{dt_obj_2}/cn-hangzhou/oss/aliyun_v4_request",  # Indicates the parameter set for deriving the key
        'x_oss_date': dt_obj_1,  # Request time
        'signature': signature,  # Signature authentication description information
        'host': host,
        'dir': upload_dir,
        'security_token': security_token  # Security token
    }
    return jsonify(response_data)


if __name__ == "__main__":
    app.run(host="127.0.0.1", port=8000)  # If you need to listen on other addresses like 0.0.0.0, you need to add an authentication mechanism on the server side.