package com.aliyun.service;

import com.aliyun.entity.Test;
import com.aliyun.mapper.TestMapper;
import com.clickhouse.data.ClickHouseOutputStream;
import com.clickhouse.data.ClickHouseWriter;
import com.zaxxer.hikari.HikariDataSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.util.List;

@Service
public class TestService {
    @Autowired
    TestMapper testMapper;
    @Autowired
    private HikariDataSource dataSource;

    /**
     * create table
     * @param enterprise true for enterprise version
     * @throws Exception
     */
    public void createTable(boolean enterprise) throws Exception {
        try(Connection conn = dataSource.getConnection()) {
            if (enterprise) {
                conn.createStatement().execute("CREATE TABLE IF NOT EXISTS `default`.`test` ON CLUSTER default (id Int64, name String) ENGINE = MergeTree() ORDER BY id;");
            } else {
                // create local table
                conn.createStatement().execute("CREATE TABLE IF NOT EXISTS `default`.`test_local` ON CLUSTER default (id Int64, name String) ENGINE = MergeTree() ORDER BY id;");
                // create distributed table
                conn.createStatement().execute("CREATE TABLE IF NOT EXISTS `default`.`test` ON CLUSTER default (id Int64, name String) ENGINE = Distributed(default, default, test_local, rand());");
            }
        }
    }

    /**
     * select by id
     * @param id id
     * @return Test
     * @throws Exception
     */
    public Test selectById(Long id) throws Exception {
        return testMapper.selectById(id);
    }

    /**
     * count table
     * @return rows number
     * @throws Exception
     */
    public Long count() throws Exception {
        return testMapper.count();
    }

    /**
     * batch insert by native
     * @param tests tests
     * @param optimizeLevel insert optimize level, 3 is faster than 2, 2 is faster than 1<br/>
     *                      1: insert into `default`.`test` (id, name) values(?, ?) -- with additional query for getting table structure.
     *                         It's portable.<br/>
     *                      2: insert into `default`.`test` select id, name from input('id Int64, name String') -- effectively convert and insert data sent to the server
     *                         with given structure to the table with another structure. It's NOT portable(as it's limited to ClickHouse).<br/>
     *                      3: insert into `default`.`test` format RowBinary -- fastest(close to Java client) with streaming mode but requires manual serialization and it's
     *                         NOT portable(as it's limited to ClickHouse).
     * @throws Exception
     */
    public void insertTestsByNative(List<Test> tests, int optimizeLevel) throws Exception {
        try(Connection conn = dataSource.getConnection()) {
            // prepared statement
            PreparedStatement preparedStatement = null;
            switch (optimizeLevel) {
                case 1:
                    preparedStatement = conn.prepareStatement("insert into `default`.`test` (id, name) values(?, ?)");
                    break;
                case 2:
                    preparedStatement = conn.prepareStatement("insert into `default`.`test` select id, name from input('id Int64, name String')");
                    break;
                case 3:
                    preparedStatement = conn.prepareStatement("insert into `default`.`test` format RowBinary");
                    break;
                default:
                    throw new IllegalArgumentException("optimizeLevel must be 1, 2 or 3");
            }

            // insert data
            switch (optimizeLevel) {
                case 1:
                case 2:
                    for (Test test : tests) {
                        preparedStatement.setLong(1, test.id);
                        preparedStatement.setString(2, test.name);
                        preparedStatement.addBatch();
                    }
                    preparedStatement.executeBatch();
                    break;
                case 3:
                    class MyClickHouseWriter implements ClickHouseWriter {
                        @Override
                        public void write(ClickHouseOutputStream clickHouseOutputStream) throws IOException {
                            for (Test test : tests) {
                                // write id(Int64)
                                ByteBuffer buffer = ByteBuffer.allocate(Long.BYTES);
                                buffer.order(ByteOrder.LITTLE_ENDIAN);
                                buffer.putLong(test.id);
                                clickHouseOutputStream.write(buffer.array());
                                // write name(String)
                                clickHouseOutputStream.writeUnicodeString(test.name);
                            }
                        }
                    }
                    preparedStatement.setObject(1, new MyClickHouseWriter());
                    preparedStatement.executeUpdate();
                    break;
            }
        }
    }

    /**
     * batch insert by mybatis<br/>
     * Note: It is NOT SUGGESTED to use mybatis' FOREACH to insert large number(>100) of records.
     * @param tests tests
     * @throws Exception
     */
    public void insertTestsByMyBatis(List<Test> tests) throws Exception {
        testMapper.insertTests(tests);
    }
}
