// Copyright (C) 2015, Alibaba Cloud Computing

#ifndef MNS_PROTOCOL_H
#define MNS_PROTOCOL_H

#include <iostream>
#include <fstream>
#include <map>
#include <vector>
#include <string>
#include "pugixml.hpp"

namespace mns
{
namespace sdk
{

struct ErrorInfo
{
    std::string errorMessage;
    std::string code;
    std::string requestId;
    std::string hostId;
    int httpStatus;
};

enum CursorType
{
    CursorNull,
    BEGIN,
    NEXT
};

enum NotifyStrategy
{
    STRATEGY_NULL,
    BACKOFF_RETRY,
    EXPONENTIAL_DECAY_RETRY
};

enum SubscriptionState
{
    StateNull,
    ACTIVE,
    INACTIVE
};

struct SendMessageItem
{
    SendMessageItem(const std::string& body = "",
                    const int32_t delaySec = -1,
                    const int32_t prio = -1)
        : messageBody(body)
        , delaySeconds(delaySec)
        , priority(prio)
    {
    }
    std::string messageBody;
    int32_t delaySeconds;
    int32_t priority;
};

struct SendMessageSucceedItem
{
    SendMessageSucceedItem()
        : messageId(""), messageBodyMD5("")
    {
    }
    std::string messageId;
    std::string messageBodyMD5;
};

struct DeleteMessageFailedItem
{
    DeleteMessageFailedItem()
        : errorCode("")
        , errorMessage("")
        , receiptHandle("")
    {
    }
    std::string errorCode;
    std::string errorMessage;
    std::string receiptHandle;
};

struct SendMessageFailedItem
{
    SendMessageFailedItem()
        : errorCode("")
        , errorMessage("")
    {
    }
    std::string errorCode;
    std::string errorMessage;
};

class ReceiveMessageResponse;
class BatchReceiveMessageResponse;

class Message
{
public:
    Message()
        : mMessageId("")
        , mReceiptHandle("")
        , mMessageBody("")
        , mMessageBodyMD5("")
        , mEnqueueTime(-1)
        , mNextVisibleTime(-1)
        , mFirstDequeueTime(-1)
        , mDequeueCount(-1)
        , mPriority(-1)
    {
    }

    const std::string& getMessageId()
    {
        return mMessageId;
    }

    const std::string& getReceiptHandle()
    {
        return mReceiptHandle;
    }

    const std::string& getMessageBody()
    {
        return mMessageBody;
    }

    const std::string& getMessageBodyMD5()
    {
        return mMessageBodyMD5;
    }

    int64_t getEnqueueTime()
    {
        return mEnqueueTime;
    }

    int64_t getFirstDequeueTime()
    {
        return mFirstDequeueTime;
    }

    int64_t getNextVisibleTime()
    {
        return mNextVisibleTime;
    }

    int32_t getDequeueCount()
    {
        return mDequeueCount;
    }

    int32_t getPriority()
    {
        return mPriority;
    }

    friend class ReceiveMessageResponse;
    friend class BatchReceiveMessageResponse;
    friend class PeekMessageResponse;
    friend class BatchPeekMessageResponse;

protected:
    void initFromXml(const pugi::xml_node& messageNode);

protected:
    std::string mMessageId;
    std::string mReceiptHandle;
    std::string mMessageBody;
    std::string mMessageBodyMD5;
    int64_t mEnqueueTime;
    int64_t mNextVisibleTime;
    int64_t mFirstDequeueTime;
    int32_t mDequeueCount;
    int32_t mPriority;
};

class GetQueueAttributesResponse;

class QueueAttributes
{
public:
    QueueAttributes()
    {
        mQueueName = "";
        mVisibilityTimeout = -1;
        mDelaySeconds = -1;
        mMaximumMessageSize = -1;
        mMessageRetentionPeriod = -1;
        mActiveMessages = -1;
        mInactiveMessages = -1;
        mDelayMessages = -1;
        mCreateTime = -1;
        mLastModifyTime = -1;
        mPollingWaitSeconds = -1;
    }

    int64_t getVisibilityTimeout() const
    {
        return mVisibilityTimeout;
    }

    void setVisibilityTimeout(int64_t visibilityTimeout)
    {
        mVisibilityTimeout = visibilityTimeout;
    }

    int64_t getDelaySeconds() const
    {
        return mDelaySeconds;
    }

    void setDelaySeconds(int64_t delaySeconds)
    {
        mDelaySeconds = delaySeconds;
    }

    int64_t getMaximumMessageSize() const
    {
        return mMaximumMessageSize;
    }

    void setMaximumMessageSize(int64_t maximumMessageSize)
    {
        mMaximumMessageSize = maximumMessageSize;
    }

    int64_t getMessageRetentionPeriod() const
    {
        return mMessageRetentionPeriod;
    }

    void setMessageRetentionPeriod(int64_t messageRetentionPeriod)
    {
        mMessageRetentionPeriod = messageRetentionPeriod;
    }

    int32_t getPollingWaitSeconds() const
    {
        return mPollingWaitSeconds;
    }

    void setPollingWaitSeconds(int32_t pollingWaitSeconds)
    {
        mPollingWaitSeconds = pollingWaitSeconds;
    }

    std::string getQueueName() const
    {
        return mQueueName;
    }

    int64_t getActiveMessages() const
    {
        return mActiveMessages;
    }

    int64_t getInactiveMessages() const
    {
        return mInactiveMessages;
    }

    int64_t getDelayMessages() const
    {
        return mDelayMessages;
    }

    int64_t getCreateTime() const
    {
        return mCreateTime;
    }

    int64_t getLastModifyTime() const
    {
        return mLastModifyTime;
    }

    friend class GetQueueAttributesResponse;

protected:
    std::string mQueueName;
    int64_t mVisibilityTimeout;
    int64_t mDelaySeconds;
    int64_t mMaximumMessageSize;
    int64_t mMessageRetentionPeriod;
    int64_t mActiveMessages;
    int64_t mInactiveMessages;
    int64_t mDelayMessages;
    int64_t mCreateTime;
    int64_t mLastModifyTime;
    int32_t mPollingWaitSeconds;
};

class Request
{
public:
    Request(const std::string& method)
        : mMethod(method), mCanonicalizedResource(""), mRequestBody("")
    {}
    virtual ~Request() {};

    const std::string& getMethod() const
    {
        return mMethod;
    }

    const std::string& getCanonicalizedResource()
    {
        if (mCanonicalizedResource == "")
        {
            return generateCanonicalizedResource();
        }
        return mCanonicalizedResource;
    }

    const std::string& generateCanonicalizedResource()
    {
        mCanonicalizedResource = getResourcePath();
        if (getQueryString() != "")
        {
            mCanonicalizedResource += "?" + getQueryString();
        }
        return mCanonicalizedResource;
    }

    const std::string& getRequestBody()
    {
        if (mRequestBody == "")
        {
            return generateRequestBody();
        }
        return mRequestBody;
    }

    virtual std::string getQueryString() = 0;

    virtual std::string getResourcePath() = 0;

    virtual const std::string& generateRequestBody() = 0;

    const std::map<std::string, std::string>& getHeaders()
    {
        return mHeaders;
    }

    std::string getHeader(const std::string& key)
    {
        typeof(mHeaders.begin()) iter = mHeaders.find(key);
        if (iter != mHeaders.end())
        {
            return iter->second;
        }
        return "";
    }

    void setHeader(std::string key, std::string value)
    {
        mHeaders[key] = value;
    }

protected:
    std::string mMethod;
    std::string mCanonicalizedResource;
    std::string mRequestBody;
    std::map<std::string, std::string> mHeaders;
};

class Response
{
public:
    Response();
    virtual ~Response() {};

    int getStatus()
    {
        return mStatus;
    }

    void setStatus(int32_t status)
    {
        mStatus = status;
    }

    const std::map<std::string, std::string>& getHeaders()
    {
        return mHeaders;
    }

    void setHeader(std::string key, std::string value)
    {
        mHeaders[key] = value;
    }

    std::string getHeader(std::string key)
    {
        typeof(mHeaders.begin()) iter = mHeaders.find(key);
        if (iter != mHeaders.end())
        {
            return iter->second;
        }
        return NULL;
    }

    std::string* getRawDataPtr()
    {
        return &mRawData;
    }

    void clearRawData()
    {
        mRawData.clear();
    }

    virtual bool isSuccess() = 0;
    virtual void parseResponse() = 0;

protected:
    pugi::xml_node toXML();
    void parseCommonError(const pugi::xml_node& rootNode);
    bool isCommonError(const pugi::xml_node& rootNode);

protected:
    pugi::xml_document mDoc;
    std::map<std::string, std::string> mHeaders;
    std::string mRawData;
    int32_t mStatus;
};

class CreateQueueRequest : public Request
{
public:
    CreateQueueRequest(const std::string& queueName);
    CreateQueueRequest(const std::string& queueName,
        const QueueAttributes& attributes);

    std::string getQueueName()
    {
        return mQueueName;
    }

    std::string getQueryString();
    std::string getResourcePath();
    const std::string& generateRequestBody();
protected:
    std::string mQueueName;
    const QueueAttributes* mAttributes;
};

class CreateQueueResponse : public Response
{
public:
    CreateQueueResponse();
    virtual ~CreateQueueResponse() {};

    std::string getQueueURL()
    {
        return mQueueURL;
    }

    void setQueueURL(std::string queueURL)
    {
        mQueueURL = queueURL;
    }

    void parseResponse();
    bool isSuccess();

protected:
    std::string mQueueURL;
};

class DeleteQueueRequest : public Request
{
public:
    DeleteQueueRequest(const std::string& queueName);
    virtual ~DeleteQueueRequest() {}

    std::string getQueryString();
    std::string getResourcePath();
    const std::string& generateRequestBody();
protected:
    std::string mQueueName;
};

class DeleteQueueResponse : public Response
{
public:
    void parseResponse();
    bool isSuccess();
};

class ListQueueRequest : public Request
{
public:
    ListQueueRequest(const std::string& prefix = "",
                     const std::string& marker = "",
                     const int32_t retNum = -1);
    virtual ~ListQueueRequest() {}

    void setMarker(const std::string& marker);
    void setPrefix(const std::string& prefix);
    void setRetNum(const int32_t retNum);

    std::string getQueryString();
    std::string getResourcePath();
    const std::string& generateRequestBody();

protected:
    std::string mPrefix;
    std::string mMarker;
    int32_t mRetNum;
};

class ListQueueResponse : public Response
{
public:
    ListQueueResponse(std::vector<std::string>& queueNames);
    virtual ~ListQueueResponse() {}

    void parseResponse();
    bool isSuccess();

protected:
    std::vector<std::string>* mQueueNames;
};

class GetQueueAttributesRequest : public Request
{
public:
    GetQueueAttributesRequest(const std::string& queueName);
    virtual ~GetQueueAttributesRequest() {}

    std::string getQueryString();
    std::string getResourcePath();
    const std::string& generateRequestBody();

protected:
    std::string mQueueName;
};

class GetQueueAttributesResponse : public Response
{
public:
    GetQueueAttributesResponse(QueueAttributes& attributes);
    virtual ~GetQueueAttributesResponse() {}

    void parseResponse();
    bool isSuccess();

protected:
    QueueAttributes* mQueueAttributes;
};

class SetQueueAttributesRequest : public Request
{
public:
    SetQueueAttributesRequest(const std::string& queueName,
        const QueueAttributes& queueAttributes);
    virtual ~SetQueueAttributesRequest() {}

    std::string getQueryString();
    std::string getResourcePath();
    const std::string& generateRequestBody();

protected:
    std::string mQueueName;
    const QueueAttributes* mQueueAttributes;
};

class SetQueueAttributesResponse : public Response
{
public:
    bool isSuccess();
    void parseResponse();
};

class MessageRequest : public Request
{
public:
    MessageRequest(const std::string& method)
        : Request(method), mQueueName(NULL)
    {
    }

    virtual ~MessageRequest() {}

    std::string getResourcePath()
    {
        if (mQueueName == NULL)
            return "";
        return "/queues/" + *mQueueName + "/messages";
    }

    /* QueueName will be automatically set when sending request. */
    void setQueueName(const std::string& queueName)
    {
        mQueueName = &queueName;
    }

protected:
    const std::string* mQueueName;
};

class SendMessageRequest : public MessageRequest
{
public:
    SendMessageRequest(const std::string& messageBody,
        const int32_t delaySeconds = -1, const int32_t priority = -1);
    virtual ~SendMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    const std::string* mMessageBody;
    int32_t mDelaySeconds;
    int32_t mPriority;
};

class SendMessageResponse : public Response
{
public:
    SendMessageResponse();
    virtual ~SendMessageResponse() {}

    void parseResponse();
    bool isSuccess();

protected:
    std::string mMessageId;
    std::string mMessageBodyMD5;
};

class BatchSendMessageRequest : public MessageRequest
{
public:
    BatchSendMessageRequest(
        const std::vector<SendMessageItem>& sendMessageItems);
    virtual ~BatchSendMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    const std::vector<SendMessageItem>* mSendMessageItems;
};

class BatchSendMessageResponse : public Response
{
public:
    void parseResponse();
    bool isSuccess();

    const std::vector<SendMessageSucceedItem>& getSendMessageSucceedItems()
    {
        return mSendMessageSucceedItems;
    }

    const std::vector<SendMessageFailedItem>& getSendMessageFailedItems()
    {
        return mSendMessageFailedItems;
    }

protected:
    void parseBatchSendResponse(pugi::xml_node& rootNode);

protected:
    std::vector<SendMessageSucceedItem> mSendMessageSucceedItems;
    std::vector<SendMessageFailedItem> mSendMessageFailedItems;
};

class ReceiveMessageRequest : public MessageRequest
{
public:
    ReceiveMessageRequest(const int32_t waitSeconds = -1);
    virtual ~ReceiveMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    int32_t mWaitSeconds;
};

class ReceiveMessageResponse : public Response
{
public:
    ReceiveMessageResponse(Message& message);
    virtual ~ReceiveMessageResponse() {}

    void parseResponse();
    bool isSuccess();

protected:
    Message* mMessage;
};

class BatchReceiveMessageRequest : public MessageRequest
{
public:
    BatchReceiveMessageRequest(const int32_t numOfMessages,
        const int32_t waitSeconds = -1);
    virtual ~BatchReceiveMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    int32_t mNumOfMessages;
    int32_t mWaitSeconds;
};

class BatchReceiveMessageResponse : public Response
{
public:
    BatchReceiveMessageResponse(std::vector<Message>& messages);
    virtual ~BatchReceiveMessageResponse() {}

    void parseResponse();
    bool isSuccess();

protected:
    std::vector<Message>* mMessages;
};

class DeleteMessageRequest : public MessageRequest
{
public:
    DeleteMessageRequest(const std::string receiptHandle);
    virtual ~DeleteMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    std::string mReceiptHandle;
};

class DeleteMessageResponse : public Response
{
public:
    void parseResponse();
    bool isSuccess();
};

class BatchDeleteMessageRequest : public MessageRequest
{
public:
    BatchDeleteMessageRequest(const std::vector<std::string>& receiptHandles);
    virtual ~BatchDeleteMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    const std::vector<std::string>* mReceiptHandles;
};

class BatchDeleteMessageResponse : public Response
{
public:
    void parseResponse();
    bool isSuccess();

    const std::vector<DeleteMessageFailedItem>& getDeleteMessageFailedItem()
    {
        return mDeleteMessageFailedItems;
    }

protected:
    std::vector<DeleteMessageFailedItem> mDeleteMessageFailedItems;
};

class PeekMessageRequest : public MessageRequest
{
public:
    PeekMessageRequest();
    virtual ~PeekMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();
};

class PeekMessageResponse : public Response
{
public:
    PeekMessageResponse(Message& message);
    virtual ~PeekMessageResponse() {}

    void parseResponse();
    bool isSuccess();

protected:
    Message* mMessage;
};

class BatchPeekMessageRequest : public MessageRequest
{
public:
    BatchPeekMessageRequest(const int32_t numOfMessages);
    virtual ~BatchPeekMessageRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    int32_t mNumOfMessages;
};

class BatchPeekMessageResponse : public Response
{
public:
    BatchPeekMessageResponse(std::vector<Message>& mMessages);
    virtual ~BatchPeekMessageResponse() {}

    void parseResponse();
    bool isSuccess();

protected:
    std::vector<Message>* mMessages;
};

class ChangeMessageVisibilityRequest : public MessageRequest
{
public:
    ChangeMessageVisibilityRequest(const std::string& receiptHandle,
        const int32_t visibilityTimeout);
    virtual ~ChangeMessageVisibilityRequest() {}

    std::string getQueryString();
    const std::string& generateRequestBody();

protected:
    std::string mReceiptHandle;
    int32_t mVisibilityTimeout;
};

class ChangeMessageVisibilityResponse : public Response
{
public:
    void parseResponse();
    bool isSuccess();

    const std::string& getReceiptHandle()
    {
        return mReceiptHandle;
    }

    int64_t getNextVisibleTime()
    {
        return mNextVisibleTime;
    }

protected:
    std::string mReceiptHandle;
    int64_t mNextVisibleTime;
};

}
}

#endif
