#include "mns_protocol.h"
#include "mns_utils.h"
#include "mns_exception.h"
#include "mns_network_tool.h"
#include "mns_common_tool.h"
#include "mns_exception.h"
#include "constants.h"
#include "mns_utils.h"

#include <iostream>

using namespace std;
using namespace pugi;
using namespace mns::sdk;

void Message::initFromXml(const xml_node& messageNode)
{
    xml_node iterNode = messageNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(MESSAGE_ID, name))
        {
            mMessageId = iterNode.text().get();
        }
        else if (0 == strcmp(RECEIPT_HANDLE, name))
        {
            mReceiptHandle = iterNode.text().get();
        }
        else if (0 == strcmp(MESSAGE_BODY, name))
        {
            mMessageBody = iterNode.text().get();
        }
        else if (0 == strcmp(MESSAGE_BODY_MD5, name))
        {
            mMessageBodyMD5 = iterNode.text().get();
        }
        else if (0 == strcmp(ENQUEUE_TIME, name))
        {
            mEnqueueTime = atol(iterNode.text().get());
        }
        else if (0 == strcmp(FIRST_DEQUEUE_TIME, name))
        {
            mFirstDequeueTime = atol(iterNode.text().get());
        }
        else if (0 == strcmp(NEXT_VISIBLE_TIME, name))
        {
            mNextVisibleTime = atol(iterNode.text().get());
        }
        else if (0 == strcmp(DEQUEUE_COUNT, name))
        {
            mDequeueCount = atoi(iterNode.text().get());
        }
        else if (0 == strcmp(PRIORITY, name))
        {
            mPriority = atoi(iterNode.text().get());
        }

        iterNode = iterNode.next_sibling();
    }
}

Response::Response()
    : mRawData(""), mStatus(0)
{
}

bool Response::isCommonError(const pugi::xml_node& rootNode)
{
    const pugi::char_t* name = rootNode.name();
    return 0 == strcmp(ERROR, name);
}

pugi::xml_node Response::toXML()
{
    mDoc.reset();
    pugi::xml_parse_result result = mDoc.load_string(mRawData.c_str());
    if (!result)
    {
        MNS_THROW(MNSExceptionBase, mRawData);
    }
    xml_node iterNode = mDoc.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() == node_element) {
            return iterNode;
        }

        iterNode = iterNode.next_sibling();
    }
    MNS_THROW(MNSExceptionBase, mRawData);
}

void Response::parseCommonError(const pugi::xml_node& rootNode)
{
    if (!isCommonError(rootNode))
    {
        MNS_THROW(MNSExceptionBase, "Unknown CommonError XML: " + mRawData);
    }

    ErrorInfo errorInfo;
    errorInfo.httpStatus = mStatus;

    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(CODE, name))
            errorInfo.code = iterNode.text().get();
        else if (0 == strcmp(MESSAGE, name))
            errorInfo.errorMessage = iterNode.text().get();
        else if (0 == strcmp(REQUEST_ID, name))
            errorInfo.requestId = iterNode.text().get();
        else if (0 == strcmp(HOST_ID, name))
            errorInfo.hostId = iterNode.text().get();

        iterNode = iterNode.next_sibling();
    }
    MNS_THROW(MNSServerException, errorInfo);
}

CreateQueueRequest::CreateQueueRequest(const string& queueName)
    : mQueueName(queueName), Request("PUT"), mAttributes(NULL)
{
}

CreateQueueRequest::CreateQueueRequest(const string& queueName,
                                       const QueueAttributes& attributes)
    : mQueueName(queueName), mAttributes(&attributes), Request("PUT")
{
}

std::string CreateQueueRequest::getQueryString()
{
    return "";
}

std::string CreateQueueRequest::getResourcePath()
{
    return "/queues/" + mQueueName;
}

const string& CreateQueueRequest::generateRequestBody()
{
    pugi::xml_document doc;
    doc.load("<?xml version=\"1.0\" encoding=\"UTF-8\"?>", pugi::parse_declaration);
    pugi::xml_node node = doc.append_child("Queue");
    node.append_attribute("xmlns") = MNS_XML_NAMESPACE_V1;

    if (mAttributes != NULL)
    {
        string visibilityTimeoutStr = StringTool::ToString(
            mAttributes->getVisibilityTimeout());
        string maximumMessageSizeStr = StringTool::ToString(
            mAttributes->getMaximumMessageSize());
        string messageRetentionPeriodStr = StringTool::ToString(
            mAttributes->getMessageRetentionPeriod());
        string delaySecondsStr = StringTool::ToString(
            mAttributes->getDelaySeconds());
        string pollingWaitSecondsStr = StringTool::ToString(
            mAttributes->getPollingWaitSeconds());

        if (mAttributes->getVisibilityTimeout() > 0)
            node.append_child("VisibilityTimeout").append_child(
                pugi::node_pcdata).set_value(visibilityTimeoutStr.c_str());

        if (mAttributes->getMaximumMessageSize() > 0)
            node.append_child("MaximumMessageSize").append_child(
                pugi::node_pcdata).set_value(maximumMessageSizeStr.c_str());

        if (mAttributes->getMessageRetentionPeriod() > 0)
            node.append_child("MessageRetentionPeriod").append_child(
                pugi::node_pcdata).set_value(messageRetentionPeriodStr.c_str());

        if (mAttributes->getDelaySeconds() > 0)
            node.append_child("DelaySeconds").append_child(
                pugi::node_pcdata).set_value(delaySecondsStr.c_str());

        if (mAttributes->getPollingWaitSeconds() > 0)
            node.append_child("PollingWaitSeconds").append_child(
                pugi::node_pcdata).set_value(pollingWaitSecondsStr.c_str());
    }

    ostringstream os;
    doc.save(os);
    mRequestBody = os.str();
    return mRequestBody;
}

CreateQueueResponse::CreateQueueResponse()
    : mQueueURL(""), Response()
{
}

bool CreateQueueResponse::isSuccess()
{
    return mStatus == 201 || mStatus == 204;
}

void CreateQueueResponse::parseResponse()
{
    if (isSuccess())
    {
        mQueueURL = getHeader("Location");
        return;
    }

    pugi::xml_node rootNode = toXML();
    parseCommonError(rootNode);
}

DeleteQueueRequest::DeleteQueueRequest(const string& queueName)
    : mQueueName(queueName), Request("DELETE")
{
}

string DeleteQueueRequest::getQueryString()
{
    return "";
}

string DeleteQueueRequest::getResourcePath()
{
    return "/queues/" + mQueueName;
}

const string& DeleteQueueRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

bool DeleteQueueResponse::isSuccess()
{
    return mStatus == 204;
}

void DeleteQueueResponse::parseResponse()
{
    if (isSuccess())
    {
        return;
    }

    pugi::xml_node rootNode = toXML();
    parseCommonError(rootNode);
}

ListQueueRequest::ListQueueRequest(const string& prefix,
                                   const string& marker,
                                   const int32_t retNum)
    : Request("GET")
{
    setMarker(marker);
    setPrefix(prefix);
    setRetNum(retNum);
}

void ListQueueRequest::setMarker(const string& marker)
{
    mMarker = marker;
    if (mMarker != "")
    {
        mHeaders["x-mns-marker"] = mMarker;
    }
    else
    {
        mHeaders.erase("x-mns-marker");
    }
}

void ListQueueRequest::setPrefix(const string& prefix)
{
    mPrefix = prefix;
    if (mPrefix != "")
    {
        mHeaders["x-mns-prefix"] = mPrefix;
    }
    else
    {
        mHeaders.erase("x-mns-prefix");
    }
}

void ListQueueRequest::setRetNum(const int32_t retNum)
{
    mRetNum = retNum;
    if (mRetNum > 0)
    {
        mHeaders["x-mns-ret-number"] = mRetNum;
    }
    else
    {
        mHeaders.erase("x-mns-ret-number");
    }
}

string ListQueueRequest::getQueryString()
{
    return "";
}

string ListQueueRequest::getResourcePath()
{
    return "/queues";
}

const string& ListQueueRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

ListQueueResponse::ListQueueResponse(std::vector<std::string>& queueNames)
    : mQueueNames(&queueNames), Response()
{
}

bool ListQueueResponse::isSuccess()
{
    return mStatus == 200;
}

void ListQueueResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(QUEUE_URL, name))
        {
            string queueName = MNSUtils::getQueueNameFromURL(
                    iterNode.text().get());
            mQueueNames->push_back(queueName);
        }

        iterNode = iterNode.next_sibling();
    }
}

GetQueueAttributesRequest::GetQueueAttributesRequest(const string& queueName)
    : mQueueName(queueName), Request("GET")
{
}

string GetQueueAttributesRequest::getQueryString()
{
    return "";
}

string GetQueueAttributesRequest::getResourcePath()
{
    return "/queues/" + mQueueName;
}

const string& GetQueueAttributesRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

GetQueueAttributesResponse::GetQueueAttributesResponse(
    QueueAttributes& attributes)
    : mQueueAttributes(&attributes), Response()
{
}

bool GetQueueAttributesResponse::isSuccess()
{
    return mStatus == 200;
}

void GetQueueAttributesResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(QUEUE_NAME, name))
        {
            mQueueAttributes->mQueueName = iterNode.text().get();
        }
        else if (0 == strcmp(CREATE_TIME, name))
        {
            mQueueAttributes->mCreateTime = atol(iterNode.text().get());
        }
        else if (0 == strcmp(LAST_MODIFY_TIME, name))
        {
            mQueueAttributes->mLastModifyTime = atol(iterNode.text().get());
        }
        else if (0 == strcmp(VISIBILITY_TIMEOUT, name))
        {
            mQueueAttributes->mVisibilityTimeout = atol(iterNode.text().get());
        }
        else if (0 == strcmp(DELAY_SECONDS, name))
        {
            mQueueAttributes->mDelaySeconds = atol(iterNode.text().get());
        }
        else if (0 == strcmp(MAXIMUM_MESSAGE_SIZE, name))
        {
            mQueueAttributes->mMaximumMessageSize = atol(iterNode.text().get());
        }
        else if (0 == strcmp(MAXIMUM_MESSAGE_SIZE, name))
        {
            mQueueAttributes->mMessageRetentionPeriod = atol(iterNode.text().get());
        }
        else if (0 == strcmp(ACTIVE_MESSAGES, name))
        {
            mQueueAttributes->mActiveMessages = atol(iterNode.text().get());
        }
        else if (0 == strcmp(INACTIVE_MESSAGES, name))
        {
            mQueueAttributes->mInactiveMessages = atol(iterNode.text().get());
        }
        else if (0 == strcmp(DELAY_MESSAGES, name))
        {
            mQueueAttributes->mDelayMessages = atol(iterNode.text().get());
        }
        else if (0 == strcmp(POLLING_WAIT_SECONDS, name))
        {
            mQueueAttributes->mPollingWaitSeconds = atoi(iterNode.text().get());
        }

        iterNode = iterNode.next_sibling();
    }
}

SetQueueAttributesRequest::SetQueueAttributesRequest(
    const string& queueName,
    const QueueAttributes& queueAttributes)
    : mQueueName(queueName), mQueueAttributes(&queueAttributes), Request("PUT")
{
}

string SetQueueAttributesRequest::getQueryString()
{
    return "metaoverride=true";
}

string SetQueueAttributesRequest::getResourcePath()
{
    return "/queues/" + mQueueName;
}

const string& SetQueueAttributesRequest::generateRequestBody()
{
    pugi::xml_document doc;
    doc.load("<?xml version=\"1.0\" encoding=\"UTF-8\"?>", pugi::parse_declaration);
    pugi::xml_node node = doc.append_child("Queue");
    node.append_attribute("xmlns") = MNS_XML_NAMESPACE_V1;

    string visibilityTimeoutStr = StringTool::ToString(
        mQueueAttributes->getVisibilityTimeout());
    string maximumMessageSizeStr = StringTool::ToString(
        mQueueAttributes->getMaximumMessageSize());
    string messageRetentionPeriodStr = StringTool::ToString(
        mQueueAttributes->getMessageRetentionPeriod());
    string delaySecondsStr = StringTool::ToString(
        mQueueAttributes->getDelaySeconds());
    string pollingWaitSecondsStr = StringTool::ToString(
        mQueueAttributes->getPollingWaitSeconds());

    if (mQueueAttributes->getVisibilityTimeout() > 0)
        node.append_child("VisibilityTimeout").append_child(
            pugi::node_pcdata).set_value(visibilityTimeoutStr.c_str());

    if (mQueueAttributes->getMaximumMessageSize() > 0)
        node.append_child("MaximumMessageSize").append_child(
            pugi::node_pcdata).set_value(maximumMessageSizeStr.c_str());

    if (mQueueAttributes->getMessageRetentionPeriod() > 0)
        node.append_child("MessageRetentionPeriod").append_child(
            pugi::node_pcdata).set_value(messageRetentionPeriodStr.c_str());

    if (mQueueAttributes->getDelaySeconds() > 0)
        node.append_child("DelaySeconds").append_child(
            pugi::node_pcdata).set_value(delaySecondsStr.c_str());

    if (mQueueAttributes->getPollingWaitSeconds() > 0)
        node.append_child("PollingWaitSeconds").append_child(
            pugi::node_pcdata).set_value(pollingWaitSecondsStr.c_str());

    ostringstream os;
    doc.save(os);
    mRequestBody = os.str();
    return mRequestBody;
}

bool SetQueueAttributesResponse::isSuccess()
{
    return mStatus == 204;
}

void SetQueueAttributesResponse::parseResponse()
{
    if (isSuccess())
    {
        return;
    }

    pugi::xml_node rootNode = toXML();
    parseCommonError(rootNode);
}

SendMessageRequest::SendMessageRequest(const std::string& messageBody,
                                       const int32_t delaySeconds,
                                       const int32_t priority)
    : MessageRequest("POST"), mMessageBody(&messageBody),
      mDelaySeconds(delaySeconds), mPriority(priority)
{
}

string SendMessageRequest::getQueryString()
{
    return "";
}

const string& SendMessageRequest::generateRequestBody()
{
    pugi::xml_document doc;
    doc.load("<?xml version=\"1.0\" encoding=\"UTF-8\"?>", pugi::parse_declaration);
    pugi::xml_node node = doc.append_child("Message");
    node.append_attribute("xmlns") = MNS_XML_NAMESPACE_V1;

    node.append_child("MessageBody").append_child(
        pugi::node_pcdata).set_value(mMessageBody->c_str());

    string delaySecondsStr = StringTool::ToString(mDelaySeconds);
    string priorityStr = StringTool::ToString(mPriority);

    if (mDelaySeconds > 0)
        node.append_child("DelaySeconds").append_child(
            pugi::node_pcdata).set_value(delaySecondsStr.c_str());

    if (mPriority > 0)
        node.append_child("Priority").append_child(
            pugi::node_pcdata).set_value(priorityStr.c_str());

    ostringstream os;
    doc.save(os);
    mRequestBody = os.str();
    return mRequestBody;
}

SendMessageResponse::SendMessageResponse()
    : mMessageId(""), mMessageBodyMD5("")
{
}

bool SendMessageResponse::isSuccess()
{
    return mStatus == 201;
}

void SendMessageResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(MESSAGE_ID, name))
        {
            mMessageId = iterNode.text().get();
        }
        else if (0 == strcmp(MESSAGE_BODY_MD5, name))
        {
            mMessageBodyMD5 = iterNode.text().get();
        }

        iterNode = iterNode.next_sibling();
    }
}

BatchSendMessageRequest::BatchSendMessageRequest(
    const std::vector<SendMessageItem>& sendMessageItems)
    : MessageRequest("POST"), mSendMessageItems(&sendMessageItems)
{
}

std::string BatchSendMessageRequest::getQueryString()
{
    return "";
}

const std::string& BatchSendMessageRequest::generateRequestBody()
{
    pugi::xml_document doc;
    doc.load("<?xml version=\"1.0\" encoding=\"UTF-8\"?>", pugi::parse_declaration);
    pugi::xml_node node = doc.append_child("Messages");
    node.append_attribute("xmlns") = MNS_XML_NAMESPACE_V1;

    for (typeof(mSendMessageItems->begin()) iter = mSendMessageItems->begin();
        iter != mSendMessageItems->end(); ++iter)
    {
        pugi::xml_node messgeNode = node.append_child("Message");

        messgeNode.append_child("MessageBody").append_child(
            pugi::node_pcdata).set_value(iter->messageBody.c_str());

        string delaySecondsStr = StringTool::ToString(iter->delaySeconds);
        string priorityStr = StringTool::ToString(iter->priority);

        if (iter->delaySeconds > 0)
            messgeNode.append_child("DelaySeconds").append_child(
                pugi::node_pcdata).set_value(delaySecondsStr.c_str());

        if (iter->priority > 0)
            messgeNode.append_child("Priority").append_child(
                pugi::node_pcdata).set_value(priorityStr.c_str());
    }

    ostringstream os;
    doc.save(os);
    mRequestBody = os.str();
    return mRequestBody;
}

bool BatchSendMessageResponse::isSuccess()
{
    return mStatus == 201;
}

void BatchSendMessageResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (isSuccess())
    {
        parseBatchSendResponse(rootNode);
        return;
    }

    if (isCommonError(rootNode))
    {
        parseCommonError(rootNode);
    }
    else
    {
        parseBatchSendResponse(rootNode);
    }
}

void BatchSendMessageResponse::parseBatchSendResponse(xml_node& rootNode)
{
    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(MESSAGE, name))
        {
            SendMessageSucceedItem succeedItem;
            SendMessageFailedItem failedItem;

            xml_node iterMessageNode = iterNode.first_child();
            while (!iterMessageNode.empty())
            {
                if (iterMessageNode.type() != node_element)
                {
                    continue;
                }
                const pugi::char_t* iterName = iterMessageNode.name();
                if (0 == strcmp(MESSAGE_ID, iterName))
                {
                    succeedItem.messageId = iterMessageNode.text().get();
                }
                else if (0 == strcmp(MESSAGE_BODY_MD5, iterName))
                {
                    succeedItem.messageBodyMD5 = iterMessageNode.text().get();
                }
                else if (0 == strcmp(ERROR_MESSAGE, iterName))
                {
                    failedItem.errorMessage = iterMessageNode.text().get();
                }
                else if (0 == strcmp(ERROR_CODE, iterName))
                {
                    failedItem.errorCode = iterMessageNode.text().get();
                }

                iterMessageNode = iterMessageNode.next_sibling();
            }

            if (succeedItem.messageId != "")
            {
                mSendMessageSucceedItems.push_back(succeedItem);
            }
            else
            {
                mSendMessageFailedItems.push_back(failedItem);
            }
        }

        iterNode = iterNode.next_sibling();
    }
}

ReceiveMessageRequest::ReceiveMessageRequest(const int32_t waitSeconds)
    : MessageRequest("GET"), mWaitSeconds(waitSeconds)
{
}

std::string ReceiveMessageRequest::getQueryString()
{
    if (mWaitSeconds > 0)
    {
        return "waitseconds=" + StringTool::ToString(mWaitSeconds);
    }
    return "";
}

const std::string& ReceiveMessageRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

ReceiveMessageResponse::ReceiveMessageResponse(Message& message)
    : Response(), mMessage(&message)
{
}

void ReceiveMessageResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    mMessage->initFromXml(rootNode);
}

bool ReceiveMessageResponse::isSuccess()
{
    return mStatus == 200;
}

BatchReceiveMessageResponse::BatchReceiveMessageResponse(
    std::vector<Message>& messages)
    : Response(), mMessages(&messages)
{
}

BatchReceiveMessageRequest::BatchReceiveMessageRequest(
    const int32_t numOfMessages,
    const int32_t waitSeconds)
    : MessageRequest("GET"), mNumOfMessages(numOfMessages),
      mWaitSeconds(waitSeconds)
{
}

std::string BatchReceiveMessageRequest::getQueryString()
{
    string queryStr = "numOfMessages=" + StringTool::ToString(mNumOfMessages);
    if (mWaitSeconds > 0)
    {
        queryStr += "&waitseconds=" + StringTool::ToString(mWaitSeconds);
    }
    return queryStr;
}

const std::string& BatchReceiveMessageRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

void BatchReceiveMessageResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    int32_t index = 0;
    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(MESSAGE, name))
        {
            Message message;
            mMessages->push_back(message);
            mMessages->at(index).initFromXml(iterNode);
            index += 1;
        }

        iterNode = iterNode.next_sibling();
    }
}

bool BatchReceiveMessageResponse::isSuccess()
{
    return mStatus == 200;
}

DeleteMessageRequest::DeleteMessageRequest(const std::string receiptHandle)
    : MessageRequest("DELETE"), mReceiptHandle(receiptHandle)
{
}

std::string DeleteMessageRequest::getQueryString()
{
    return "ReceiptHandle=" + mReceiptHandle;
}

const std::string& DeleteMessageRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

void DeleteMessageResponse::parseResponse()
{
    if (!isSuccess())
    {
        pugi::xml_node rootNode = toXML();
        parseCommonError(rootNode);
        return;
    }
}

bool DeleteMessageResponse::isSuccess()
{
    return mStatus == 204;
}

BatchDeleteMessageRequest::BatchDeleteMessageRequest(
    const std::vector<std::string>& receiptHandles)
    : MessageRequest("DELETE"), mReceiptHandles(&receiptHandles)
{
}

std::string BatchDeleteMessageRequest::getQueryString()
{
    return "";
}

const std::string& BatchDeleteMessageRequest::generateRequestBody()
{
    pugi::xml_document doc;
    doc.load("<?xml version=\"1.0\" encoding=\"UTF-8\"?>", pugi::parse_declaration);
    pugi::xml_node node = doc.append_child("ReceiptHandles");
    node.append_attribute("xmlns") = MNS_XML_NAMESPACE_V1;

    for (typeof(mReceiptHandles->begin()) iter = mReceiptHandles->begin();
        iter != mReceiptHandles->end(); ++iter)
    {
        node.append_child("ReceiptHandle").append_child(
            pugi::node_pcdata).set_value(iter->c_str());
    }

    ostringstream os;
    doc.save(os);
    mRequestBody = os.str();
    return mRequestBody;
}

void BatchDeleteMessageResponse::parseResponse()
{
    if (isSuccess())
    {
        return;
    }

    pugi::xml_node rootNode = toXML();

    if (isCommonError(rootNode))
    {
        parseCommonError(rootNode);
        return;
    }

    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(ERROR, name))
        {
            DeleteMessageFailedItem failedItem;

            xml_node iterErrorNode = iterNode.first_child();
            while (!iterErrorNode.empty())
            {
                if (iterErrorNode.type() != node_element)
                {
                    continue;
                }
                const pugi::char_t* itemName = iterErrorNode.name();
                if (0 == strcmp(ERROR_MESSAGE, itemName))
                {
                    failedItem.errorMessage = iterErrorNode.text().get();
                }
                else if (0 == strcmp(ERROR_CODE, itemName))
                {
                    failedItem.errorCode = iterErrorNode.text().get();
                }
                else if (0 == strcmp(RECEIPT_HANDLE, itemName))
                {
                    failedItem.receiptHandle = iterErrorNode.text().get();
                }

                iterErrorNode = iterErrorNode.next_sibling();
            }
            mDeleteMessageFailedItems.push_back(failedItem);
        }

        iterNode = iterNode.next_sibling();
    }
}

bool BatchDeleteMessageResponse::isSuccess()
{
    return mStatus == 204;
}

PeekMessageRequest::PeekMessageRequest()
    : MessageRequest("GET")
{
}

std::string PeekMessageRequest::getQueryString()
{
    return "peekonly=true";
}

const std::string& PeekMessageRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

PeekMessageResponse::PeekMessageResponse(Message& message)
    : Response(), mMessage(&message)
{
}

void PeekMessageResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    mMessage->initFromXml(rootNode);
}

bool PeekMessageResponse::isSuccess()
{
    return mStatus == 200;
}

BatchPeekMessageRequest::BatchPeekMessageRequest(
    const int32_t numOfMessages)
    : MessageRequest("GET"), mNumOfMessages(numOfMessages)
{
}

std::string BatchPeekMessageRequest::getQueryString()
{
    return "peekonly=true&numOfMessages="
        + StringTool::ToString(mNumOfMessages);
}

const std::string& BatchPeekMessageRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

BatchPeekMessageResponse::BatchPeekMessageResponse(
    std::vector<Message>& messages)
    : Response(), mMessages(&messages)
{
}

void BatchPeekMessageResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    int32_t index = 0;
    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(MESSAGE, name))
        {
            Message message;
            mMessages->push_back(message);
            mMessages->at(index).initFromXml(iterNode);
            index += 1;
        }

        iterNode = iterNode.next_sibling();
    }
}

bool BatchPeekMessageResponse::isSuccess()
{
    return mStatus == 200;
}

ChangeMessageVisibilityRequest::ChangeMessageVisibilityRequest(
    const std::string& receiptHandle,
    const int32_t visibilityTimeout)
    : MessageRequest("PUT"), mReceiptHandle(receiptHandle),
      mVisibilityTimeout(visibilityTimeout)
{
}

std::string ChangeMessageVisibilityRequest::getQueryString()
{
    return "receiptHandle=" + mReceiptHandle
        + "&visibilityTimeout=" + StringTool::ToString(mVisibilityTimeout);
}

const std::string& ChangeMessageVisibilityRequest::generateRequestBody()
{
    mRequestBody = "";
    return mRequestBody;
}

void ChangeMessageVisibilityResponse::parseResponse()
{
    pugi::xml_node rootNode = toXML();
    if (!isSuccess())
    {
        parseCommonError(rootNode);
        return;
    }

    xml_node iterNode = rootNode.first_child();
    while (!iterNode.empty())
    {
        if (iterNode.type() != node_element)
        {
            continue;
        }
        const pugi::char_t* name = iterNode.name();
        if (0 == strcmp(RECEIPT_HANDLE, name))
        {
            mReceiptHandle = iterNode.text().get();
        }
        else if (0 == strcmp(NEXT_VISIBLE_TIME, name))
        {
            mNextVisibleTime = atol(iterNode.text().get());
        }

        iterNode = iterNode.next_sibling();
    }
}

bool ChangeMessageVisibilityResponse::isSuccess()
{
}
