// Copyright (C) 2015, Alibaba Cloud Computing

#ifndef MNS_SDK_NETWORK_H
#define MNS_SDK_NETWORK_H

#include "mns_exception.h"
#include "mns_common_tool.h"

#include <queue>
#include <curl/curl.h>
#include <errno.h>
#include <tr1/memory>
#include <pthread.h>
#include <iostream>
#include <fstream>
#include <sstream>
#include <map>
#include <vector>

namespace mns
{
namespace sdk
{

#define MNS_LOCK_SAFE(x) \
    do                      \
    {                       \
        int r = (x);        \
        if (r != 0)         \
        {                   \
            int e = errno;  \
            std::cerr << "MNS_LOCK_SAFE to abort() @ " << __FILE__ << ":" <<__LINE__ << " in " << __FUNCTION__ << std::endl \
                      << "Ret:" << r << " errno:" << e << "(" << strerror(e) << ')' << std::endl; \
            abort();        \
        }                   \
    } while(false)


class PTMutex
{
public:
    PTMutex()
    {
        MNS_LOCK_SAFE(pthread_mutex_init(&mutex, NULL));
    }

    ~PTMutex()
    {
        MNS_LOCK_SAFE(pthread_mutex_destroy(&mutex));
    }

    void lock()
    {
        MNS_LOCK_SAFE(pthread_mutex_lock(&mutex));
    }

    void unlock()
    {
        MNS_LOCK_SAFE(pthread_mutex_unlock(&mutex));
    }

    pthread_mutex_t& get_pthread_mutex() const
    {
        return mutex;
    }

protected:
    mutable pthread_mutex_t mutex;
private:
#ifdef APSARA_X0125
    PTMutex(const PTMutex&);
    PTMutex& operator=(const PTMutex&);
#endif
};

class PTRecursiveMutex
{
public:
    PTRecursiveMutex()
    {
        pthread_mutexattr_t mta;
        MNS_LOCK_SAFE(pthread_mutexattr_init(&mta));
        MNS_LOCK_SAFE(pthread_mutexattr_settype(&mta, PTHREAD_MUTEX_RECURSIVE));
        MNS_LOCK_SAFE(pthread_mutex_init(&mutex, &mta));
        MNS_LOCK_SAFE(pthread_mutexattr_destroy(&mta));
    }

    ~PTRecursiveMutex()
    {
        MNS_LOCK_SAFE(pthread_mutex_destroy(&mutex));
    }

    void lock()
    {
        MNS_LOCK_SAFE(pthread_mutex_lock(&mutex));
    }

    int trylock()
    {
        return pthread_mutex_trylock(&mutex);
    }

    void unlock()
    {
        MNS_LOCK_SAFE(pthread_mutex_unlock(&mutex));
    }

    pthread_mutex_t& get_pthread_mutex() const
    {
        return mutex;
    }

protected:
    mutable pthread_mutex_t mutex;
};
typedef PTRecursiveMutex Mutex;

class PTCond
{
public:
    PTCond()
    {
        MNS_LOCK_SAFE(pthread_cond_init(&cond, NULL));
    }

    ~PTCond()
    {
        MNS_LOCK_SAFE(pthread_cond_destroy(&cond));
    }

    int wait(const PTMutex& mutex, int64_t usec)
    {
        int ret;

        if (usec == -1)
        {
            // never fail
            ret = pthread_cond_wait(&cond, &mutex.get_pthread_mutex());
        }
        else
        {
            if (usec < 0)
            {
                MNS_THROW(MNSExceptionBase, "PTCond::wait only accepts positve usec (expect usec = -1 for infinity wait)");
            }
            timespec ts;
            timeval tp;
            gettimeofday(&tp, 0);

#define MICROSECONDS_PER_SECOND 1000000
#define NANOSECONDS_PER_MICROSECONDS 1000

            /// Calculate "second" part. Caution, usec may exceeds MICROSECONDS_PER_SECOND.
            ts.tv_sec = tp.tv_sec + (usec / MICROSECONDS_PER_SECOND);
            /// Calcuate "microsecond" part.
            int usec_sum = tp.tv_usec + usec % MICROSECONDS_PER_SECOND;

            /// Adjust "second" part if usec_sum exceeds MICROSECONDS_PER_SECOND
            ts.tv_sec += usec_sum / MICROSECONDS_PER_SECOND;
            usec_sum %= MICROSECONDS_PER_SECOND;

            /// Calcuate nanosecond
            ts.tv_nsec = usec_sum * NANOSECONDS_PER_MICROSECONDS;
#undef MICROSECONDS_PER_SECOND
#undef NANOSECONDS_PER_MICROSECONDS
            // fail if timed-out or interupted by a signal
            ret = pthread_cond_timedwait(&cond, &mutex.get_pthread_mutex(), &ts);
        }

        return ret;
    }

    void signal()
    {
        MNS_LOCK_SAFE(pthread_cond_signal(&cond));
    }

    void broadcast()
    {
        MNS_LOCK_SAFE(pthread_cond_broadcast(&cond));
    }

protected:
    pthread_cond_t cond;
};


class WaitObject
{
    friend class PTScopedLock;
    PTMutex mutex;
    PTCond cond;
public:
    WaitObject() {}
    int wait(int64_t timeout = -1)
    {
        return cond.wait(mutex, timeout);
    }
    void signal()
    {
        cond.signal();
    }
    void broadcast()
    {
        cond.broadcast();
    }
};

typedef std::tr1::shared_ptr<WaitObject> WaitObjectPtr;

class PTScopedLock
{
private:
    PTMutex& mDevice;
    PTScopedLock(const PTScopedLock&);
    PTScopedLock& operator=(const PTScopedLock&);

public:
    // lock when construct
    explicit PTScopedLock(PTMutex& m) : mDevice(m)
    {
        mDevice.lock();
    }
    explicit PTScopedLock(WaitObject& wo) : mDevice(wo.mutex)
    {
        mDevice.lock();
    }

    // unlock when destruct
    ~PTScopedLock()
    {
        mDevice.unlock();
    }
};

class ScopedLock
{
private:
    Mutex* mDevice;
    ScopedLock(const ScopedLock&);
    ScopedLock& operator=(const ScopedLock&);

public:
    ScopedLock() : mDevice(0) {}

    // lock when construct
    explicit ScopedLock(Mutex& m) : mDevice(&m)
    {
        mDevice->lock();
    }

    // Try to lock m. If successful, return true, and m will be
    // unlocked when this ScopedLock destructs. If failed, return
    // false.
    bool TryLock(Mutex& m)
    {
        if (mDevice)
            MNS_THROW(MNSExceptionBase, "ScopedLock::TryLock: this lock is already tied to a Mutex");

        int n = m.trylock();
        if (n == 0)
        {
            mDevice = &m;
            return true;
        }
        else if (n == EBUSY)
            return false;
        else
            MNS_THROW(MNSExceptionBase, "ScopedLock::TryLock: pthread_mutex_trylock returned "+StringTool::ToString(n));
    }

    // unlock when destruct
    ~ScopedLock()
    {
        if (mDevice)
        {
            mDevice->unlock();
        }
    }
private:
    uint64_t mStartTime;
};


class MNSConnectionTool
{
public:
    MNSConnectionTool(const int32_t curlPoolSize)
        : mCurlPoolSize(curlPoolSize)
        , mCurrentPoolSize(0)
    {
    }
    ~MNSConnectionTool();

    CURL* InvokeCurlConnection(bool& isLongConnection);
    void RevokeCurlConnection(CURL* curlConnection, const bool isLongConnection);

private:
    int32_t mCurlPoolSize;
    int32_t mCurrentPoolSize;
    WaitObject mWaitObject;

    std::queue<CURL*> mCurlPool;
};
typedef std::tr1::shared_ptr<MNSConnectionTool> MNSConnectionToolPtr;

class MNSNetworkTool
{
public:
    static void SendRequest(const std::string& endpoint,
                            Request& req,
                            Response& resp,
                            MNSConnectionToolPtr mnsConnTool);

    static std::string Signature(const std::string& method,
                                 const std::string& canonicalizedResource,
                                 const std::string& accessId,
                                 const std::string& accessKey,
                                 const std::map<std::string, std::string>& headers);

protected:
    static char* Sign(const char* data, const std::string& accessId, const std::string& accessKey);
    static char* Base64(const char* input, unsigned int& length);
};

}
}

#endif
