#include "mns_exception.h"

#include <execinfo.h>
#include <stdlib.h>
#include <cxxabi.h>
#include <iostream>
#include <sstream>

using namespace std;
using namespace mns::sdk;

MNSExceptionBase::MNSExceptionBase(const std::string& msg) throw()
    : mMsg(msg)
    , mFile("<unknown file>")
    , mFunc("<unknown func>")
    , mLine(-1)
    , mStackTraceSize(0)
{}

MNSExceptionBase::~MNSExceptionBase() throw()
{}

void MNSExceptionBase::Init(const char* file, const char* func, int line)
{
    mFile = file;
    mFunc = func;
    mLine = line;
    mStackTraceSize = backtrace(mStackTrace, MAX_STACK_TRACE_SIZE);
}

std::string MNSExceptionBase::GetClassName() const
{
    return "MNSExceptionBase";
}

const char* MNSExceptionBase::what() const throw()
{
    return ToString().c_str();
}

const std::string& MNSExceptionBase::ToString() const
{
    if (mWhat.empty())
    {
        stringstream sstr("");
        if (mLine > 0)
        {
            sstr << mFile << "(" << mLine << ")";
        }
        sstr << ": " << GetClassName();
        if (!GetMessage().empty())
        {
            sstr << ": " << GetMessage();
        }
        sstr << "\nStack Trace:\n";
        sstr << GetStackTrace();
        mWhat = sstr.str();
    }
    return mWhat;
}

std::string MNSExceptionBase::GetMessage() const
{
    return mMsg;
}

std::string MNSExceptionBase::GetStackTrace() const
{
    if (mStackTraceSize == 0)
        return "<No stack trace>\n";
    char** strings = backtrace_symbols(mStackTrace, mStackTraceSize);
    if (strings == NULL)
        return "<Unknown error: backtrace_symbols returned NULL>\n";

    std::string result;
    for (size_t i = 0; i < mStackTraceSize; ++i)
    {
        std::string mangledName = strings[i];
        std::string::size_type begin = mangledName.find('(');
        std::string::size_type end = mangledName.find('+', begin);
        if (begin == std::string::npos || end == std::string::npos)
        {
            result += mangledName;
            result += '\n';
            continue;
        }
        ++begin;
        int status;
        char* s = abi ::__cxa_demangle(mangledName.substr(begin, end-begin).c_str(), NULL, 0, &status);

        if (status != 0)
        {
            result += mangledName;
            result += '\n';
            continue;
        }

        std::string demangledName(s);
        free(s);

        result += mangledName.substr(0, begin);
        result += demangledName;
        result += mangledName.substr(end);
        result += '\n';
    }
    free(strings);
    return result;
}
