// Copyright (C) 2015, Alibaba Cloud Computing

#ifndef MNS_CLIENT_H
#define MNS_CLIENT_H

#include "mns_protocol.h"
#include "mns_network_tool.h"
#include <map>
#include <vector>
#include <tr1/memory>

namespace mns
{
namespace sdk
{

class Queue;
class MNSClient;

typedef std::tr1::shared_ptr<Queue> QueuePtr;

/*
 * use MNSClient to access MNS service
 *
 * CAUTION:
 *     Make sure to catch Exception when calling any function
 *     MNSClient throws MNSServerException when the request fails.
 *     MNSClient throws MNSExceptionBase for client errors
 */
class MNSClient
{
public:
    /* init the MNSClient for calling MNS service
     *
     * @param endpoint:
     *      http://{AccountId}.mns.cn-hangzhou.aliyuncs.com
     *      "cn-hangzhou" is the region place
     * @param accessId:  accessId from aliyun.com
     * @param accessKey: accessKey from aliyun.com
     * @param connPoolSize:
     *      MNSClient keeps a pool of connections and reuse them
     */
    MNSClient(const std::string& endpoint,
              const std::string& accessId,
              const std::string& accessKey,
              const int32_t connPoolSize = 200);

    virtual ~MNSClient() {}

    /* Create Queue with specified name
     *
     * @param queueName: the queue name
     *
     * @return: the created queue instance
     * @throws: MNSServerException
     */
    QueuePtr createQueue(const std::string& queueName);

    /* Create Queue with specified name and attributes
     *
     * @param queueName: the queue name
     * @param attributes: the queue attributes
     *
     * @return: the created queue instance
     */
    QueuePtr createQueue(const std::string& queueName,
                      const QueueAttributes& attributes);

    /* init Queue instance with specified name without creating queue
     *
     * @param queueName: the queue name
     *
     * @return: the inited queue instance
     */
    QueuePtr getQueueRef(const std::string& queueName);

    /* delete queue with specified name
     *
     * @param queueName: the queue name
     */
    void deleteQueue(const std::string& queueName);

    /* list queues with retNum=1000 and marker=""
     *
     * @param prefix: the prefix of queueName
     *
     * @param queueNames: the response queueNames
     */
    void listQueue(const std::string& prefix,
        std::vector<std::string>& queueNames);

    /* list queues
     *
     * @param req: the ListQueueRequest containing filters
     * @param queueNames: the response queueNames
     */
    void listQueue(ListQueueRequest& req,
        std::vector<std::string>& queueNames);

    const std::string& getEndpoint() const
    {
        return mEndpoint;
    }
    const std::string& GetAccessId() const
    {
        return mAccessId;
    }
    const std::string& GetAccessKey() const
    {
        return mAccessKey;
    }

public:
    static void sendRequest(Request& req,
                            Response& response,
                            const std::string& endpoint,
                            const std::string& accessId,
                            const std::string& accessKey,
                            MNSConnectionToolPtr mnsConnTool);

    static void signRequest(Request& req,
                            const std::string& endpoint,
                            const std::string& accessId,
                            const std::string& accessKey);

protected:
    std::string mEndpoint;
    std::string mAccessId;
    std::string mAccessKey;
    MNSConnectionToolPtr mMnsConnTool;
};

/*
 * access MNS Queue service
 *
 * CAUTION:
 *     Make sure to catch Exception when calling any function
 *     Queue throws MNSServerException when the request is failed.
 *     Queue throws MNSExceptionBase for client errors
 */
class Queue
{
public:
    virtual ~Queue() {}

    const std::string& getQueueName()
    {
        return mQueueName;
    }

    /* get queue attributes
     *
     * @param attributes: the returned queue attributes
     */
    void getAttributes(QueueAttributes& attributes);

    /* set queue attributes
     *
     * @param attributes: the attributes to set
     */
    void setAttributes(const QueueAttributes& attributes);

    /* send one message
     *
     * @param messageBody: the message body
     * @param resp: the Response containing MessageId and BodyMD5
     */
    void sendMessage(const std::string& messageBody,
                     SendMessageResponse& resp);

    /* send one message
     *
     * @param messageBody: the message body
     * @delaySeconds: the message cannot be received in DelaySeconds
     * @priority:
     *     whether this message has a higher/lower priority, default is 8
     * @param resp: the Response containing MessageId and BodyMD5
     */
    void sendMessage(const std::string& messageBody,
                     const int32_t delaySeconds,
                     const int32_t priority,
                     SendMessageResponse& resp);

    /* send a batch of messages
     *
     * @param sendMessageItems: the messages to send
     * @param resp: the Response containing MessageIds and BodyMD5s
     *     1. responses of sent messages are in mSendMessageSucceedItems
     *     2. check resp.isSuccess(), if not success, some messages are not sent
     *        the failInfos are recorded in mSendMessageFailedItems
     */
    void batchSendMessage(const std::vector<SendMessageItem>& sendMessageItems,
                          BatchSendMessageResponse& resp);

    /* peek one message, the message still can be received by others
     *
     * @param message: the peeked message
     */
    void peekMessage(Message& message);

    /* batch peek
     *
     * @param numOfMessages: the batch size
     * @param messages: the peeked messages
     */
    void batchPeekMessage(const int32_t numOfMessages,
                          std::vector<Message>& messages);

    /* receive one active message,
     *    the message is changed to be inactive.
     *    message will become active after the "MessageVisibilityTimeout"
     *    ChangeMessageVisibility could be used to change the "VisibilityTime"
     *
     * @param message: the received message
     */
    void receiveMessage(Message& message);

    /* receive one active message,
     *    the message is changed to be inactive.
     *    message will become active after the "MessageVisibilityTimeout"
     *    ChangeMessageVisibility could be used to change the "VisibilityTime"
     *
     * @param waitSeconds:
     *     if no message exist in the queue, the request will block for
     *         "waitSeconds" util timeout or any message coming.
     * @param message: the received message
     */
    void receiveMessage(const int32_t waitSeconds,
                        Message& message);

    /* receive active messages,
     *    the message is changed to be inactive.
     *    message will become active after the "MessageVisibilityTimeout"
     *    ChangeMessageVisibility could be used to change the "VisibilityTime"
     *
     * @param numOfMessages: the batch size
     * @param messages: received messages
     */
    void batchReceiveMessage(const int32_t numOfMessages,
                             std::vector<Message>& messages);

    /* receive active messages,
     *    the message is changed to be inactive.
     *    message will become active after the "MessageVisibilityTimeout"
     *    ChangeMessageVisibility could be used to change the "VisibilityTime"
     *
     * @param numOfMessages: the batch size
     * @param waitSeconds:
     *     if no message exist in the queue, the request will block for
     *         "waitSeconds" util timeout or any message coming.
     * @param messages: received messages
     */
    void batchReceiveMessage(const int32_t numOfMessages,
                             const int32_t waitSeconds,
                             std::vector<Message>& messages);

    /* delete one inactive message,
     *    after receiving the message, a ReceiptHandle is returned in Message
     *    this ReceiptHandle is valid util the "VisibilityTimeout"
     *    this ReceiptHandle could be used to delete the message
     *
     * @param receiptHandle: the ReceiptHandle
     */
    void deleteMessage(const std::string& receiptHandle);

    /* delete several inactive messages,
     *    after receiving the message, a ReceiptHandle is returned in Message
     *    this ReceiptHandle is valid util the "VisibilityTimeout"
     *    this ReceiptHandle could be used to delete the message
     *
     * @param receiptHandles: the ReceiptHandles
     */
    void batchDeleteMessage(const std::vector<std::string>& receiptHandles,
                            BatchDeleteMessageResponse& resp);

    /* change the visibilityTimeout for inactive message
     *    after receiving the message, a ReceiptHandle is returned in Message
     *    this ReceiptHandle is valid util the "VisibilityTimeout"
     *    a valid ReceiptHandle could be used to change the "VisibilityTimeout"
     *
     * @param receiptHandle: the ReceiptHandle
     * @param visibilityTimeout:
     *     the message will be active again after "visibilityTimeout" seconds
     */
    void changeMessageVisibility(const std::string& receiptHandle,
                                 const int32_t visibilityTimeout,
                                 ChangeMessageVisibilityResponse& resp);

    friend class MNSClient;

protected:
    Queue(const std::string& queueName,
          const std::string& endpoint,
          const std::string& accessId,
          const std::string& accessKey,
          MNSConnectionToolPtr mnsConnTool);

protected:
    std::string mQueueName;;
    std::string mEndpoint;
    std::string mAccessId;
    std::string mAccessKey;
    MNSConnectionToolPtr mMnsConnTool;
};

}
}

#endif
